#![allow(clippy::result_large_err)]

use std::{
    collections::{HashMap, HashSet},
    fs::OpenOptions,
    path::{Path, PathBuf},
};

use log::{debug, info, trace};
use tempfile::{tempdir_in, TempDir};

use crate::{
    action::Context,
    cloud_init::{CloudInitError, LocalDataStore, LocalDataStoreBuilder},
    config2::Config,
    git::{git_head, git_is_clean, is_git, GitError},
    plan::{construct_runnable_plan, runnable_plan_from_trusted_actions, PlanError, RunnablePlan},
    project::{Project, ProjectError, Projects, State},
    qemu::{QemuError, QemuRunner},
    util::{mkdir, mkdir_child, recreate_dir, UtilError},
    vdrive::{VirtualDrive, VirtualDriveBuilder, VirtualDriveError},
};

#[allow(clippy::too_many_arguments)]
pub fn cmd_run(
    config: &Config,
    projects: &Path,
    chosen_projects: Option<&[String]>,
    dry_run: bool,
    force: bool,
    rsync_target: Option<&str>,
    rsync_target_base: Option<&str>,
    rsync_target_map: Option<&HashMap<String, String>>,
    dput_target: Option<&str>,
    executor: &Path,
) -> Result<(), RunError> {
    trace!("cmd_run: config={config:#?}");
    trace!("cmd_run: rsync_target={rsync_target:#?}");
    trace!("cmd_run: dput_target={dput_target:#?}");
    trace!("cmd_run: executor={executor:?}");

    let projects = Projects::from_file(projects)?;

    let statedir = config.state();
    if !statedir.exists() {
        debug!(
            "Create state directory shared by all projects {}",
            statedir.display()
        );
        mkdir(statedir).map_err(|e| RunError::MkdirState(statedir.into(), e))?;
    }

    for (name, project) in chosen(&projects, chosen_projects) {
        let (do_run, mut state) = should_run(dry_run, force, statedir, name, project)?;

        if do_run {
            info!("project {name}: running CI");

            let rsync_target = rsync_target_from_config(
                name,
                &rsync_target,
                &rsync_target_base,
                &rsync_target_map,
            );

            debug!("Executing pre-plan steps");
            let pre_plan = runnable_plan_from_trusted_actions(
                project,
                &state,
                rsync_target.as_deref(),
                dput_target,
                project.pre_plan(),
            );
            pre_plan.execute(Context::default())?;

            let tmp = tempdir_in(config.tmpdir()).map_err(RunError::TempDir)?;

            let source_drive = create_source_vdrive(&tmp, project.source())?;
            let deps_drive = create_dependencies_vdrive(&tmp, &state)?;
            let plan = construct_runnable_plan(project.plan())?;
            let executor_drive = create_executor_vdrive(&tmp, &plan, executor)?;

            let artifactsdir = state.artifactsdir();
            let artifact_drive = create_artifacts_vdrive(&tmp, config, project, &artifactsdir)?;

            let cachedir = state.cachedir();
            let cache_drive = create_cache_vdrive(&tmp, config, project, &cachedir)?;

            state.remove_run_log()?;
            let log = state.create_run_log()?;

            let ds = create_cloud_init_iso()?;

            let exit = QemuRunner::default()
                .config(config)
                .image(project.image())
                .excutor(&executor_drive)
                .source(&source_drive)
                .cache(&cache_drive)
                .dependencies(&deps_drive)
                .artifacts(&artifact_drive)
                .run_log(&log)
                .cloud_init(&ds)
                .run()?;
            debug!("CI run exit code from QEMU: {exit:?}");

            if exit == 0 {
                artifact_drive.extract_to(&artifactsdir)?;

                if cachedir.exists() {
                    debug!("remove old cache");
                    recreate_dir(&cachedir)?;
                }

                debug!("extract cache");
                cache_drive
                    .extract_to(&cachedir)
                    .map_err(|e| QemuError::ExtractCache(cachedir.clone(), Box::new(e)))?;

                debug!("Executing post-plan steps");
                let post_plan = runnable_plan_from_trusted_actions(
                    project,
                    &state,
                    rsync_target.as_deref(),
                    dput_target,
                    project.post_plan(),
                );
                post_plan.execute(Context::default())?;
            }

            if is_git(project.source()) {
                let head = git_head(project.source())?;
                state.set_latest_commot(Some(&head));
            } else {
                state.set_latest_commot(None);
            };
            state.write_to_file()?;

            if exit != 0 {
                return Err(RunError::RunFailed);
            }
        } else {
            info!("project {name}: NOT running CI");
        }
    }

    Ok(())
}

fn rsync_target_from_config(
    name: &str,
    rsync_target: &Option<&str>,
    rsync_target_base: &Option<&str>,
    rsync_target_map: &Option<&HashMap<String, String>>,
) -> Option<String> {
    fn join(base: &str, x: &str) -> Option<String> {
        Some(format!("{base}/{x}"))
    }

    match (rsync_target, rsync_target_base, rsync_target_map) {
        (Some(target), _, _) => Some(target.to_string()),
        (None, None, _) => None,
        (None, Some(base), None) => join(base, name),
        (None, Some(base), Some(map)) => {
            if let Some(x) = map.get(name) {
                join(base, x)
            } else {
                join(base, name)
            }
        }
    }
}

pub fn create_cloud_init_iso() -> Result<LocalDataStore, RunError> {
    const BOOTSTRAP: &str = r#"
(set -xeu
env
dir="$(mktemp -d)"
cd "$dir"
tar -xvf /dev/vdb
find -ls || true
ldd ./run-ci || true
echo ================================ BEGIN ================================
export RUST_BACKTRACE=1
if ./run-ci; then
        echo "EXIT CODE: 0"
else
        echo "EXIT CODE: $?"
fi) > /dev/ttyS1 2>&1
"#;
    LocalDataStoreBuilder::default()
        .with_hostname("ambient")
        .with_runcmd("echo xyzzy > /dev/ttyS1")
        .with_runcmd(BOOTSTRAP)
        .with_runcmd("poweroff")
        .build()
        .map_err(RunError::CloudInit)
}

pub fn create_source_vdrive(tmp: &TempDir, source_dir: &Path) -> Result<VirtualDrive, RunError> {
    create_tar(tmp.path().join("src.tar"), source_dir)
}

fn create_artifacts_vdrive(
    tmp: &TempDir,
    config: &Config,
    project: &Project,
    artifactsdir: &Path,
) -> Result<VirtualDrive, RunError> {
    recreate_dir(artifactsdir)?;
    create_tar_with_size(
        tmp.path().join("artifacts.tar"),
        artifactsdir,
        project
            .artifact_max_size()
            .unwrap_or(config.artifacts_max_size()),
    )
}

fn create_dependencies_vdrive(tmp: &TempDir, state: &State) -> Result<VirtualDrive, RunError> {
    let dependencies = state.dependenciesdir();
    if !dependencies.exists() {
        mkdir(&dependencies)
            .map_err(|e| RunError::MkdirProjectSubState(dependencies.clone(), e))?;
    }
    create_tar(tmp.path().join("deps.tar"), &dependencies)
}

fn create_cache_vdrive(
    tmp: &TempDir,
    config: &Config,
    project: &Project,
    cachedir: &Path,
) -> Result<VirtualDrive, RunError> {
    if !cachedir.exists() {
        mkdir(cachedir).map_err(|e| RunError::MkdirProjectSubState(cachedir.into(), e))?;
    }
    create_tar_with_size(
        tmp.path().join("cache.tar"),
        cachedir,
        project.cache_max_size().unwrap_or(config.cache_max_size()),
    )
}

pub fn create_executor_vdrive(
    tmp: &TempDir,
    plan: &RunnablePlan,
    executor: &Path,
) -> Result<VirtualDrive, RunError> {
    debug!("create_executor_vdrive: executor={executor:?}");
    debug!("create_executor_vdrive: plan={plan:#?}");
    debug!("executor bin {}", executor.display());
    assert!(executor.exists());

    let dirname =
        mkdir_child(tmp.path(), "ambient-execute-plan").map_err(RunError::MkdirProjectRunCi)?;
    let bin2 = dirname.join("run-ci");

    debug!("copying {} to {}", executor.display(), bin2.display());
    std::fs::copy(executor, &bin2).map_err(|e| RunError::Copy(executor.into(), bin2.clone(), e))?;
    debug!("copy OK: {}", bin2.exists());

    let plan_filename = dirname.join("plan.yaml");
    plan.to_file(&plan_filename)?;

    create_tar(tmp.path().join("executor.tar"), &dirname)
}

fn create_tar(tar_filename: PathBuf, dirname: &Path) -> Result<VirtualDrive, RunError> {
    assert!(!tar_filename.starts_with(dirname));
    debug!("create virtual drive {}", tar_filename.display());
    let tar = VirtualDriveBuilder::default()
        .filename(&tar_filename)
        .root_directory(dirname)
        .create()
        .map_err(|e| QemuError::Tar(dirname.into(), Box::new(e)))?;
    Ok(tar)
}

fn create_tar_with_size(
    tar_filename: PathBuf,
    dirname: &Path,
    size: u64,
) -> Result<VirtualDrive, RunError> {
    let tar = VirtualDriveBuilder::default()
        .filename(&tar_filename)
        .root_directory(dirname)
        .size(size)
        .create()
        .map_err(|e| QemuError::Tar(dirname.into(), Box::new(e)))?;

    let metadata = std::fs::metadata(&tar_filename)
        .map_err(|err| QemuError::Metadata(tar_filename.clone(), err))?;
    if metadata.len() < size {
        let file = OpenOptions::new()
            .write(true)
            .truncate(false)
            .open(&tar_filename)
            .map_err(|err| QemuError::Metadata(tar_filename.clone(), err))?;
        file.set_len(size)
            .map_err(|err| QemuError::SetLen(size, tar_filename.clone(), err))?;
    }

    let metadata = std::fs::metadata(&tar_filename)
        .map_err(|err| QemuError::Metadata(tar_filename.clone(), err))?;
    if metadata.len() > size {
        return Err(RunError::DriveTooBig(metadata.len(), size));
    }

    Ok(tar)
}

fn should_run(
    dry_run: bool,
    force: bool,
    statedir: &Path,
    name: &str,
    project: &Project,
) -> Result<(bool, State), RunError> {
    let mut decision = Decision::default();

    let state = State::from_file(statedir, name)?;
    if let Some(latest_commit) = state.latest_commit() {
        debug!("latest commit: {latest_commit:?}");
        decision.latest_commit(latest_commit);
    } else {
        debug!("no latest commit stored");
        // No need to set latest commit to anything, the default is OK.
    }

    let is_git = is_git(project.source());
    if is_git {
        debug!("is a git repository");
        decision.is_git();
    } else {
        debug!("is not a git repository");
        decision.is_not_git();
    }

    if git_is_clean(project.source()) {
        debug!("git repository is clean");
        decision.is_clean();
    } else {
        debug!("git repository is dirty");
        decision.is_dirty();
    }

    if is_git {
        let head = git_head(project.source())?;
        debug!("current (HEAD) commit: {head}");
        decision.current_commit(&head);
    } else {
        debug!("no current commit due to not git repository");
        // No need to set current commit to anything, the default is OK.
    }

    if dry_run {
        debug!("dry run requested");
        decision.dry_run();
    } else {
        debug!("no dry run requested");
        decision.no_dry_run();
    }

    if force {
        debug!("forced run requested");
        decision.force();
    } else {
        debug!("no forced run requested");
        decision.no_force();
    }

    let do_run = decision.should_run() == ShouldRun::Run;
    debug!("run? {do_run}");

    Ok((do_run, state))
}

fn chosen<'a>(projects: &'a Projects, chosen: Option<&[String]>) -> Vec<(&'a str, &'a Project)> {
    let set: HashSet<&str> = match chosen {
        Some(v) if !v.is_empty() => v.iter().map(|s| s.as_str()).collect(),
        _ => projects.iter().map(|(k, _)| k).collect(),
    };
    let mut projects: Vec<(&'a str, &'a Project)> =
        projects.iter().filter(|(k, _)| set.contains(k)).collect();
    projects.sort_by(|(a_name, _), (b_name, _)| a_name.cmp(b_name));
    projects
}

#[derive(Debug, thiserror::Error)]
pub enum RunError {
    #[error(transparent)]
    Project(#[from] ProjectError),

    #[error(transparent)]
    Util(#[from] UtilError),

    #[error("failed to create a context for executing actions")]
    Context(#[source] crate::action::ActionError),

    #[error("failed to create temporary directory for running CI build")]
    TempDir(#[source] std::io::Error),

    #[error("failed to create a general state directory")]
    MkdirState(PathBuf, #[source] UtilError),

    #[error("failed to create a project state sub-directory")]
    MkdirProjectSubState(PathBuf, #[source] UtilError),

    #[error("failed to create a temporary directory for action runner")]
    MkdirProjectRunCi(#[source] UtilError),

    #[error("failed to create a cloud-init ISO file")]
    CloudInit(#[source] CloudInitError),

    #[error("virtual drive is too big: {0} > {1}")]
    DriveTooBig(u64, u64),

    #[error(transparent)]
    Qemu(#[from] QemuError),

    #[error("CI run failed inside QEMU")]
    RunFailed,

    #[error("failed to copy {0} to {1}")]
    Copy(PathBuf, PathBuf, #[source] std::io::Error),

    #[error(transparent)]
    Plan(#[from] PlanError),

    #[error(transparent)]
    Git(#[from] GitError),

    #[error(transparent)]
    VDrive(#[from] VirtualDriveError),
}

#[derive(Debug, Default)]
struct Decision {
    dry_run: Option<bool>,
    force_run: Option<bool>,
    is_git: Option<bool>,
    latest_commit: Option<String>,
    current_commit: Option<String>,
    source_is_dirty: Option<bool>,
}

impl Decision {
    fn dry_run(&mut self) {
        self.dry_run = Some(true);
    }

    fn no_dry_run(&mut self) {
        self.dry_run = Some(false);
    }

    fn force(&mut self) {
        self.force_run = Some(true);
    }

    fn no_force(&mut self) {
        self.force_run = Some(false);
    }

    fn is_git(&mut self) {
        self.is_git = Some(true);
    }

    fn is_not_git(&mut self) {
        self.is_git = Some(false);
    }

    fn latest_commit(&mut self, commit: &str) {
        self.latest_commit = Some(commit.into());
    }

    fn current_commit(&mut self, commit: &str) {
        self.current_commit = Some(commit.into());
    }

    fn is_clean(&mut self) {
        self.source_is_dirty = Some(false);
    }

    fn is_dirty(&mut self) {
        self.source_is_dirty = Some(true);
    }

    fn should_run(&self) -> ShouldRun {
        let dry_run = self.dry_run == Some(true);
        let force = self.force_run == Some(true);
        let is_git = self.is_git == Some(true);
        let dirty = self.source_is_dirty == Some(true);

        if dry_run {
            Self::log("dry run");
            ShouldRun::DontRun
        } else if force {
            Self::log("force");
            ShouldRun::Run
        } else if !is_git {
            Self::log("not git");
            ShouldRun::Run
        } else if dirty {
            Self::log("dirty");
            ShouldRun::Run
        } else if self.current_commit == self.latest_commit {
            Self::log("commits are equal");
            ShouldRun::DontRun
        } else {
            Self::log("nothing prevents run");
            ShouldRun::Run
        }
    }

    #[allow(unused_variables)]
    fn log(msg: &str) {
        #[cfg(test)]
        println!("{msg}");
    }
}

// Use a custom enum to avoid confusing reader with interpreting
// boolean values, specially in tests.
#[derive(Debug, Eq, PartialEq, Copy, Clone)]
enum ShouldRun {
    Run,
    DontRun,
}

#[cfg(test)]
mod test_run_decision {
    use super::{Decision, ShouldRun};

    #[test]
    fn is_not_git() {
        let mut d = Decision::default();
        d.no_dry_run();
        d.no_force();
        d.is_not_git();
        d.is_clean();
        assert_eq!(d.should_run(), ShouldRun::Run);
    }

    #[test]
    fn unchanged() {
        let mut d = Decision::default();
        d.no_dry_run();
        d.no_force();
        d.is_git();
        d.is_clean();
        d.latest_commit("abcd");
        d.current_commit("abcd");
        assert_eq!(d.should_run(), ShouldRun::DontRun);
    }

    #[test]
    fn unchanged_with_force() {
        let mut d = Decision::default();
        d.no_dry_run();
        d.force();
        d.is_git();
        d.is_clean();
        d.latest_commit("abcd");
        d.current_commit("abcd");
        assert_eq!(d.should_run(), ShouldRun::Run);
    }

    #[test]
    fn unchanged_commit_but_dirty() {
        let mut d = Decision::default();
        d.no_dry_run();
        d.no_force();
        d.is_git();
        d.is_dirty();
        d.latest_commit("abcd");
        d.current_commit("abcd");
        assert_eq!(d.should_run(), ShouldRun::Run);
    }

    #[test]
    fn commit_changed() {
        let mut d = Decision::default();
        d.no_dry_run();
        d.no_force();
        d.is_git();
        d.is_clean();
        d.latest_commit("abcd");
        d.current_commit("efgh");
        assert_eq!(d.should_run(), ShouldRun::Run);
    }

    #[test]
    fn dry_run_for_unchanged() {
        let mut d = Decision::default();
        d.dry_run();
        d.no_force();
        d.is_git();
        d.is_clean();
        d.latest_commit("abcd");
        d.current_commit("abcd");
        assert_eq!(d.should_run(), ShouldRun::DontRun);
    }

    #[test]
    fn dry_run_for_unchanged_but_dirty() {
        let mut d = Decision::default();
        d.dry_run();
        d.no_force();
        d.is_git();
        d.is_dirty();
        d.latest_commit("abcd");
        d.current_commit("efgh");
        assert_eq!(d.should_run(), ShouldRun::DontRun);
    }

    #[test]
    fn dry_run_for_commit_changed() {
        let mut d = Decision::default();
        d.dry_run();
        d.no_force();
        d.is_git();
        d.is_clean();
        d.latest_commit("abcd");
        d.current_commit("efgh");
        assert_eq!(d.should_run(), ShouldRun::DontRun);
    }

    #[test]
    fn dry_run_for_unchanged_with_force() {
        let mut d = Decision::default();
        d.dry_run();
        d.no_force();
        d.is_git();
        d.is_clean();
        d.latest_commit("abcd");
        d.current_commit("abcd");
        assert_eq!(d.should_run(), ShouldRun::DontRun);
    }

    #[test]
    fn dry_run_for_unchanged_but_dirty_with_force() {
        let mut d = Decision::default();
        d.dry_run();
        d.no_force();
        d.is_git();
        d.is_dirty();
        d.latest_commit("abcd");
        d.current_commit("efgh");
        assert_eq!(d.should_run(), ShouldRun::DontRun);
    }

    #[test]
    fn dry_run_for_commit_changed_with_force() {
        let mut d = Decision::default();
        d.dry_run();
        d.no_force();
        d.is_git();
        d.is_clean();
        d.latest_commit("abcd");
        d.current_commit("efgh");
        assert_eq!(d.should_run(), ShouldRun::DontRun);
    }
}
