#![allow(clippy::result_large_err)]

use std::{
    collections::HashMap,
    path::{Path, PathBuf},
    process::{Command, Stdio},
};

use clingwrap::runner::CommandRunner;
use log::info;
use serde::{Deserialize, Serialize};
use tempfile::tempdir;

use crate::{
    action::{ActionError, Context, Pair},
    qemu,
    util::{changes_file, copy_partial_tree, dput, http_get_to_file, rsync_server},
    vdrive::VirtualDriveBuilder,
};

pub trait ActionImpl: std::fmt::Debug {
    fn execute(&self, context: &Context) -> Result<(), ActionError>;
}

#[derive(Debug, Clone, PartialEq, Eq, Serialize, Deserialize)]
pub struct Dummy;

impl ActionImpl for Dummy {
    fn execute(&self, _context: &Context) -> Result<(), ActionError> {
        println!("dummy action");
        Ok(())
    }
}

#[derive(Debug, Clone, PartialEq, Eq, Serialize, Deserialize)]
pub struct Pwd;

impl ActionImpl for Pwd {
    fn execute(&self, context: &Context) -> Result<(), ActionError> {
        let path = context.source_dir();
        let path = path
            .canonicalize()
            .map_err(|err| ActionError::Canonicalize(path.to_path_buf(), err))?;
        info!("cwd: {}", path.display());
        Ok(())
    }
}

#[derive(Debug, Clone, PartialEq, Eq, Serialize, Deserialize)]
pub struct CargoFetch;

impl ActionImpl for CargoFetch {
    fn execute(&self, context: &Context) -> Result<(), ActionError> {
        rust_toolchain_versions(context)?;
        let tmp = tempdir().map_err(ActionError::TempDir)?;
        let dest = tmp.path();
        copy_partial_tree(context.source_dir(), dest, |path| {
            path.ends_with("Cargo.toml")
                || path.ends_with("Cargo.lock")
                || path.as_os_str().as_encoded_bytes().ends_with(b".rs")
        })
        .map_err(ActionError::CargoFetchCopy)?;

        let lockfile = dest.join("Cargo.lock");
        let deny1 = dest.join("deny.toml");
        let deny2 = dest.join(".cargo/deny.toml");
        let deny = deny1.exists() || deny2.exists();
        if lockfile.exists() {
            spawn(context, &["cargo", "fetch", "--locked"], dest)?;
            if deny {
                spawn(context, &["cargo", "deny", "--locked", "fetch"], dest)?;
            }
        } else {
            spawn(context, &["cargo", "fetch"], dest)?;
            if deny {
                spawn(context, &["cargo", "deny", "--locked", "fetch"], dest)?;
            }
        }

        Ok(())
    }
}

#[derive(Debug, Clone, PartialEq, Eq, Serialize, Deserialize)]
pub struct CargoFmt;

impl ActionImpl for CargoFmt {
    fn execute(&self, context: &Context) -> Result<(), ActionError> {
        rust_toolchain_versions(context)?;
        spawn(context, &["cargo", "fmt", "--check"], context.source_dir())?;
        Ok(())
    }
}

#[derive(Debug, Clone, PartialEq, Eq, Serialize, Deserialize)]
pub struct CargoClippy;

impl ActionImpl for CargoClippy {
    fn execute(&self, context: &Context) -> Result<(), ActionError> {
        rust_toolchain_versions(context)?;
        spawn(
            context,
            &[
                "cargo",
                "clippy",
                "--offline",
                "--locked",
                "--workspace",
                "--all-targets",
                "--no-deps",
                "--",
                "--deny",
                "warnings",
            ],
            context.source_dir(),
        )?;
        Ok(())
    }
}

#[derive(Debug, Clone, PartialEq, Eq, Serialize, Deserialize)]
pub struct CargoDeny;

impl ActionImpl for CargoDeny {
    fn execute(&self, context: &Context) -> Result<(), ActionError> {
        rust_toolchain_versions(context)?;
        spawn(
            context,
            &[
                "cargo",
                "deny",
                "--offline",
                "--locked",
                "--workspace",
                "check",
            ],
            context.source_dir(),
        )?;
        Ok(())
    }
}

#[derive(Debug, Clone, PartialEq, Eq, Serialize, Deserialize)]
pub struct CargoDoc;

impl ActionImpl for CargoDoc {
    fn execute(&self, context: &Context) -> Result<(), ActionError> {
        rust_toolchain_versions(context)?;
        spawn(
            context,
            &[
                "env",
                "RUSTDOCFLAGS=-D warnings",
                "cargo",
                "doc",
                "--workspace",
            ],
            context.source_dir(),
        )?;
        Ok(())
    }
}

#[derive(Debug, Clone, PartialEq, Eq, Serialize, Deserialize)]
pub struct CargoBuild;

impl ActionImpl for CargoBuild {
    fn execute(&self, context: &Context) -> Result<(), ActionError> {
        rust_toolchain_versions(context)?;
        spawn(
            context,
            &[
                "cargo",
                "build",
                "--offline",
                "--locked",
                "--workspace",
                "--all-targets",
            ],
            context.source_dir(),
        )?;
        Ok(())
    }
}

#[derive(Debug, Clone, PartialEq, Eq, Serialize, Deserialize)]
pub struct CargoTest;

impl ActionImpl for CargoTest {
    fn execute(&self, context: &Context) -> Result<(), ActionError> {
        rust_toolchain_versions(context)?;
        spawn(
            context,
            &["cargo", "test", "--offline", "--locked", "--workspace"],
            context.source_dir(),
        )?;
        Ok(())
    }
}

#[derive(Debug, Clone, PartialEq, Eq, Serialize, Deserialize)]
pub struct CargoInstall;

impl ActionImpl for CargoInstall {
    fn execute(&self, context: &Context) -> Result<(), ActionError> {
        rust_toolchain_versions(context)?;
        spawn(
            context,
            &[
                "cargo",
                "install",
                "--offline",
                "--locked",
                "--bins",
                "--path=.",
                "--root",
                &context.artifacts_dir().to_string_lossy(),
            ],
            context.source_dir(),
        )?;
        Ok(())
    }
}

#[derive(Debug, Clone, PartialEq, Eq, Serialize, Deserialize)]
pub struct Deb;

impl ActionImpl for Deb {
    fn execute(&self, context: &Context) -> Result<(), ActionError> {
        let shell = format!(
            r#"#!/bin/bash
set -xeuo pipefail

echo "PATH at start: $PATH"
export PATH="/root/.cargo/bin:$PATH"
export CARGO_HOME=/workspace/deps
export DEBEMAIL=liw@liw.fi
export DEBFULLNAME="Lars Wirzenius"
/bin/env

command -v cargo
command -v rustc

cargo --version
rustc --version

# Get name and version of source package.
name="$(dpkg-parsechangelog -SSource)"
version="$(dpkg-parsechangelog -SVersion)"

# Get upstream version: everything before the last dash.
uv="$(echo "$version" | sed 's/-[^-]*$//')"

# Files that will be created.
arch="$(dpkg --print-architecture)"
orig="../${{name}}_${{uv}}.orig.tar.xz"
deb="../${{name}}_${{version}}_${{arch}}.deb"
changes="../${{name}}_${{version}}_${{arch}}.changes"

# Create "upstream tarball".
git archive HEAD | xz >"$orig"

# Build package.
dpkg-buildpackage -us -uc

# Dump some information to make it easier to visually verify
# everything looks OK. Also, test the package with the lintian tool.

ls -l ..
for x in ../*.deb; do dpkg -c "$x"; done
# FIXME: disabled while this prevents radicle-native-ci deb from being built.
# lintian -i --allow-root --fail-on warning ../*.changes

# Move files to artifacts directory.
mv ../*_* {}
        "#,
            qemu::ARTIFACTS_DIR
        );

        spawn(context, &["/bin/bash", "-c", &shell], context.source_dir())?;

        Ok(())
    }
}

#[derive(Debug, Clone, PartialEq, Eq, Serialize, Deserialize)]
pub struct Shell {
    shell: String,
}

impl Shell {
    pub fn new(shell: String) -> Self {
        Self { shell }
    }
}

impl ActionImpl for Shell {
    fn execute(&self, context: &Context) -> Result<(), ActionError> {
        let snippet = format!("set -xeuo pipefail\n{}\n", self.shell);
        spawn(
            context,
            &["/bin/bash", "-c", &snippet],
            context.source_dir(),
        )?;
        Ok(())
    }
}

#[derive(Debug, Clone, PartialEq, Eq, Serialize, Deserialize)]
pub struct HttpGet {
    items: Vec<Pair>,
}

impl HttpGet {
    pub fn new(items: Vec<Pair>) -> Self {
        Self { items }
    }

    pub fn items(&self) -> &[Pair] {
        &self.items
    }
}

impl ActionImpl for HttpGet {
    fn execute(&self, context: &Context) -> Result<(), ActionError> {
        let deps = context.deps_dir();
        info!("http_get: deps={}", deps.display());
        for pair in self.items.iter() {
            let filename = deps.join(pair.filename());
            http_get_to_file(pair.url(), &filename)
                .map_err(|err| ActionError::Get(pair.url().to_string(), err))?;
        }
        Ok(())
    }
}

#[derive(Debug, Clone, PartialEq, Eq, Serialize, Deserialize)]
pub struct Rsync {
    rsync_target: Option<String>,
}

impl Rsync {
    pub fn new(rsync_target: Option<String>) -> Self {
        Self { rsync_target }
    }
}

impl ActionImpl for Rsync {
    fn execute(&self, context: &Context) -> Result<(), ActionError> {
        if let Some(target) = &self.rsync_target {
            rsync_server(context.artifacts_dir(), target)?;
        }
        Ok(())
    }
}

#[derive(Debug, Clone, PartialEq, Eq, Serialize, Deserialize)]
pub struct Dput {
    dput_target: Option<String>,
}

impl Dput {
    pub fn new(dput_target: Option<String>) -> Self {
        Self { dput_target }
    }
}

impl ActionImpl for Dput {
    fn execute(&self, context: &Context) -> Result<(), ActionError> {
        if let Some(target) = &self.dput_target {
            let changes = changes_file(context.artifacts_dir())?;
            dput(target, &changes)?;
        }
        Ok(())
    }
}

#[derive(Debug, Clone, PartialEq, Eq, Serialize, Deserialize)]
pub struct Mkdir {
    pathname: PathBuf,
}

impl Mkdir {
    pub fn new(pathname: PathBuf) -> Self {
        Self { pathname }
    }
}

impl ActionImpl for Mkdir {
    fn execute(&self, _context: &Context) -> Result<(), ActionError> {
        if !self.pathname.exists() {
            std::fs::create_dir(&self.pathname)
                .map_err(|e| ActionError::Mkdir(self.pathname.clone(), e))?;
        }
        Ok(())
    }
}

#[derive(Debug, Clone, PartialEq, Eq, Serialize, Deserialize)]
pub struct TarCreate {
    archive: PathBuf,
    directory: PathBuf,
}

impl TarCreate {
    pub fn new(archive: PathBuf, directory: PathBuf) -> Self {
        Self { archive, directory }
    }
}

impl ActionImpl for TarCreate {
    fn execute(&self, _context: &Context) -> Result<(), ActionError> {
        VirtualDriveBuilder::default()
            .filename(&self.archive)
            .root_directory(&self.directory)
            .create()
            .map_err(|e| ActionError::TarCreate(self.archive.clone(), self.directory.clone(), e))?;
        Ok(())
    }
}

#[derive(Debug, Clone, PartialEq, Eq, Serialize, Deserialize)]
pub struct TarExtract {
    archive: PathBuf,
    directory: PathBuf,
}

impl TarExtract {
    pub fn new(archive: PathBuf, directory: PathBuf) -> Self {
        Self { archive, directory }
    }
}

impl ActionImpl for TarExtract {
    fn execute(&self, _context: &Context) -> Result<(), ActionError> {
        let tar = VirtualDriveBuilder::default()
            .filename(&self.archive)
            .root_directory(&self.directory)
            .open()
            .map_err(|e| ActionError::TarOpen(self.archive.clone(), e))?;
        tar.extract_to(&self.directory).map_err(|e| {
            ActionError::TarExtract(self.archive.clone(), self.directory.clone(), e)
        })?;
        Ok(())
    }
}

#[derive(Debug, Clone, PartialEq, Eq, Serialize, Deserialize)]
pub struct Custom {
    name: String,

    #[serde(default)]
    args: HashMap<String, serde_norway::Value>,
}

impl Custom {
    pub fn new(name: String, args: HashMap<String, serde_norway::Value>) -> Self {
        Self { name, args }
    }
}

impl ActionImpl for Custom {
    fn execute(&self, context: &Context) -> Result<(), ActionError> {
        let source = context.source_dir().to_path_buf();
        let exe = Path::new(".ambient").join(&self.name);
        let json = serde_json::to_string(&self.args).map_err(ActionError::ArgsToJson)?;

        eprintln!("custom: source={}", source.display());
        eprintln!("custom: exe={exe:?} exists={}", exe.exists());
        let mut cmd = Command::new(exe);
        cmd.current_dir(&source);
        cmd.envs(context.env());
        for (key, value) in self.args.iter() {
            let key = format!("AMBIENT_CI_{key}");
            let value = serde_json::to_string(value).map_err(ActionError::ArgsToJson)?;
            cmd.env(key, value);
        }
        let mut runner = CommandRunner::new(cmd);
        runner.feed_stdin(json.as_bytes());
        let output = runner
            .execute()
            .map_err(|err| ActionError::Custom(self.name.to_string(), err))?;
        println!(
            "custom action {:?} exit code {:?}",
            self.name,
            output.status.code()
        );
        Ok(())
    }
}

fn rust_toolchain_versions(context: &Context) -> Result<(), ActionError> {
    spawn(context, &["cargo", "--version"], context.source_dir())?;
    spawn(
        context,
        &["cargo", "clippy", "--version"],
        context.source_dir(),
    )?;
    spawn(context, &["rustc", "--version"], context.source_dir())?;
    Ok(())
}

fn spawn(context: &Context, argv: &[&str], cwd: &Path) -> Result<(), ActionError> {
    println!("SPAWN: argv={argv:?}");
    println!("       cwd={} (exists? {})", cwd.display(), cwd.exists());

    let argv0 = if let Some(argv0) = argv.first() {
        argv0
    } else {
        return Err(ActionError::SpawnNoArgv0);
    };

    let mut cmd = Command::new(argv0);
    cmd.args(&argv[1..])
        .envs(context.env())
        .current_dir(cwd)
        .stdin(Stdio::null())
        .current_dir(cwd);

    let runner = CommandRunner::new(cmd);
    match runner.execute() {
        Ok(_) => Ok(()),
        Err(err) => Err(ActionError::Execute(argv0.to_string(), err)),
    }
}
