use serde::{Deserialize, Serialize};
use tempfile::tempdir;

use crate::{
    action::{ActionError, Context},
    action_impl::{rust_toolchain_versions, spawn, ActionImpl},
    util::copy_partial_tree,
};

#[derive(Debug, Clone, PartialEq, Eq, Serialize, Deserialize)]
pub struct CargoFetch;

impl ActionImpl for CargoFetch {
    fn execute(&self, context: &Context) -> Result<(), ActionError> {
        rust_toolchain_versions(context)?;
        let tmp = tempdir().map_err(ActionError::TempDir)?;
        let dest = tmp.path();
        copy_partial_tree(context.source_dir(), dest, |path| {
            path.ends_with("Cargo.toml")
                || path.ends_with("Cargo.lock")
                || path.as_os_str().as_encoded_bytes().ends_with(b".rs")
        })
        .map_err(ActionError::CargoFetchCopy)?;

        let lockfile = dest.join("Cargo.lock");
        let deny1 = dest.join("deny.toml");
        let deny2 = dest.join(".cargo/deny.toml");
        let deny = deny1.exists() || deny2.exists();
        if lockfile.exists() {
            spawn(context, &["cargo", "fetch", "--locked"], dest)?;
            if deny {
                spawn(context, &["cargo", "deny", "--locked", "fetch"], dest)?;
            }
        } else {
            spawn(context, &["cargo", "fetch"], dest)?;
            if deny {
                spawn(context, &["cargo", "deny", "--locked", "fetch"], dest)?;
            }
        }

        Ok(())
    }
}

#[derive(Debug, Clone, PartialEq, Eq, Serialize, Deserialize)]
pub struct CargoFmt;

impl ActionImpl for CargoFmt {
    fn execute(&self, context: &Context) -> Result<(), ActionError> {
        rust_toolchain_versions(context)?;
        spawn(context, &["cargo", "fmt", "--check"], context.source_dir())?;
        Ok(())
    }
}

#[derive(Debug, Clone, PartialEq, Eq, Serialize, Deserialize)]
pub struct CargoClippy;

impl ActionImpl for CargoClippy {
    fn execute(&self, context: &Context) -> Result<(), ActionError> {
        rust_toolchain_versions(context)?;
        spawn(
            context,
            &[
                "cargo",
                "clippy",
                "--offline",
                "--locked",
                "--workspace",
                "--all-targets",
                "--no-deps",
                "--",
                "--deny",
                "warnings",
            ],
            context.source_dir(),
        )?;
        Ok(())
    }
}

#[derive(Debug, Clone, PartialEq, Eq, Serialize, Deserialize)]
pub struct CargoDeny;

impl ActionImpl for CargoDeny {
    fn execute(&self, context: &Context) -> Result<(), ActionError> {
        rust_toolchain_versions(context)?;
        spawn(
            context,
            &[
                "cargo",
                "deny",
                "--offline",
                "--locked",
                "--workspace",
                "check",
            ],
            context.source_dir(),
        )?;
        Ok(())
    }
}

#[derive(Debug, Clone, PartialEq, Eq, Serialize, Deserialize)]
pub struct CargoDoc;

impl ActionImpl for CargoDoc {
    fn execute(&self, context: &Context) -> Result<(), ActionError> {
        rust_toolchain_versions(context)?;
        spawn(
            context,
            &[
                "env",
                "RUSTDOCFLAGS=-D warnings",
                "cargo",
                "doc",
                "--workspace",
            ],
            context.source_dir(),
        )?;
        Ok(())
    }
}

#[derive(Debug, Clone, PartialEq, Eq, Serialize, Deserialize)]
pub struct CargoBuild;

impl ActionImpl for CargoBuild {
    fn execute(&self, context: &Context) -> Result<(), ActionError> {
        rust_toolchain_versions(context)?;
        spawn(
            context,
            &[
                "cargo",
                "build",
                "--offline",
                "--locked",
                "--workspace",
                "--all-targets",
            ],
            context.source_dir(),
        )?;
        Ok(())
    }
}

#[derive(Debug, Clone, PartialEq, Eq, Serialize, Deserialize)]
pub struct CargoTest;

impl ActionImpl for CargoTest {
    fn execute(&self, context: &Context) -> Result<(), ActionError> {
        rust_toolchain_versions(context)?;
        spawn(
            context,
            &["cargo", "test", "--offline", "--locked", "--workspace"],
            context.source_dir(),
        )?;
        Ok(())
    }
}

#[derive(Debug, Clone, PartialEq, Eq, Serialize, Deserialize)]
pub struct CargoInstall;

impl ActionImpl for CargoInstall {
    fn execute(&self, context: &Context) -> Result<(), ActionError> {
        rust_toolchain_versions(context)?;
        spawn(
            context,
            &[
                "cargo",
                "install",
                "--offline",
                "--locked",
                "--bins",
                "--path=.",
                "--root",
                &context.artifacts_dir().to_string_lossy(),
            ],
            context.source_dir(),
        )?;
        Ok(())
    }
}
