use std::path::PathBuf;

use serde::{Deserialize, Serialize};

use crate::{
    action::{ActionError, Context},
    action_impl::ActionImpl,
    vdrive::VirtualDriveBuilder,
};

#[derive(Debug, Clone, PartialEq, Eq, Serialize, Deserialize)]
pub struct TarCreate {
    archive: PathBuf,
    directory: PathBuf,
}

impl TarCreate {
    pub fn new(archive: PathBuf, directory: PathBuf) -> Self {
        Self { archive, directory }
    }
}

impl ActionImpl for TarCreate {
    fn execute(&self, _context: &Context) -> Result<(), ActionError> {
        VirtualDriveBuilder::default()
            .filename(&self.archive)
            .root_directory(&self.directory)
            .create()
            .map_err(|e| ActionError::TarCreate(self.archive.clone(), self.directory.clone(), e))?;
        Ok(())
    }
}

#[derive(Debug, Clone, PartialEq, Eq, Serialize, Deserialize)]
pub struct TarExtract {
    archive: PathBuf,
    directory: PathBuf,
}

impl TarExtract {
    pub fn new(archive: PathBuf, directory: PathBuf) -> Self {
        Self { archive, directory }
    }
}

impl ActionImpl for TarExtract {
    fn execute(&self, _context: &Context) -> Result<(), ActionError> {
        let tar = VirtualDriveBuilder::default()
            .filename(&self.archive)
            .root_directory(&self.directory)
            .open()
            .map_err(|e| ActionError::TarOpen(self.archive.clone(), e))?;
        tar.extract_to(&self.directory).map_err(|e| {
            ActionError::TarExtract(self.archive.clone(), self.directory.clone(), e)
        })?;
        Ok(())
    }
}
