use std::path::{Path, PathBuf};

use clap::Parser;
use log::debug;
use tempfile::tempdir_in;

use ambient_ci::{
    cloud_init::CloudInitError,
    git::GitError,
    plan::{PlanError, RunnablePlan},
    qemu::{QemuError, QemuRunner},
    run::{create_cloud_init_iso, create_executor_vdrive},
    util::{cat_text_file, mkdir, UtilError},
    vdrive::{VirtualDrive, VirtualDriveBuilder, VirtualDriveError},
};

use super::{AmbientDriverError, Config, Leaf};

/// Run QEMU with a specific runnable plan. Empty source, cache,
/// dependencies, and artifacts.
#[derive(Debug, Parser)]
pub struct QemuCmd {
    /// File with runnable plan.
    #[clap(long)]
    plan: PathBuf,

    /// Use this virtual machine image as the base image. The base
    /// image will not be modified, even if the virtual machine changes
    /// things on its disk. The changes are written to a copy-on-write
    /// temporary image (but see `--persist`).
    #[clap(long)]
    image: PathBuf,

    /// Save the image after the VM shuts to this file. This allows
    /// capturing changes made inside the virtual machine.
    #[clap(long)]
    persist: Option<PathBuf>,

    /// Allow network?
    #[clap(long)]
    network: bool,
}

impl QemuCmd {
    fn helper(&self, config: &Config) -> Result<(), QemuCmdError> {
        let runnable_plan = RunnablePlan::from_file(&self.plan)?;

        let tmp = tempdir_in(config.tmpdir()).map_err(QemuCmdError::TempDir)?;

        let console_log = tmp.path().join("console.log");
        let run_log = tmp.path().join("run.log");

        let empty = tmp.path().join("src");
        mkdir(&empty)?;

        let executor = config.executor().ok_or(QemuCmdError::NoExecutor)?;
        let executor_drive = create_executor_vdrive(&tmp, &runnable_plan, executor)?;
        let source_drive = create_tar(tmp.path().join("src.tar"), &empty)?;
        let artifacts_drive = create_tar(tmp.path().join("artifacts.tar"), &empty)?;
        let ds = create_cloud_init_iso()?;

        let mut qemu = QemuRunner::default()
            .config(config)
            .image(&self.image)
            .excutor(&executor_drive)
            .cloud_init(&ds)
            .source(&source_drive)
            .artifacts(&artifacts_drive)
            .cloud_init(&ds)
            .console_log(&console_log)
            .run_log(&run_log)
            .network(self.network);

        if let Some(persist) = &self.persist {
            qemu = qemu.new_image(persist);
        }

        let exit = qemu.run()?;
        debug!("CI run exit code from QEMU: {exit:?}");

        let run_log = cat_text_file(&run_log)?;
        print!("{run_log}");
        Ok(())
    }
}

impl Leaf for QemuCmd {
    fn run(&self, config: &Config) -> Result<(), AmbientDriverError> {
        Ok(self.helper(config)?)
    }
}

// FIXME: duplicate from run.rs
fn create_tar(tar_filename: PathBuf, dirname: &Path) -> Result<VirtualDrive, QemuCmdError> {
    assert!(!tar_filename.starts_with(dirname));
    debug!("create virtual drive {}", tar_filename.display());
    let tar = VirtualDriveBuilder::default()
        .filename(&tar_filename)
        .root_directory(dirname)
        .create()
        .map_err(|e| QemuError::Tar(dirname.into(), Box::new(e)))?;
    Ok(tar)
}

#[derive(Debug, thiserror::Error)]
pub enum QemuCmdError {
    #[error(transparent)]
    Qemu(#[from] QemuError),

    #[error("failed to create a temporary directory")]
    TempDir(#[source] std::io::Error),

    #[error(transparent)]
    Util(#[from] UtilError),

    #[error(transparent)]
    VDrive(#[from] VirtualDriveError),

    #[error(transparent)]
    CloudInit(#[from] CloudInitError),

    #[error(transparent)]
    CreateDrive(#[from] ambient_ci::run::RunError),

    #[error(transparent)]
    Plan(#[from] PlanError),

    #[error(transparent)]
    Git(#[from] GitError),

    #[error("no executor specified in configuration")]
    NoExecutor,
}
