use std::{
    fs::OpenOptions,
    path::{Path, PathBuf},
};

use clap::Parser;
use log::{debug, info};
use tempfile::tempdir_in;

use ambient_ci::{
    cloud_init::CloudInitError,
    git::GitError,
    plan::{PlanError, RunnablePlan},
    qemu::{QemuError, QemuRunner},
    qemu_utils::convert_image,
    run::{create_cloud_init_iso, create_executor_vdrive},
    util::{mkdir, UtilError},
    vdrive::{VirtualDrive, VirtualDriveBuilder, VirtualDriveError},
};

use super::{AmbientDriverError, Config, Leaf};

/// Run QEMU with a specific runnable plan. Empty source, cache,
/// dependencies, and artifacts.
#[derive(Debug, Parser)]
pub struct QemuCmd {
    /// File with runnable plan.
    #[clap(long)]
    plan: PathBuf,

    /// Use this virtual machine image as the base image. The base
    /// image will not be modified, even if the virtual machine changes
    /// things on its disk. The changes are written to a copy-on-write
    /// temporary image (but see `--persist`).
    #[clap(long)]
    image: PathBuf,

    /// Save the image after the VM shuts to this file. This allows
    /// capturing changes made inside the virtual machine.
    #[clap(long)]
    persist: Option<PathBuf>,

    /// Allow network?
    #[clap(long)]
    network: bool,

    /// Write console log to this file.
    #[clap(long)]
    console: Option<PathBuf>,

    /// Write run log to this file.
    #[clap(long)]
    run_log: Option<PathBuf>,

    /// Write artifacts to this file.
    #[clap(long)]
    artifacts: Option<PathBuf>,

    /// Use UEFI.
    #[clap(long)]
    uefi: bool,
}

impl QemuCmd {
    fn helper(&self, config: &Config) -> Result<(), QemuCmdError> {
        let runnable_plan = RunnablePlan::from_file(&self.plan)?;

        let tmp = tempdir_in(config.tmpdir()).map_err(QemuCmdError::TempDir)?;

        let console_log = self
            .console
            .clone()
            .unwrap_or_else(|| tmp.path().join("console.log"));
        let run_log = self
            .run_log
            .clone()
            .unwrap_or_else(|| tmp.path().join("run.log"));

        let empty = tmp.path().join("src");
        mkdir(&empty)?;

        let executor = config.executor().ok_or(QemuCmdError::NoExecutor)?;
        let executor_drive = create_executor_vdrive(&tmp, &runnable_plan, executor)?;
        let source_drive = create_tar(tmp.path().join("src.tar"), &empty)?;
        let artifacts_drive = create_tar_with_size(
            self.artifacts
                .clone()
                .unwrap_or_else(|| tmp.path().join("artifacts.tar")),
            &empty,
            1024 * 1024 * 1024,
        )?;
        let ds = create_cloud_init_iso(self.network)?;

        let backing_image = self
            .image
            .canonicalize()
            .map_err(|err| QemuCmdError::Canonicalize(self.image.clone(), err))?;
        let new_image = tmp.path().join("vm.qcow2");

        let qemu = QemuRunner::default()
            .config(config)
            .image(&backing_image)
            .new_image(&new_image)
            .excutor(&executor_drive)
            .cloud_init(&ds)
            .source(&source_drive)
            .artifacts(&artifacts_drive)
            .cloud_init(&ds)
            .console_log(&console_log)
            .run_log(&run_log)
            .network(self.network)
            .uefi(self.uefi || config.uefi());

        let exit = qemu.run()?;
        debug!("CI run exit code from QEMU: {exit:?}");

        if let Some(persist) = &self.persist {
            info!("persisting VM image to {}", persist.display());
            convert_image(&new_image, persist).map_err(|err| {
                QemuCmdError::ConvertImage(new_image.clone(), persist.to_path_buf(), err)
            })?;
        }

        Ok(())
    }
}

impl Leaf for QemuCmd {
    fn run(&self, config: &Config) -> Result<(), AmbientDriverError> {
        Ok(self.helper(config)?)
    }
}

// FIXME: duplicate from run.rs
fn create_tar(tar_filename: PathBuf, dirname: &Path) -> Result<VirtualDrive, QemuCmdError> {
    assert!(!tar_filename.starts_with(dirname));
    debug!("create virtual drive {}", tar_filename.display());
    let tar = VirtualDriveBuilder::default()
        .filename(&tar_filename)
        .root_directory(dirname)
        .create()
        .map_err(|e| QemuError::Tar(dirname.into(), Box::new(e)))?;
    Ok(tar)
}

// FIXME: duplicate from run.rs
fn create_tar_with_size(
    tar_filename: PathBuf,
    dirname: &Path,
    size: u64,
) -> Result<VirtualDrive, QemuCmdError> {
    let tar = VirtualDriveBuilder::default()
        .filename(&tar_filename)
        .root_directory(dirname)
        .size(size)
        .create()
        .map_err(|e| QemuError::Tar(dirname.into(), Box::new(e)))?;

    let metadata = std::fs::metadata(&tar_filename)
        .map_err(|err| QemuError::Metadata(tar_filename.clone(), err))?;
    if metadata.len() < size {
        let file = OpenOptions::new()
            .write(true)
            .truncate(false)
            .open(&tar_filename)
            .map_err(|err| QemuError::Metadata(tar_filename.clone(), err))?;
        file.set_len(size)
            .map_err(|err| QemuError::SetLen(size, tar_filename.clone(), err))?;
    }

    let metadata = std::fs::metadata(&tar_filename)
        .map_err(|err| QemuError::Metadata(tar_filename.clone(), err))?;
    if metadata.len() > size {
        return Err(QemuCmdError::DriveTooBig(metadata.len(), size));
    }

    Ok(tar)
}

#[derive(Debug, thiserror::Error)]
pub enum QemuCmdError {
    #[error(transparent)]
    Qemu(#[from] QemuError),

    #[error("failed to create a temporary directory")]
    TempDir(#[source] std::io::Error),

    #[error(transparent)]
    Util(#[from] UtilError),

    #[error(transparent)]
    VDrive(#[from] VirtualDriveError),

    #[error(transparent)]
    CloudInit(#[from] CloudInitError),

    #[error(transparent)]
    CreateDrive(#[from] ambient_ci::run::RunError),

    #[error("virtual drive is too big: {0} > {1}")]
    DriveTooBig(u64, u64),

    #[error(transparent)]
    Plan(#[from] PlanError),

    #[error(transparent)]
    Git(#[from] GitError),

    #[error("no executor specified in configuration")]
    NoExecutor,

    #[error("failed to convert image {0} to {1}")]
    ConvertImage(
        PathBuf,
        PathBuf,
        #[source] ambient_ci::qemu_utils::QemuUtiluError,
    ),

    #[error("failed to make path absolute: {0}")]
    Canonicalize(PathBuf, #[source] std::io::Error),
}
