use std::path::PathBuf;

use clap::Parser;

use super::{AmbientDriverError, Config, Leaf};

/// Show configuration to user.
#[derive(Debug, Parser)]
pub struct ConfigCmd {
    #[clap(long)]
    oneline: bool,

    #[clap(long)]
    output: Option<PathBuf>,
}

impl Leaf for ConfigCmd {
    fn run(&self, config: &Config) -> Result<(), AmbientDriverError> {
        let json = if self.oneline {
            serde_json::to_string(config).map_err(ConfigError::ToJson)?
        } else {
            serde_json::to_string_pretty(config).map_err(ConfigError::ToJson)?
        };

        if let Some(filename) = &self.output {
            std::fs::write(filename, json.as_bytes())
                .map_err(|err| ConfigError::Write(filename.into(), err))?
        } else {
            println!("{json}");
        }

        Ok(())
    }
}

#[derive(Debug, thiserror::Error)]
pub enum ConfigError {
    #[error("failed to serialize configuration to JSON")]
    ToJson(#[source] serde_json::Error),

    #[error("failed to write output to {0}")]
    Write(PathBuf, #[source] std::io::Error),
}
