//! Virtual drive handling for ambient-run.

use std::{
    fs::{File, OpenOptions},
    path::{Path, PathBuf},
};

use log::{debug, error, trace};

use crate::util::{tar_create, tar_extract, UtilError};

/// A virtual drive.
#[derive(Debug, Clone)]
pub struct VirtualDrive {
    filename: PathBuf,
}

impl VirtualDrive {
    /// Path to file containing virtual drive.
    pub fn filename(&self) -> &Path {
        self.filename.as_path()
    }

    /// Extract files in the virtual drive into a directory. Create
    /// the directory if it doesn't exist.
    pub fn extract_to(&self, dirname: &Path) -> Result<(), VirtualDriveError> {
        trace!(
            "extracting {} to {}",
            self.filename.display(),
            dirname.display()
        );
        if !dirname.exists() {
            std::fs::create_dir(dirname)
                .map_err(|e| VirtualDriveError::Extract(dirname.into(), e))?;
        }
        tar_extract(&self.filename, dirname)?;
        trace!("extraction OK");
        Ok(())
    }
}

/// Builder for [`VirtualDrive`].
#[derive(Debug, Default)]
pub struct VirtualDriveBuilder {
    filename: Option<PathBuf>,
    root: Option<PathBuf>,
    size: Option<u64>,
}

impl VirtualDriveBuilder {
    /// Set filename for virtual drive.
    pub fn filename(mut self, filename: &Path) -> Self {
        self.filename = Some(filename.into());
        self
    }

    /// Set directory of tree to copy into virtual drive.
    pub fn root_directory(mut self, dirname: &Path) -> Self {
        self.root = Some(dirname.into());
        self
    }

    /// Set size of new drive. This is important when the build VM
    /// writes to the drive.
    pub fn size(mut self, size: u64) -> Self {
        self.size = Some(size);
        self
    }

    /// Create a virtual drive.
    pub fn create(self) -> Result<VirtualDrive, VirtualDriveError> {
        trace!("creating virtual drive (tar archive): {self:#?}");

        let filename = self.filename.expect("filename has been set");
        trace!(
            "tar archive to be created: {}; exists? {}",
            filename.display(),
            filename.exists()
        );

        trace!("create archive file {}", filename.display());

        // Create the file, either empty or to the desired size. If we
        // don't have self.root set, the file created (and maybe
        // truncated) will be.
        {
            let archive = File::create(&filename)
                .map_err(|e| VirtualDriveError::Create(filename.clone(), e))?;
            if let Some(size) = self.size {
                trace!("restrict file {} to {} bytes", filename.display(), size);
                archive
                    .set_len(size)
                    .map_err(|e| VirtualDriveError::Create(filename.clone(), e))?;
            }
        }

        if let Some(root) = self.root {
            trace!("directory {} exists? {}", root.display(), root.exists());
            trace!("add contents of {} as .", root.display());
            trace!("calling tar_create");
            match tar_create(&filename, &root) {
                Ok(_) => (),
                Err(UtilError::TarFailed(cmd, filename, exit, stderr)) => {
                    error!(
                        "tar {cmd} failed: exit={exit} archive={}, stderr=\n{stderr}",
                        filename.display()
                    );
                    Err(VirtualDriveError::Tar)?;
                }
                Err(err) => {
                    trace!("tar failed: {err:?}");
                    Err(err)?;
                }
            }
        }

        trace!("created virtual drive {}", filename.display());
        Ok(VirtualDrive { filename })
    }

    /// Open an existing virtual drive.
    pub fn open(self) -> Result<VirtualDrive, VirtualDriveError> {
        let filename = self.filename.expect("filename has been set");
        Ok(VirtualDrive { filename })
    }
}

/// Create a tar archive out of a directory.
pub fn create_tar(
    tar_filename: PathBuf,
    dirname: &Path,
) -> Result<VirtualDrive, VirtualDriveError> {
    assert!(!tar_filename.starts_with(dirname));
    debug!("create virtual drive {}", tar_filename.display());
    VirtualDriveBuilder::default()
        .filename(&tar_filename)
        .root_directory(dirname)
        .create()
}

/// Create a tar archive with a fixed length, out of a directory.
pub fn create_tar_with_size(
    tar_filename: PathBuf,
    dirname: &Path,
    size: u64,
) -> Result<VirtualDrive, VirtualDriveError> {
    let tar = VirtualDriveBuilder::default()
        .filename(&tar_filename)
        .root_directory(dirname)
        .size(size)
        .create()?;

    let metadata = std::fs::metadata(&tar_filename)
        .map_err(|err| VirtualDriveError::Metadata(tar_filename.clone(), err))?;
    if metadata.len() < size {
        let file = OpenOptions::new()
            .write(true)
            .truncate(false)
            .open(&tar_filename)
            .map_err(|err| VirtualDriveError::CreateTar(tar_filename.clone(), err))?;
        file.set_len(size)
            .map_err(|err| VirtualDriveError::SetLen(size, tar_filename.clone(), err))?;
    }

    let metadata = std::fs::metadata(&tar_filename)
        .map_err(|err| VirtualDriveError::Metadata(tar_filename.clone(), err))?;
    if metadata.len() > size {
        return Err(VirtualDriveError::DriveTooBig(metadata.len(), size));
    }

    Ok(tar)
}

/// Errors that may be returned from [`VirtualDrive`] use.
#[allow(missing_docs)]
#[derive(Debug, thiserror::Error)]
pub enum VirtualDriveError {
    #[error("failed to create virtual drive {0}")]
    Create(PathBuf, #[source] std::io::Error),

    #[error("failed to create tar archive for virtual drive from {0}")]
    CreateTar(PathBuf, #[source] std::io::Error),

    #[error("failed to open virtual drive {0}")]
    Open(PathBuf, #[source] std::io::Error),

    #[error("failed to list files in virtual drive {0}")]
    List(PathBuf, #[source] std::io::Error),

    #[error("failed to create directory {0}")]
    Extract(PathBuf, #[source] std::io::Error),

    #[error("failed to extract {0} to {1}")]
    ExtractEntry(PathBuf, PathBuf, #[source] std::io::Error),

    #[error("tar command failed")]
    Tar,

    #[error(transparent)]
    Util(#[from] crate::util::UtilError),

    #[error("failed to get length of file {0}")]
    Metadata(PathBuf, #[source] std::io::Error),

    #[error("failed to set length of file to {0}: {1}")]
    SetLen(u64, PathBuf, #[source] std::io::Error),

    #[error("virtual drive is too big: {0} > {1}")]
    DriveTooBig(u64, u64),
}
