use std::path::{Path, PathBuf};

use serde::{Deserialize, Serialize};

use crate::{
    action::{ActionError, Context},
    action_impl::ActionImpl,
    util::rsync_server,
};

/// Upload build artifacts using `rsync` to configured server.
#[derive(Debug, Clone, PartialEq, Eq, Serialize, Deserialize)]
pub struct Rsync {
    artifacts: PathBuf,
    rsync_target: Option<String>,
}

impl Rsync {
    pub fn new<P: AsRef<Path>>(artifacts: P, rsync_target: Option<String>) -> Self {
        Self {
            artifacts: artifacts.as_ref().to_path_buf(),
            rsync_target,
        }
    }
}

impl ActionImpl for Rsync {
    fn execute(&self, context: &Context) -> Result<(), ActionError> {
        if let Some(target) = &self.rsync_target {
            rsync_server(&context.artifacts_dir().join(&self.artifacts), target)?;
        }
        Ok(())
    }
}
