use std::path::{Path, PathBuf};

use serde::{Deserialize, Serialize};

use crate::{
    action::{ActionError, Context},
    action_impl::ActionImpl,
    util::{changes_file, dput},
};

/// Upload a built `deb` package using the `dput` command.
///
/// The host must have `dput` configured to support the specified upload
/// target.
#[derive(Debug, Clone, PartialEq, Eq, Serialize, Deserialize)]
pub struct Dput {
    artifacts: PathBuf,
    dput_target: Option<String>,
}

impl Dput {
    pub fn new<P: AsRef<Path>>(artifacts: P, dput_target: Option<String>) -> Self {
        Self {
            artifacts: artifacts.as_ref().to_path_buf(),
            dput_target,
        }
    }
}

impl ActionImpl for Dput {
    fn execute(&self, context: &Context) -> Result<(), ActionError> {
        if let Some(target) = &self.dput_target {
            let changes = changes_file(context.artifacts_dir())?;
            dput(target, &changes)?;
        }
        Ok(())
    }
}
