//! Action implementations.

#![allow(clippy::result_large_err)]

use std::{
    path::Path,
    process::{Command, Stdio},
};

use clingwrap::runner::CommandRunner;

use crate::action::{ActionError, Context};

mod cargo;
pub use cargo::*;

mod custom;
pub use custom::*;

mod deb;
pub use deb::*;

mod dput;
pub use dput::*;

mod dummy;
pub use dummy::*;

mod http_get;
pub use http_get::*;

mod mkdir;
pub use mkdir::*;

mod pwd;
pub use pwd::*;

mod rsync;
pub use rsync::*;

mod shell;
pub use shell::*;

mod tar;
pub use tar::*;

pub trait ActionImpl: std::fmt::Debug {
    fn execute(&self, context: &Context) -> Result<(), ActionError>;
}

fn rust_toolchain_versions(context: &Context) -> Result<(), ActionError> {
    spawn(context, &["cargo", "--version"], context.source_dir())?;
    spawn(
        context,
        &["cargo", "clippy", "--version"],
        context.source_dir(),
    )?;
    spawn(context, &["rustc", "--version"], context.source_dir())?;
    Ok(())
}

fn spawn(context: &Context, argv: &[&str], cwd: &Path) -> Result<(), ActionError> {
    println!("SPAWN: argv={argv:?}");
    println!("       cwd={} (exists? {})", cwd.display(), cwd.exists());

    let argv0 = if let Some(argv0) = argv.first() {
        argv0
    } else {
        return Err(ActionError::SpawnNoArgv0);
    };

    let mut cmd = Command::new(argv0);
    cmd.args(&argv[1..])
        .envs(context.env())
        .current_dir(cwd)
        .stdin(Stdio::null())
        .current_dir(cwd);

    let runner = CommandRunner::new(cmd);
    match runner.execute() {
        Ok(_) => Ok(()),
        Err(err) => Err(ActionError::Execute(argv0.to_string(), err)),
    }
}
