//! Action that use Rust `cargo` too.

use std::path::{Path, PathBuf};

use log::debug;
use serde::{Deserialize, Serialize};
use tempfile::tempdir;
use walkdir::WalkDir;

use crate::{
    action::{ActionError, Context},
    action_impl::{rust_toolchain_versions, spawn, ActionImpl},
    util::{copy_file, mkdir, UtilError},
};

/// Download Rust crate dependencies using `cargo fetch`.
#[derive(Debug, Clone, PartialEq, Eq, Serialize, Deserialize)]
pub struct CargoFetch;

impl ActionImpl for CargoFetch {
    fn execute(&self, context: &Context) -> Result<(), ActionError> {
        rust_toolchain_versions(context)?;
        let tmp = tempdir().map_err(ActionError::TempDir)?;
        let dest = tmp.path();
        copy_partial_tree(context.source_dir(), dest, |path| {
            path.ends_with("Cargo.toml")
                || path.ends_with("Cargo.lock")
                || path.as_os_str().as_encoded_bytes().ends_with(b".rs")
        })?;

        let lockfile = dest.join("Cargo.lock");
        let deny1 = dest.join("deny.toml");
        let deny2 = dest.join(".cargo/deny.toml");
        let deny = deny1.exists() || deny2.exists();
        if lockfile.exists() {
            spawn(context, &["cargo", "fetch", "--locked"], dest)?;
            if deny {
                spawn(context, &["cargo", "deny", "--locked", "fetch"], dest)?;
            }
        } else {
            spawn(context, &["cargo", "fetch"], dest)?;
            if deny {
                spawn(context, &["cargo", "deny", "--locked", "fetch"], dest)?;
            }
        }

        Ok(())
    }
}

/// Check that Rust code is formatted in the canonical way, using `cargo fmt --check`.
#[derive(Debug, Clone, PartialEq, Eq, Serialize, Deserialize)]
pub struct CargoFmt;

impl ActionImpl for CargoFmt {
    fn execute(&self, context: &Context) -> Result<(), ActionError> {
        rust_toolchain_versions(context)?;
        spawn(context, &["cargo", "fmt", "--check"], context.source_dir())?;
        Ok(())
    }
}

/// Check that Rust code is correct and idiomatic using `cargo clippy`.
///
/// Warnings are treated as errors.
#[derive(Debug, Clone, PartialEq, Eq, Serialize, Deserialize)]
pub struct CargoClippy;

impl ActionImpl for CargoClippy {
    fn execute(&self, context: &Context) -> Result<(), ActionError> {
        rust_toolchain_versions(context)?;
        spawn(
            context,
            &[
                "cargo",
                "clippy",
                "--offline",
                "--locked",
                "--workspace",
                "--all-targets",
                "--no-deps",
                "--",
                "--deny",
                "warnings",
            ],
            context.source_dir(),
        )?;
        Ok(())
    }
}

/// Check Rust code for denied stuff, using `cargo deny`.
#[derive(Debug, Clone, PartialEq, Eq, Serialize, Deserialize)]
pub struct CargoDeny;

impl ActionImpl for CargoDeny {
    fn execute(&self, context: &Context) -> Result<(), ActionError> {
        rust_toolchain_versions(context)?;
        spawn(
            context,
            &[
                "cargo",
                "deny",
                "--offline",
                "--locked",
                "--workspace",
                "check",
            ],
            context.source_dir(),
        )?;
        Ok(())
    }
}

/// Render Rust documentation comments, using `cargo doc`.
#[derive(Debug, Clone, PartialEq, Eq, Serialize, Deserialize)]
pub struct CargoDoc;

impl ActionImpl for CargoDoc {
    fn execute(&self, context: &Context) -> Result<(), ActionError> {
        rust_toolchain_versions(context)?;
        spawn(
            context,
            &[
                "env",
                "RUSTDOCFLAGS=-D warnings",
                "cargo",
                "doc",
                "--workspace",
            ],
            context.source_dir(),
        )?;
        Ok(())
    }
}

/// Build a Rust project.
///
/// Run `cargo build` in a way that all parts of the project are built.
#[derive(Debug, Clone, PartialEq, Eq, Serialize, Deserialize)]
pub struct CargoBuild;

impl ActionImpl for CargoBuild {
    fn execute(&self, context: &Context) -> Result<(), ActionError> {
        rust_toolchain_versions(context)?;
        spawn(
            context,
            &[
                "cargo",
                "build",
                "--offline",
                "--locked",
                "--workspace",
                "--all-targets",
            ],
            context.source_dir(),
        )?;
        Ok(())
    }
}

/// Run automated test suite for a Rust project.
#[derive(Debug, Clone, PartialEq, Eq, Serialize, Deserialize)]
pub struct CargoTest;

impl ActionImpl for CargoTest {
    fn execute(&self, context: &Context) -> Result<(), ActionError> {
        rust_toolchain_versions(context)?;
        spawn(
            context,
            &["cargo", "test", "--offline", "--locked", "--workspace"],
            context.source_dir(),
        )?;
        Ok(())
    }
}

/// Install a Rust project into the artifacts directory.
#[derive(Debug, Clone, PartialEq, Eq, Serialize, Deserialize)]
pub struct CargoInstall;

impl ActionImpl for CargoInstall {
    fn execute(&self, context: &Context) -> Result<(), ActionError> {
        rust_toolchain_versions(context)?;
        spawn(
            context,
            &[
                "cargo",
                "install",
                "--offline",
                "--locked",
                "--bins",
                "--path=.",
                "--root",
                &context.artifacts_dir().to_string_lossy(),
            ],
            context.source_dir(),
        )?;
        Ok(())
    }
}

fn copy_partial_tree<P, PP, F>(src: P, dest: PP, wanted: F) -> Result<(), CargoError>
where
    P: AsRef<Path>,
    PP: AsRef<Path>,
    F: Fn(&Path) -> bool,
{
    let src = src.as_ref();
    let dest = dest.as_ref();
    debug!("copy_partial_tree: {} => {}", src.display(), dest.display());

    mkdir(dest)?;
    for e in WalkDir::new(src) {
        let path = e
            .map_err(|err| CargoError::CopyTreeWalkDir(src.into(), err))?
            .path()
            .to_path_buf();
        debug!("path={} wanted={}", path.display(), wanted(&path));
        if wanted(&path) {
            let dest = dest.join(path.strip_prefix(src).unwrap_or(&path));
            debug!(
                "copy_partial_tree: copy {} => {}",
                path.display(),
                dest.display()
            );
            if let Some(parent) = dest.parent() {
                if !parent.exists() {
                    mkdir(parent)?;
                }
            }
            copy_file(&path, &dest)?;
        }
    }
    Ok(())
}

/// Errors from Cargo actions.
#[derive(Debug, thiserror::Error)]
pub enum CargoError {
    /// Forwarded from `util` module.
    #[error(transparent)]
    Util(#[from] UtilError),

    /// Can't list files.
    #[error("failed to list contents of upload directory")]
    WalkDir(#[source] walkdir::Error),

    /// Can't find a .changes file.
    #[error("no *.changes file built for deb project")]
    NoChanges,

    /// Found more than one .changes file.
    #[error("more than one *.changes file built for deb project")]
    ManyChanges,

    /// Can't copy directory tree.
    #[error("failed to list files in directory {0} when copying files")]
    CopyTreeWalkDir(PathBuf, #[source] walkdir::Error),
}
