use std::path::PathBuf;

use serde::{Deserialize, Serialize};

use crate::{
    action::{ActionError, Context},
    action_impl::ActionImpl,
    vdrive::VirtualDriveBuilder,
};

/// Create a `tar` archive from a directory.
///
/// This is meant for internal use by Ambient. It can't be used in any kind
/// of plan, pre-plan, or post-plan. It can be used in a runnable plan.
/// It is generated by Ambient to set up execution of a runnable plan.
#[derive(Debug, Clone, PartialEq, Eq, Serialize, Deserialize)]
pub struct TarCreate {
    archive: PathBuf,
    directory: PathBuf,
}

impl TarCreate {
    /// Create a new `TarCreate` action.
    pub fn new(archive: PathBuf, directory: PathBuf) -> Self {
        Self { archive, directory }
    }
}

impl ActionImpl for TarCreate {
    fn execute(&self, _context: &Context) -> Result<(), ActionError> {
        VirtualDriveBuilder::default()
            .filename(&self.archive)
            .root_directory(&self.directory)
            .create()
            .map_err(|e| ActionError::TarCreate(self.archive.clone(), self.directory.clone(), e))?;
        Ok(())
    }
}

/// Extract a tar archive into a directory.
///
/// This is meant for internal use by Ambient. It can't be used in any kind
/// of plan, pre-plan, or post-plan. It can be used in a runnable plan.
/// It is generated by Ambient to set up execution of a runnable plan.
#[derive(Debug, Clone, PartialEq, Eq, Serialize, Deserialize)]
pub struct TarExtract {
    archive: PathBuf,
    directory: PathBuf,
}

impl TarExtract {
    /// Create a new `TarExtract` action.
    pub fn new(archive: PathBuf, directory: PathBuf) -> Self {
        Self { archive, directory }
    }
}

impl ActionImpl for TarExtract {
    fn execute(&self, _context: &Context) -> Result<(), ActionError> {
        let tar = VirtualDriveBuilder::default()
            .filename(&self.archive)
            .root_directory(&self.directory)
            .open()
            .map_err(|e| ActionError::TarOpen(self.archive.clone(), e))?;
        tar.extract_to(&self.directory).map_err(|e| {
            ActionError::TarExtract(self.archive.clone(), self.directory.clone(), e)
        })?;
        Ok(())
    }
}
