//! Look for common problems in configuration and CI plan.

use std::path::{Path, PathBuf};

use log::debug;

use crate::{
    action::{PostPlanAction, PrePlanAction},
    config::Config,
    project::{Project, Projects},
};

/// Look for common problems.
#[derive(Debug)]
pub struct Linter {
    lints: Vec<Box<dyn Lint>>,
}

impl Linter {
    /// Lint a configuration and projects list.
    pub fn lint(&self, config: &Config, projects: &Projects) -> Result<(), LinterError> {
        for (name, project) in projects.iter() {
            for lint in self.lints.iter() {
                debug!("linting project {name} with {}", lint.name());
                lint.check(config, name, project)?;
            }
        }
        Ok(())
    }
}

impl Default for Linter {
    fn default() -> Self {
        Self {
            lints: vec![Box::new(RsyncTarget), Box::new(HttpGet)],
        }
    }
}

trait Lint: std::fmt::Debug {
    fn name(&self) -> &'static str;
    fn check(
        &self,
        config: &Config,
        project_name: &str,
        project: &Project,
    ) -> Result<(), LinterError>;
}

#[derive(Debug)]
struct RsyncTarget;

impl Lint for RsyncTarget {
    fn name(&self) -> &'static str {
        "rsync_target"
    }

    fn check(
        &self,
        config: &Config,
        project_name: &str,
        project: &Project,
    ) -> Result<(), LinterError> {
        let uses_rsync = project
            .post_plan()
            .iter()
            .any(|a| matches!(a, PostPlanAction::Rsync | PostPlanAction::Rsync2));
        let configs_rsync = config.rsync_target_for_project(project_name).is_some();
        if uses_rsync && !configs_rsync {
            return Err(LinterError::rsync_target_missing(project_name));
        }
        Ok(())
    }
}

#[derive(Debug)]
struct HttpGet;

impl Lint for HttpGet {
    fn name(&self) -> &'static str {
        "http_get"
    }

    fn check(
        &self,
        _config: &Config,
        project_name: &str,
        project: &Project,
    ) -> Result<(), LinterError> {
        for action in project.pre_plan() {
            let mut filenames = vec![];
            if let PrePlanAction::HttpGet { items } = action {
                for item in items {
                    if filenames.contains(&item.filename) {
                        return Err(LinterError::http_get_duplicate_filename(
                            project_name,
                            &item.filename,
                        ));
                    }
                    filenames.push(item.filename.clone());
                }
            }
        }

        Ok(())
    }
}

/// Problems found by a linter.
#[derive(Debug, thiserror::Error)]
pub enum LinterError {
    /// `rsync` or `rsync2` action used in plan, but no `rsync` target configured.
    #[error(
        "rsync or rsync2 action used in project {project}, but no `rsync` target in configuration"
    )]
    RsyncTargetMissing {
        /// Name of project.
        project: String,
    },

    /// `http_get` action uses the same filename for items.
    #[error("in project {project_name} http_get pre-plan action uses the same filename for more than one item: {filename}")]
    HttpGetDuplicateFilename {
        /// Project name.
        project_name: String,

        /// Duplicate filename.
        filename: PathBuf,
    },
}

impl LinterError {
    fn rsync_target_missing(project_name: &str) -> Self {
        Self::RsyncTargetMissing {
            project: project_name.into(),
        }
    }

    fn http_get_duplicate_filename(project_name: &str, filename: &Path) -> Self {
        Self::HttpGetDuplicateFilename {
            project_name: project_name.into(),
            filename: filename.into(),
        }
    }
}
