use std::path::PathBuf;

use clap::Parser;

use super::{AmbientDriverError, Config, Leaf};
use ambient_ci::{
    project::{ProjectError, State},
    runlog::{RunLog, RunLogError},
};

/// Show available actions to the user.
#[derive(Debug, Parser)]
pub struct Log {
    /// Which project log should be written out?
    #[clap(conflicts_with = "filename", required_unless_present = "html")]
    project: Option<String>,

    /// Read run log from this file.
    #[clap(long)]
    filename: Option<PathBuf>,

    /// Format JSON log as HTML. Default is raw for both input and output.
    #[clap(long)]
    html: bool,

    /// Write log to this file, not stdout.
    #[clap(long)]
    output: Option<PathBuf>,
}

impl Log {
    fn write(&self, s: String) -> Result<(), LogError> {
        if let Some(output) = &self.output {
            std::fs::write(output, s.as_bytes())
                .map_err(|err| LogError::Write(output.to_path_buf(), err))?;
        } else {
            print!("{s}");
        }
        Ok(())
    }
}

impl Leaf for Log {
    fn run(&self, config: &Config, _runlog: &mut RunLog) -> Result<(), AmbientDriverError> {
        if let Some(project) = &self.project {
            let statedir = config.state();

            if !statedir.exists() {
                return Err(LogError::NoStateDir(project.clone(), statedir.into()))?;
            }

            let state = State::from_file(statedir, project).map_err(LogError::Project)?;
            if self.html {
                let filename = state.run_log_filename();
                let file = std::fs::File::open(&filename)
                    .map_err(|err| LogError::OpenJson(filename.clone(), err))?;
                let runlog = RunLog::read_jsonl(file)
                    .map_err(|err| LogError::LoadJson(filename.to_path_buf(), err))?;
                self.write(runlog.to_html().to_string())?;
            } else {
                let filename = state.raw_log_filename();
                let data = std::fs::read(&filename)
                    .map_err(|err| LogError::Read(filename.clone(), err))?;
                self.write(String::from_utf8_lossy(&data).to_string())?;
            }
        } else if let Some(filename) = &self.filename {
            if self.html {
                let file = std::fs::File::open(filename)
                    .map_err(|err| LogError::OpenJson(filename.to_path_buf(), err))?;
                let runlog = RunLog::read_jsonl(file)
                    .map_err(|err| LogError::LoadJson(filename.to_path_buf(), err))?;
                self.write(runlog.to_html().to_string())?;
            } else {
                let data =
                    std::fs::read(filename).map_err(|err| LogError::Read(filename.clone(), err))?;
                self.write(String::from_utf8_lossy(&data).to_string())?;
            }
        }

        Ok(())
    }
}

#[derive(Debug, thiserror::Error)]
pub enum LogError {
    #[error("state directory for project {0} does not exist: {1}")]
    NoStateDir(String, PathBuf),

    #[error(transparent)]
    Project(#[from] ProjectError),

    #[error("failed top open Ambient JSON run log file {0}")]
    OpenJson(PathBuf, #[source] std::io::Error),

    #[error("failed to load Ambient JSON run log file {0}")]
    LoadJson(PathBuf, #[source] RunLogError),

    #[error("failed to read log file {0}")]
    Read(PathBuf, #[source] std::io::Error),

    #[error("failed to write output to {0}")]
    Write(PathBuf, #[source] std::io::Error),
}
