//! Utility functions for git operations.

use std::{path::Path, process::Command};

use log::trace;

const UNKNOWN_EXIT: i32 = -1;

/// Is a directory a git checkout?
pub fn is_git(dirname: &Path) -> bool {
    let dotgit = dirname.join(".git");
    trace!(".git exists? {}", dotgit.exists());
    dirname.exists() && git_head(dirname).is_ok()
}

/// Return commit id for HEAD in a git checkout.
pub fn git_head(dirname: &Path) -> Result<String, GitError> {
    let stdout = git(&["rev-parse", "HEAD"], dirname)?;
    let head = stdout.trim().into();
    trace!("git HEAD in {} is {}", dirname.display(), head);
    Ok(head)
}

/// Is the git checkout clean?
pub fn git_is_clean(dirname: &Path) -> bool {
    if let Ok(stdout) = git(&["status", "--short"], dirname) {
        stdout.is_empty()
    } else {
        false // some error happened, assume checkout is not clean
    }
}

// Run git command in a directory; if successful, return stdout as text.
fn git(args: &[&str], dirname: &Path) -> Result<String, GitError> {
    let output = Command::new("git")
        .args(args)
        .current_dir(dirname)
        .output()
        .map_err(GitError::Invoke)?;
    let exit = output.status.code().unwrap_or(UNKNOWN_EXIT);
    if exit == 0 {
        Ok(String::from_utf8_lossy(&output.stdout).into())
    } else {
        Err(GitError::Failed(
            exit,
            String::from_utf8_lossy(&output.stdout).into(),
        ))
    }
}

/// Possible errors from git operations.
#[derive(Debug, thiserror::Error)]
pub enum GitError {
    #[error("failed to run git at all")]
    Invoke(#[source] std::io::Error),

    #[error("git failed with exit code {0}; stderr:\n{1}")]
    Failed(i32, String),
}
