# Introduction

`ambient` runs CI for a project in a VM.

# Acceptance criteria for `ambient`
## Smoke test

_Want:_ The `ambient` tool can show it's runtime
configuration.

_Why:_ If this doesn't work, there's no hope of anything
else working, either.

_Who:_ Lars


~~~scenario
given an installed ambient
given file .config/ambient/config.yaml from config.yaml
when I run ambient config
then stdout contains ""executor": "ambient-execute-plan""
~~~

~~~{#config.yaml .file .yaml}
executor: ambient-execute-plan
image_store: image-store
~~~

## Reports its version ##

_Want:_ `ambient` can be queried for its version.

_Why:_ This is useful for diagnosing problems, and also acts as a
smoke test: if this works, we know the adapter is installed and can be
run.

~~~scenario
given an installed ambient
when I run ambient --version
then stdout matches regex ^ambient \d+\.\d+\.\d+@.*$
~~~

## Backwards compatible QEMU configuration ##

_Want:_ The `ambient` tool understands its own older configuration
files, but warns about them.

_Why:_ This make migration to the newer configuration format less
harsh.

_Who:_ Lars


~~~scenario
given an installed ambient
given file .config/ambient/config.yaml from old-config.yaml
when I run ambient config --output config.json
then stderr contains "deprecated"
then stderr contains "cpus"
then stderr contains "memory"
then JSON file config.json contains qemu.cpus=8
then JSON file config.json contains qemu.memory=16 GB
~~~

~~~{#old-config.yaml .file .yaml}
cpus: 8
memory: "16 GB"
~~~

## Allows specifying a configuration file ##

_Want:_ The `ambient` tool allows the user to specify
which configuration file to use.

_Why:_ This is very convenient when one wants to try things
and temporarily not use the default configuration file. This happens
both for experimentation and for testing.

_Who:_ Lars


~~~scenario
given an installed ambient
given file .config/ambient/config.yaml from  special.yaml
given file other.yaml
when I run env ambient --config other.yaml config
then stdout contains "xyzzy"
then stdout contains ""artifacts_max_size": "1 TB""
when I run env ambient --no-config --config other.yaml config
then stdout doesn't contain "xyzzy"
then stdout contains ""artifacts_max_size": "1 TB""
~~~

~~~{#special.yaml .file .yaml}
executor: xyzzy
~~~

~~~{#other.yaml .file .yaml}
artifacts_max_size: 1TB
~~~

## Obeys TMPDIR ##

_Want:_ The default temporary directory is `$TMPDIR`, if set, or
`/tmp` otherwise.

_Why:_ This is expected Unix behavior.

_Who:_ Lars.

~~~scenario
given an installed ambient
given file special.yaml
when I run env --unset TMPDIR ambient config
then stdout contains ""tmpdir": "/tmp""
when I run env TMPDIR=/xyzzy ambient config
then stdout contains ""tmpdir": "/xyzzy""
~~~

## Capture build log when running ##

_Want:_ The stdout/stderr output of the commands executed by
the CI run must be captured.

_Why:_ This is the primary way for the user to know what
happened during a run.

_Who:_ Lars.

Note that we can't use `.` for the current directory as the `source`
field in the project file due to how Subplot sets up a temporary data
directory for each scenario, and sets `TMPDIR` to that directory.
This would mean that `.` as `source` would create a tar archive in the
temporary directory that tries to add itself, which creates an
infinitely large tar archive.

~~~scenario
given an installed ambient
given an Ambient VM image ambient.qcow2
given file .config/ambient/config.yaml from config.yaml
given file hello.yaml
given a directory srcdir
when I run env AMBIENT_LOG=trace ambient run --projects hello.yaml
then command is successful
~~~

~~~{#hello.yaml .file .yaml}
projects:
  hello:
    image: ambient.qcow2
    source: srcdir
    plan:
    - action: shell
      shell: |
        echo well hello there, friend
~~~

## Store run log in project directory ##

_Want:_ The run log should be stored persistently in the project state
directory, and there is a way to retrieve it later.

_Why:_ Run logs are often needed long after the run has happened.

_Who:_ Lars.

~~~scenario
given an installed ambient
given an Ambient VM image ambient.qcow2
given file .config/ambient/config.yaml from config.yaml
given file hello.yaml
given a directory srcdir
when I run env AMBIENT_LOG=debug  ambient run --projects hello.yaml
when I run find .local/state/ambient-ci
when I run cat .local/state/ambient-ci/projects/hello/run.log
then file .local/state/ambient-ci/projects/hello/run.log contains "hello there, friend"
when I run ambient log hello
then stdout contains "hello there, friend"
~~~


## Relative filenames in project files ##

_Want:_ When a project file has a relative filename, it's
relative to the directory containing the project file.

_Why:_ The user can then run `ambient run` from
anywhere, not just the source directory.

_Who:_ Lars.

~~~scenario
given an installed ambient
given file .config/ambient/config.yaml from config.yaml
given a directory path/to/project/srcdir
given an Ambient VM image ambient.qcow2
when I run mv ambient.qcow2 path/to/project/
given file path/to/project/hello.yaml from hello.yaml
when I run env AMBIENT_LOG=trace ambient run --projects path/to/project/hello.yaml
then command is successful
~~~

## Working directory in pre- and post-plan actions ##

_Want:_ When a pre- or post-plan action is executed, the
current working directory should be the source directory.

_Why:_ Many actions can only usefully be executed in the
source directory.

_Who:_ Lars.

Note that we can't use `.` for the current directory as the `source`
field in the project file due to how Subplot sets up a temporary data
directory for each scenario, and sets `TMPDIR` to that directory.
This would mean that `.` as `source` would create a tar archive in the
temporary directory that tries to add itself, which creates an
infinitely large tar archive.

~~~scenario
given an installed ambient
given file .config/ambient/config.yaml from config.yaml
given an Ambient VM image ambient.qcow2
given file cwd.yaml
given a directory path/to/project/srcdir
when I run env AMBIENT_LOG=trace ambient run --projects cwd.yaml
then stderr matches regex INFO.*cwd:.*/path/to/project/srcdir
~~~

~~~{#cwd.yaml .file .yaml}
projects:
  pwd:
    image: ambient.qcow2
    source: path/to/project/srcdir
    pre_plan:
    - action: pwd
    post_plan:
    - action: pwd
    plan:
    - action: shell
      shell: |
        pwd
~~~

## Run CI only for some projects ##

_Want:_ When the projects file has more than one project, user
can choose only specific ones to run CI for.

_Why:_ User may be only interested in one project right now.

_Who:_ Lars

~~~scenario
given an installed ambient
given file .config/ambient/config.yaml from config.yaml
given an Ambient VM image ambient.qcow2
given file multi.yaml
given a directory foo
given a directory bar
when I run env AMBIENT_LOG=debug ambient run --dry-run --projects multi.yaml foo
then stderr contains "project foo: NOT running CI"
then stderr doesn't contain "project bar:"
~~~

~~~{#multi.yaml .file .yaml}
projects:
  foo:
    image: ambient.qcow2
    source: foo
    plan:
    - action: shell
      shell: |
        echo foo project
  bar:
    image: ambient.qcow2
    source: bar
    plan:
    - action: shell
      shell: |
        echo bar project
~~~

## List names of projects ##

_Want:_ The user can list names of projects in a projects file.

_Why:_ This is handy for checking and scripting.

_Who:_ Lars

~~~scenario
given an installed ambient
given file .config/ambient/config.yaml from config.yaml
given an Ambient VM image ambient.qcow2
given file multi.yaml
given a directory foo
given a directory bar
when I run ambient projects --oneline --projects multi.yaml
then stdout contains "{"projects":["bar","foo"]}"
~~~

## List names of actions ##

_Want:_ The user can list names of projects in a projects file.

_Why:_ This is handy for checking and scripting.

_Who:_ Lars

~~~scenario
given an installed ambient
given file .config/ambient/config.yaml from config.yaml
given an Ambient VM image ambient.qcow2
when I run ambient actions
then stdout contains ""pre_actions": [\n"
then stdout contains ""actions": [\n"
then stdout contains ""post_actions": [\n"
~~~
## Cache persists between CI runs ##

_Want:_ Cache data is persisted between CI runs.

_Why:_ This allows incrementally building a project after
changes.

_Who:_ Lars

~~~scenario
given an installed ambient
given file .config/ambient/config.yaml from config.yaml
given an Ambient VM image ambient.qcow2
given file cache.yaml
given a directory srcdir

when I run ambient run --projects cache.yaml
when I run ambient log hello
then stdout contains "counter is now 1."

when I run ambient run --projects cache.yaml
when I run ambient log hello
then stdout contains "counter is now 2."

when I run ambient run --projects cache.yaml
when I run ambient log hello
then stdout contains "counter is now 3."
~~~

~~~{#cache.yaml .file .yaml}
projects:
  hello:
    image: ambient.qcow2
    source: srcdir
    plan:
    - action: shell
      shell: |
        cache=/workspace/cache
        counter="$cache/counter"
        if [ -e "$counter" ]; then
            n="$(expr "$(cat "$counter")" + 1)"
            echo "$n" > "$counter"
        else
            echo 1 > "$counter"
        fi
        echo "counter is now $(cat "$counter")."
        find "$cache" -ls
~~~

## Publish files via rsync ##

_Want:_ Artifacts can be published via rsync to a server.

_Why:_ This allows publishing many kinds of things.

_Who:_ Lars

~~~scenario
given an installed ambient
given file .config/ambient/config.yaml from config.yaml
given an Ambient VM image ambient.qcow2
given file rsync.yaml
given a directory srcdir
given file srcdir/data.dat from rsync.yaml

given a directory serverdir
when I run env AMBIENT_LOG=trace ambient run --projects rsync.yaml --target serverdir
then file serverdir/hello.txt contains "hello, world"
~~~

~~~{#rsync.yaml .file .yaml}
projects:
  hello:
    image: ambient.qcow2
    source: srcdir
    plan:
    - action: shell
      shell: |
        echo hello, world > /workspace/artifacts/hello.txt
    post_plan:
    - action: rsync
~~~

## Verify an image is acceptable to Ambient

<!-- This is too slow for now.

_What:_ The user can use `ambient` to verify an image is
acceptable for use with Ambient.

_Why:_ This is needed so that a newly built image can be checked.

_Who:_ Lars

~~~scenario
given an installed ambient
given file .config/ambient/config.yaml from config.yaml
given an Ambient VM image ambient.qcow2
when I run ambient image import default ambient.qcow2
then file image-store/default.qcow2 exists
when I run env AMBIENT_LOG=debug ambient image verify --name default --executor ambient-execute-plan
then command is successful
~~~

-->

## Manage VM images ##

_Want:_ `ambient` has an image store and the user can manage
images in the image store via the tool.

_Why:_ A user may need to have many images, perhaps with different
operating systems or versions of thereof, or with different
dependencies installed.

~~~scenario
given an installed ambient
given file .config/ambient/config.yaml from config.yaml
given file dummy.qcow2

when I run ambient image list
then stdout is exactly ""

when I run ambient image import default dummy.qcow2 -d mcdummy -u https://example.com/image.qcow2 --uefi
then file image-store/default.qcow2 exists
when I run ambient image list
then stdout is exactly "default\n"

when I run ambient image show default
then stdout contains ""name": "default""
then stdout contains ""filename": "default.qcow2""
then stdout contains ""sha256": "01ba4719c80b6fe911b091a7c05124b64eeece964e09c058ef8f9805daca546b""
then stdout contains ""description": "mcdummy""
then stdout contains ""url": "https://example.com/image.qcow2""
then stdout contains ""import_date":"
then stdout contains ""uefi": true"

when I run env AMBIENT_LOG=debug ambient image remove default
when I run ambient image list
then stdout is exactly ""
~~~

~~~{#dummy.qcow2 .file}
~~~

## Handles symlinks in source tree ##

_Want:_ When constructing the source tree for the VM, `ambient` does
not follow symbolic links.

_Why:_ We don't want a project to create a link a file outside the
source, but we do want to handle broken links. These are both best
handled by not following symbolic links, when constructing the tar
archive of the source tree.

~~~scenario
given an installed ambient
given an Ambient VM image ambient.qcow2
given file .config/ambient/config.yaml from config.yaml
given file symlink.yaml
given a directory srcdir
when I run touch srcdir/README
when I run ln -s README srcdir/ok
when I run ln -s no-such-file srcdir/broken
when I run ln -s /etc/passwd srcdir/passwd
when I run env AMBIENT_LOG=trace ambient run --projects symlink.yaml
when I run ambient log hello
then stdout contains "./ok -> README"
then stdout contains "./broken -> no-such-file"
then stdout contains "./passwd -> /etc/passwd"
~~~

~~~{#symlink.yaml .file .yaml}
projects:
  hello:
    image: ambient.qcow2
    source: srcdir
    plan:
    - action: shell
      shell: |
        find -ls
~~~
# Acceptance criteria for `ambient-execute-plan`

`ambient-execute-plan` is the program that executes actions inside the
virtual machine, specifically actions in the "runnable plan". The
scenarios in this chapter verify that it work, without having to run a
full virtual machine to do so.

## Default log level is trace

_Want:_ The default log level of `ambient-execute-plan` is TRACE.

_Why:_ This means we get the all the log output we can, which is
useful for remote debugging.
~~~scenario
given an installed ambient
given file plan.yaml from shell-action.yaml
when I run ambient-execute-plan
then stderr contains "TRACE"
~~~

## Executes `shell` action

_Want:_ `action-execute-plan` can execute a `shell` action with a
shell script snippet.

~~~scenario
given an installed ambient
given file plan.yaml from shell-action.yaml
when I run ambient-execute-plan
then stdout contains "hello, world"
~~~

~~~{#shell-action.yaml .file .yaml}
steps:
- action: shell
  shell: |
    echo hello, world
source_dir: .
~~~

## Executes `pwd` action

_Want:_ `action-execute-plan` can execute a `pwd` action.

~~~scenario
given an installed ambient
given file plan.yaml from pwd-action.yaml
when I run ambient-execute-plan
then stderr contains "cwd: /tmp"
~~~

~~~{#pwd-action.yaml .file .yaml}
steps:
- action: pwd
  sourcedir: /tmp
source_dir: .
~~~

## Executes `mkdir` action

_Want:_ `action-execute-plan` can execute a `mkdir` action.

_Why:_ This is needed to set up the VM environment for running CI.

~~~scenario
given an installed ambient
given file plan.yaml from mkdir-action.yaml
when I run ambient-execute-plan
then stdout contains "./xyzzy\n"
~~~

~~~{#mkdir-action.yaml .file .yaml}
steps:
- action: mkdir
  pathname: xyzzy
- action: shell
  shell: |
    find -type d
source_dir: .
~~~

## Executes `tar_create` and `tar_extract` actions

_Want:_ `action-execute-plan` can execute a `tar_create` action and
its complement `tar_extract`.

_Why:_ This is needed to set up the VM environment for running CI.

~~~scenario
given an installed ambient
given file plan.yaml from tar-create-action.yaml
given a directory data
when I write "" to file data/foo.txt
when I run ambient-execute-plan
then file data2/foo.txt exists
when I run tar tvf tar.tar
then stdout contains "./foo.txt"
~~~

~~~{#tar-create-action.yaml .file .yaml}
steps:
- action: tar_create
  archive: tar.tar
  directory: data
- action: tar_extract
  archive: tar.tar
  directory: data2
source_dir: .
~~~

## Executes a custom action

_Want:_ `action-execute-plan` can execute a custom action provided in
the source tree under test.

_Why:_ This allows actions not built into Ambient.

~~~scenario
given an installed ambient
given file plan.yaml from custom-action.yaml
given file .ambient/hello from hello-action.sh
when I run chmod +x .ambient/hello
when I run ambient-execute-plan
then stdout contains "hey there"
~~~

~~~{#custom-action.yaml .file .yaml}
steps:
- action: custom
  name: hello
  args:
    greeting: hey
    whom: there
source_dir: .
~~~

~~~{#hello-action.sh .file .sh}
#!/bin/sh
args="$(cat)"
greeting="$(echo -n "$args" | jq -r .greeting)"
whom="$(echo -n "$args" | jq -r .whom)"
echo "$greeting $whom"
~~~
