#![allow(clippy::result_large_err)]

use std::path::{Path, PathBuf};

use log::trace;

use serde::{Deserialize, Serialize};

use crate::action::{RunnableAction, UnsafeAction};

#[derive(Debug, Clone, Default, Serialize, Deserialize)]
pub struct Plan {
    steps: Vec<UnsafeAction>,
}

impl Plan {
    pub fn from_file(filename: &Path) -> Result<Self, PlanError> {
        let plan = std::fs::read(filename).map_err(|e| PlanError::PlanOpen(filename.into(), e))?;
        let plan = serde_norway::from_slice(&plan)
            .map_err(|e| PlanError::PlanParse(filename.into(), e))?;
        Ok(plan)
    }

    pub fn to_file(&self, filename: &Path) -> Result<(), PlanError> {
        let plan = serde_norway::to_string(&self).map_err(PlanError::PlanSerialize)?;
        std::fs::write(filename, plan).map_err(|e| PlanError::PlanWrite(filename.into(), e))?;
        Ok(())
    }

    pub fn push(&mut self, action: UnsafeAction) {
        self.steps.push(action);
    }

    pub fn iter(&self) -> impl Iterator<Item = &UnsafeAction> {
        self.steps.iter()
    }
}

#[derive(Debug, Default, Clone, Serialize, Deserialize, Eq, PartialEq)]
pub struct RunnablePlan {
    steps: Vec<RunnableAction>,
    executor_drive: Option<String>,
    source_drive: Option<String>,
    artifact_drive: Option<String>,
    cache_drive: Option<String>,
    deps_drive: Option<String>,
    workspace_dir: Option<String>,
    source_dir: Option<String>,
    deps_dir: Option<String>,
    cache_dir: Option<String>,
    artifacts_dir: Option<String>,
}

impl RunnablePlan {
    pub fn parse_str(yaml: &str) -> Result<Self, PlanError> {
        serde_norway::from_str(yaml).map_err(PlanError::PlanParseStr)
    }

    pub fn from_file(filename: &Path) -> Result<Self, PlanError> {
        let plan = std::fs::read(filename).map_err(|e| PlanError::PlanOpen(filename.into(), e))?;
        let plan = String::from_utf8_lossy(&plan);
        eprintln!(
            "RunnablePlan::from_file: filename={}\n{}\n",
            filename.display(),
            plan
        );
        let plan: Self =
            serde_norway::from_str(&plan).map_err(|e| PlanError::PlanParse(filename.into(), e))?;

        Ok(plan)
    }

    pub fn to_string(&self) -> Result<String, PlanError> {
        serde_norway::to_string(self).map_err(PlanError::PlanSerialize)
    }

    pub fn to_file(&self, filename: &Path) -> Result<(), PlanError> {
        let plan = serde_norway::to_string(&self).map_err(PlanError::PlanSerialize)?;
        trace!("RunnablePlan:to_file:\n{plan}");
        std::fs::write(filename, plan).map_err(|e| PlanError::PlanWrite(filename.into(), e))?;
        Ok(())
    }

    pub fn push(&mut self, action: RunnableAction) {
        self.steps.push(action);
    }

    pub fn push_unsafe_actions<'a>(&mut self, actions: impl Iterator<Item = &'a UnsafeAction>) {
        for action in actions {
            self.push(RunnableAction::from_unsafe_action(action));
        }
    }

    pub fn iter(&self) -> impl Iterator<Item = &RunnableAction> {
        self.steps.iter()
    }

    pub fn execute(&self) -> Result<(), PlanError> {
        for action in self.steps.iter() {
            action.execute(self)?;
        }
        Ok(())
    }

    pub fn executor_drive(&self) -> Option<&String> {
        self.executor_drive.as_ref()
    }

    pub fn source_drive(&self) -> Option<&String> {
        self.source_drive.as_ref()
    }

    pub fn run_artifact_drive(&self) -> Option<&String> {
        self.artifact_drive.as_ref()
    }

    pub fn cache_drive(&self) -> Option<&String> {
        self.cache_drive.as_ref()
    }

    pub fn deps_drive(&self) -> Option<&String> {
        self.deps_drive.as_ref()
    }

    pub fn workspace_dir(&self) -> Option<&String> {
        self.workspace_dir.as_ref()
    }

    pub fn source_dir(&self) -> Option<&String> {
        self.source_dir.as_ref()
    }

    pub fn deps_dir(&self) -> Option<&String> {
        self.deps_dir.as_ref()
    }

    pub fn cache_dir(&self) -> Option<&String> {
        self.cache_dir.as_ref()
    }

    pub fn artifacts_dir(&self) -> Option<&String> {
        self.artifacts_dir.as_ref()
    }

    pub fn set_executor_drive(&mut self, path: &str) {
        self.executor_drive = Some(path.into());
    }

    pub fn set_source_drive(&mut self, path: &str) {
        self.source_drive = Some(path.into());
    }

    pub fn set_artifact_drive(&mut self, path: &str) {
        self.artifact_drive = Some(path.into());
    }

    pub fn set_cache_drive(&mut self, path: &str) {
        self.cache_drive = Some(path.into());
    }

    pub fn set_deps_drive(&mut self, path: &str) {
        self.deps_drive = Some(path.into());
    }

    pub fn set_workspace_dir(&mut self, path: &str) {
        self.workspace_dir = Some(path.into());
    }

    pub fn set_source_dir(&mut self, path: &str) {
        self.source_dir = Some(path.into());
    }

    pub fn set_deps_dir(&mut self, path: &str) {
        self.deps_dir = Some(path.into());
    }

    pub fn set_cache_dir(&mut self, path: &str) {
        self.cache_dir = Some(path.into());
    }

    pub fn set_artifacts_dir(&mut self, path: &str) {
        self.artifacts_dir = Some(path.into());
    }
}

#[derive(Debug, thiserror::Error)]
pub enum PlanError {
    #[error("failed to read CI plan file: {0}")]
    PlanOpen(PathBuf, #[source] std::io::Error),

    #[error("failed to parse CI plan file as YAML: {0}")]
    PlanParse(PathBuf, #[source] serde_norway::Error),

    #[error("failed to parse CI plan")]
    PlanParseStr(#[source] serde_norway::Error),

    #[error("failed to serialize CI plan as YAML")]
    PlanSerialize(#[source] serde_norway::Error),

    #[error("failed to write CI plan file: {0}")]
    PlanWrite(PathBuf, #[source] std::io::Error),

    #[error(transparent)]
    Action(#[from] crate::action::ActionError),
}

#[cfg(test)]
mod test {
    use super::*;

    #[test]
    fn round_trip() -> Result<(), Box<dyn std::error::Error>> {
        let mut plan = RunnablePlan::default();
        plan.set_source_dir("/src");

        let s = plan.to_string()?;
        let des = RunnablePlan::parse_str(&s)?;

        assert_eq!(plan, des);
        Ok(())
    }
}
