use std::path::PathBuf;

use clap::Parser;
use serde::Serialize;

use ambient_ci::{
    plan::RunnablePlan,
    project::{ProjectError, Projects, State},
    run::{construct_runnable_plan, runnable_plan_from_trusted_actions},
};

use super::{AmbientDriverError, Config, Leaf};

/// Prepare image.
#[derive(Debug, Parser)]
pub struct Plan {
    /// Project specification file. May contain any number of projects.
    #[clap(long)]
    projects: Option<PathBuf>,

    /// Which project to show plan for?
    project: String,

    /// rsync target for publishing ikiwiki output.
    #[clap(long, aliases = ["rsync", "target"])]
    rsync_target: Option<String>,

    /// dput target for publishing .deb package.
    #[clap(long, alias = "dput")]
    dput_target: Option<String>,
}

impl Leaf for Plan {
    fn run(&self, config: &Config) -> Result<(), AmbientDriverError> {
        let projects = if let Some(filename) = &self.projects {
            filename.to_path_buf()
        } else {
            config.projects().into()
        };

        let projects =
            Projects::from_file(&projects).map_err(|err| PlanError::FromFile(projects, err))?;
        let project = projects
            .get(&self.project)
            .ok_or(PlanError::NoSuchProject(self.project.clone()))?;

        let rsync_target = if let Some(s) = &self.rsync_target {
            Some(s.as_str())
        } else {
            config.rsync_target()
        };

        let dput_target = if let Some(s) = &self.dput_target {
            Some(s.as_str())
        } else {
            config.dput_target()
        };
        let state = State::from_file(config.state(), &self.project).map_err(PlanError::State)?;
        let pre_plan = runnable_plan_from_trusted_actions(
            project,
            &state,
            rsync_target,
            dput_target,
            project.pre_plan(),
        );
        let plan = construct_runnable_plan(project.plan()).map_err(PlanError::Runnable)?;
        let post_plan = runnable_plan_from_trusted_actions(
            project,
            &state,
            rsync_target,
            dput_target,
            project.post_plan(),
        );

        let plans = Plans {
            pre_plan,
            plan,
            post_plan,
        };
        let json = serde_json::to_string_pretty(&plans).map_err(PlanError::ToJson)?;
        println!("{json}");

        Ok(())
    }
}

#[derive(Serialize)]
struct Plans {
    pre_plan: RunnablePlan,
    plan: RunnablePlan,
    post_plan: RunnablePlan,
}

#[derive(Debug, thiserror::Error)]
pub enum PlanError {
    #[error("failed to load project list from file {0}")]
    FromFile(PathBuf, #[source] ProjectError),

    #[error("failed to serialize plans to JSON")]
    ToJson(#[source] serde_json::Error),

    #[error("can't find project {0}")]
    NoSuchProject(String),

    #[error("can't construct runnable plan")]
    Runnable(#[source] ambient_ci::run::RunError),

    #[error("can't load project state")]
    State(#[source] ambient_ci::project::ProjectError),
}
