use std::{
    fs::{copy, create_dir_all, metadata, read, set_permissions, write, File, Metadata},
    os::unix::fs::PermissionsExt,
    path::{Path, PathBuf},
    process::Command,
    time::UNIX_EPOCH,
};

use clingwrap::runner::{CommandError, CommandRunner};
use log::{debug, error, info, trace};
use reqwest::{blocking::Client, header::IF_MODIFIED_SINCE, StatusCode};
use time::{macros::format_description, OffsetDateTime};
use walkdir::WalkDir;

pub fn copy_partial_tree<P, PP, F>(src: P, dest: PP, wanted: F) -> Result<(), UtilError>
where
    P: AsRef<Path>,
    PP: AsRef<Path>,
    F: Fn(&Path) -> bool,
{
    let src = src.as_ref();
    let dest = dest.as_ref();
    debug!("copy_partial_tree: {} => {}", src.display(), dest.display());

    mkdir(dest)?;
    for e in WalkDir::new(src) {
        let path = e
            .map_err(|err| UtilError::CopyTreeWalkDir(src.into(), err))?
            .path()
            .to_path_buf();
        if wanted(&path) {
            let dest = dest.join(path.strip_prefix(src).unwrap_or(&path));
            debug!(
                "copy_partial_tree: copy {} => {}",
                path.display(),
                dest.display()
            );
            if let Some(parent) = dest.parent() {
                if !parent.exists() {
                    // mkdir(parent)?;
                }
            }
            copy_file(&path, &dest)?;
        }
    }
    Ok(())
}

pub fn find_dirs_with_files(
    dirs: &mut Vec<PathBuf>,
    root: &Path,
    filename: &Path,
) -> Result<(), walkdir::Error> {
    fn list<F>(dir: &Path, wanted: F) -> Result<Vec<PathBuf>, walkdir::Error>
    where
        F: Fn(Metadata) -> bool,
    {
        let mut paths = vec![];
        for e in WalkDir::new(dir).min_depth(1).max_depth(1) {
            let e = e?;
            if wanted(e.metadata()?) {
                paths.push(e.path().to_path_buf());
            }
        }
        Ok(paths)
    }

    if !root.ends_with(Path::new(".git")) {
        let files = list(root, |meta| meta.is_file())?;
        if files.iter().any(|p| p.ends_with(filename)) {
            dirs.push(root.to_path_buf());
        } else {
            for subdir in list(root, |meta| meta.is_dir())? {
                find_dirs_with_files(dirs, &subdir, filename)?;
            }
        }
    }
    Ok(())
}

pub fn http_get_to_file(url: &str, filename: &Path) -> Result<(), UtilError> {
    info!("http_get_to_file: url={url:?} => {}", filename.display());

    let timestamp = if let Ok(meta) = filename.metadata() {
        meta.modified().unwrap_or(UNIX_EPOCH)
    } else {
        UNIX_EPOCH
    };

    let fmt = format_description!(
        "[weekday repr:short], [day padding:zero] [month repr:short] [year] [hour]:[minute]:[second] GMT"
    );
    let ts = OffsetDateTime::from(timestamp)
        .format(fmt)
        .map_err(UtilError::TimeFormat)?;

    let client = Client::builder().build().map_err(UtilError::ClientBuild)?;
    let req = client
        .get(url)
        .header(IF_MODIFIED_SINCE, ts)
        .build()
        .map_err(UtilError::Client)?;

    let resp = client
        .execute(req)
        .map_err(|err| UtilError::Get(url.into(), err))?;

    match resp.status() {
        StatusCode::NOT_MODIFIED => {
            info!("http_get_to_file: not modified, existing file is OK");
        }
        StatusCode::OK => {
            info!("http_get_to_file: downloaded new copy of file");
            let body = resp
                .bytes()
                .map_err(|err| UtilError::GetBody(url.into(), err))?;
            write_file(filename, &body)?;
        }
        x => {
            error!("http_get_to_file: unwanted status code {x}");
            return Err(UtilError::UnwantedStatus(x));
        }
    }

    Ok(())
}

pub fn create_file(filename: &Path) -> Result<PathBuf, UtilError> {
    debug!("create file {}", filename.display());
    File::create(filename).map_err(|e| UtilError::CreateFile(filename.into(), e))?;
    Ok(filename.into())
}

pub fn mkdir(dirname: &Path) -> Result<(), UtilError> {
    create_dir_all(dirname).map_err(|e| UtilError::CreateDir(dirname.into(), e))?;
    Ok(())
}

pub fn mkdir_child(parent: &Path, subdir: &str) -> Result<PathBuf, UtilError> {
    let pathname = parent.join(subdir);
    create_dir_all(&pathname).map_err(|e| UtilError::CreateDir(pathname.clone(), e))?;
    Ok(pathname)
}

pub fn recreate_dir(dirname: &Path) -> Result<(), UtilError> {
    if dirname.exists() {
        std::fs::remove_dir_all(dirname).map_err(|e| UtilError::RemoveDir(dirname.into(), e))?;
    }
    mkdir(dirname)?;
    Ok(())
}

pub fn cat_text_file(filename: &Path) -> Result<String, UtilError> {
    let data = read(filename).map_err(|err| UtilError::Read(filename.into(), err))?;
    let text = String::from_utf8(data).map_err(|err| UtilError::Utf8(filename.into(), err))?;
    Ok(text)
}

pub fn write_file(filename: &Path, data: &[u8]) -> Result<(), UtilError> {
    write(filename, data).map_err(|err| UtilError::WriteFile(filename.into(), err))
}

pub fn copy_file(src: &Path, dst: &Path) -> Result<(), UtilError> {
    trace!("copy file {} to {}", src.display(), dst.display());
    copy(src, dst).map_err(|err| UtilError::Copy(src.into(), dst.into(), err))?;
    Ok(())
}

pub fn copy_file_rw(src: &Path, dst: &Path) -> Result<(), UtilError> {
    copy_file(src, dst)?;
    trace!("set {} to not be read-only", dst.display());
    let mut perms = std::fs::metadata(dst)
        .map_err(|err| UtilError::GetMetadata(dst.into(), err))?
        .permissions();
    perms.set_mode(0o644);
    std::fs::set_permissions(dst, perms)
        .map_err(|err| UtilError::SetPermissions(dst.into(), err))?;
    Ok(())
}

pub fn rsync_dir(src: &Path, dest: &Path) -> Result<(), UtilError> {
    let src = src.join(".");
    let dest = dest.join(".");
    info!("rsync {} -> {}", src.display(), dest.display());
    let mut cmd = Command::new("rsync");
    cmd.arg("-a").arg("--del").arg(&src).arg(&dest);

    let mut runner = CommandRunner::new(cmd);
    runner.capture_stdout();
    runner.capture_stderr();

    let output = runner
        .execute()
        .map_err(|err| UtilError::Execute("rsync", err))?;

    if output.status.code() != Some(0) {
        let stderr = String::from_utf8_lossy(&output.stderr);
        return Err(UtilError::Rsync(
            src.clone(),
            dest.display().to_string(),
            stderr.into(),
        ));
    }

    Ok(())
}

pub fn rsync_server(src: &Path, target: &str) -> Result<(), UtilError> {
    let src = src.join(".");
    let target = format!("{target}/.");
    info!("rsync {} -> {}", src.display(), target);
    let mut cmd = Command::new("rsync");
    cmd.arg("-a").arg("--del").arg(&src).arg(&target);

    let mut runner = CommandRunner::new(cmd);
    runner.capture_stdout();
    runner.capture_stderr();

    let output = runner
        .execute()
        .map_err(|err| UtilError::Execute("rsync", err))?;

    if output.status.code() != Some(0) {
        let stderr = String::from_utf8_lossy(&output.stderr);
        return Err(UtilError::Rsync(src.clone(), target.clone(), stderr.into()));
    }

    Ok(())
}

pub fn dput(dest: &str, changes: &Path) -> Result<(), UtilError> {
    info!("dput {} {}", dest, changes.display());
    let mut cmd = Command::new("dput");
    cmd.arg(dest).arg(changes);

    let mut runner = CommandRunner::new(cmd);
    runner.capture_stdout();
    runner.capture_stderr();

    let output = runner
        .execute()
        .map_err(|err| UtilError::Execute("dput", err))?;

    if output.status.code() != Some(0) {
        let stderr = String::from_utf8_lossy(&output.stderr);
        return Err(UtilError::Dput(
            dest.into(),
            changes.to_path_buf(),
            stderr.into(),
        ));
    }

    Ok(())
}

pub fn changes_file(dir: &Path) -> Result<PathBuf, UtilError> {
    let mut result = None;
    for entry in WalkDir::new(dir).into_iter() {
        let path = entry.map_err(UtilError::WalkDir)?.path().to_path_buf();
        debug!("found {}", path.display());
        if path.display().to_string().ends_with(".changes") {
            if result.is_some() {
                return Err(UtilError::ManyChanges);
            }
            result = Some(path);
        }
    }

    if let Some(path) = result {
        Ok(path)
    } else {
        Err(UtilError::NoChanges)
    }
}

pub fn tar_create(tar_filename: &Path, dirname: &Path) -> Result<(), UtilError> {
    let output = Command::new("tar")
        .arg("-cvf")
        .arg(tar_filename)
        .arg("-C")
        .arg(dirname)
        .arg(".")
        .output()
        .map_err(|err| UtilError::Tar("create", dirname.into(), err))?;

    if let Some(exit) = output.status.code() {
        if exit != 0 {
            let stderr = String::from_utf8_lossy(&output.stderr).to_string();
            return Err(UtilError::TarFailed("create", dirname.into(), exit, stderr));
        }
    }

    Ok(())
}

pub fn tar_extract(tar_filename: &Path, dirname: &Path) -> Result<(), UtilError> {
    let output = Command::new("tar")
        .arg("-xvvvf")
        .arg(tar_filename)
        .arg("-C")
        .arg(dirname)
        .arg("--no-same-owner")
        .output()
        .map_err(|err| UtilError::Tar("extract", dirname.into(), err))?;

    if let Some(exit) = output.status.code() {
        if exit != 0 {
            let stderr = String::from_utf8_lossy(&output.stderr).to_string();
            return Err(UtilError::TarFailed(
                "extract",
                dirname.into(),
                exit,
                stderr,
            ));
        }
    }

    Ok(())
}

pub fn write_executable(filename: &Path, data: &[u8]) -> Result<(), UtilError> {
    // Unix mode bits for an executable file: read/write/exec for
    // owner, read/exec for group and others
    const EXECUTABLE: u32 = 0o755;

    write(filename, data).map_err(|err| UtilError::WriteFile(filename.into(), err))?;
    let meta = metadata(filename).map_err(|err| UtilError::GetMetadata(filename.into(), err))?;
    let mut perm = meta.permissions();
    perm.set_mode(EXECUTABLE);
    set_permissions(filename, perm).map_err(|err| UtilError::MakeExec(filename.into(), err))?;
    Ok(())
}

pub fn now() -> Result<String, UtilError> {
    let fmt = format_description!("[year]-[month]-[day] [hour]:[minute]:[second]Z");
    OffsetDateTime::now_utc()
        .format(fmt)
        .map_err(UtilError::TimeFormat)
}

#[derive(Debug, thiserror::Error)]
pub enum UtilError {
    #[error("failed to create directory {0}")]
    CreateDir(PathBuf, #[source] std::io::Error),

    #[error("ambient-run failed:\n{0}")]
    AmbientRun(String),

    #[error("dput failed to upload {1} to {0}:\n{2}")]
    Dput(String, PathBuf, String),

    #[error("rsync failed to synchronize {0} to {1}:\n{2}")]
    Rsync(PathBuf, String, String),

    #[error("failed to create project YAML file {0}")]
    CreateProject(PathBuf, #[source] std::io::Error),

    #[error("failed to generate project YAML {0}")]
    Yaml(PathBuf, #[source] serde_norway::Error),

    #[error("ambient-run didn't create the artifact tar archive {0}")]
    NoArtifact(PathBuf),

    #[error("failed to open artifact tar archive {0}")]
    OpenArtifacts(PathBuf, #[source] std::io::Error),

    #[error("failed to extract artifact archive {0} to {0}")]
    ExtractArtifacts(PathBuf, PathBuf, #[source] std::io::Error),

    #[error("failed to list contents of upload directory")]
    WalkDir(#[source] walkdir::Error),

    #[error("no *.changes file built for deb project")]
    NoChanges,

    #[error("more than one *.changes file built for deb project")]
    ManyChanges,

    #[error("failed to remove directory {0}")]
    RemoveDir(PathBuf, #[source] std::io::Error),

    #[error("failed to run system tar command: {0}: {1}")]
    Tar(&'static str, PathBuf, #[source] std::io::Error),

    #[error("failed to run system tar command: {0}: {1}")]
    TarFailed(&'static str, PathBuf, i32, String),

    #[error("failed to write file {0}")]
    WriteFile(PathBuf, #[source] std::io::Error),

    #[error("failed to get metadata for file: {0}")]
    GetMetadata(PathBuf, #[source] std::io::Error),

    #[error("failed to make a file executable: {0}")]
    MakeExec(PathBuf, #[source] std::io::Error),

    #[error("failed to copy file {0} to {1}")]
    Copy(PathBuf, PathBuf, #[source] std::io::Error),

    #[error("failed to set permissions for file {0}")]
    SetPermissions(PathBuf, #[source] std::io::Error),

    #[error("failed to read file {0}")]
    Read(PathBuf, #[source] std::io::Error),

    #[error("failed to write file {0}")]
    Write(PathBuf, #[source] std::io::Error),

    #[error("failed to understand file {0} into UTF8")]
    Utf8(PathBuf, #[source] std::string::FromUtf8Error),

    #[error("failed to format time stamp")]
    TimeFormat(#[source] time::error::Format),

    #[error("failed to create HTTP client")]
    ClientBuild(#[source] reqwest::Error),

    #[error("failed to create file {0}")]
    CreateFile(PathBuf, #[source] std::io::Error),

    #[error("failed to build a reqwest client")]
    Client(#[source] reqwest::Error),

    #[error("failed to build a reqwest request")]
    BuildRequest(#[source] reqwest::Error),

    #[error("failed to GET URL {0:?}")]
    Get(String, reqwest::Error),

    #[error("failed to get body of response from {0:?}")]
    GetBody(String, reqwest::Error),

    #[error("failure getting file with HTTP GET: status code {0}")]
    UnwantedStatus(StatusCode),

    #[error("failed to run program {0}")]
    Execute(&'static str, #[source] CommandError),

    #[error("failed to list files in directory {0} when copying files")]
    CopyTreeWalkDir(PathBuf, #[source] walkdir::Error),
}
