#![allow(clippy::result_large_err)]

use std::{
    collections::HashMap,
    ffi::OsString,
    path::{Path, PathBuf},
    process::{Command, Stdio},
};

use clingwrap::runner::{CommandError, CommandRunner};
use log::{debug, info};
use serde::{Deserialize, Serialize};
use tempfile::tempdir;

use crate::{
    // basename::Basename,
    plan::RunnablePlan,
    project::{Project, State},
    qemu,
    util::{
        changes_file, copy_partial_tree, create_file, dput, http_get_to_file, mkdir, rsync_server,
        UtilError,
    },
    vdrive::{VirtualDriveBuilder, VirtualDriveError},
};

/// A context for running an action.
#[derive(Debug, Default, Clone)]
pub struct Context {
    envs: HashMap<OsString, OsString>,
    source_dir: PathBuf,
}

impl Context {
    /// Create a new [`Context`].
    pub fn set_envs_from_plan(&mut self, plan: &RunnablePlan) -> Result<(), ActionError> {
        self.source_dir = plan
            .source_dir()
            .ok_or(ActionError::Missing("source_dir"))?
            .into();

        if let Some(path) = plan.cache_dir() {
            self.set_env("CARGO_TARGET_DIR", path);
        }

        if let Some(path) = plan.deps_dir() {
            self.set_env("CARGO_HOME", path);
        }

        let path = std::env::var("PATH").unwrap_or("/bin".into());
        self.set_env("PATH", &format!("/root/.cargo/bin:{path}"));

        Ok(())
    }

    fn env(&self) -> Vec<(OsString, OsString)> {
        self.envs
            .iter()
            .map(|(k, v)| (k.into(), v.into()))
            .collect()
    }

    /// Set environment variable for future execution of programs.
    pub fn set_env<S: Into<OsString>>(&mut self, name: S, value: S) {
        self.envs.insert(name.into(), value.into());
    }

    fn source_dir(&self) -> PathBuf {
        self.source_dir.to_path_buf()
    }
}

/// A pair of URL and basename, for an item in a `http_get` action.
#[derive(Debug, Clone, Serialize, Deserialize, Eq, PartialEq)]
pub struct Pair {
    url: String,
    filename: PathBuf,
}

impl Pair {
    pub fn filename(&self) -> &Path {
        &self.filename
    }
}

#[derive(Debug, Clone, Serialize, Deserialize, Eq, PartialEq)]
#[serde(tag = "action")]
#[serde(rename_all = "snake_case")]
pub enum RunnableAction {
    Dummy,
    Pwd {
        sourcedir: PathBuf,
    },
    CargoFetch {
        sourcedir: PathBuf,
    },
    HttpGet {
        items: Vec<Pair>,
        dependenciesdir: PathBuf,
    },
    Rsync {
        artifactsdir: PathBuf,
        rsync_target: Option<String>,
    },
    Dput {
        artifactsdir: PathBuf,
        dput_target: Option<String>,
    },
    Mkdir {
        pathname: PathBuf,
    },
    TarCreate {
        archive: PathBuf,
        directory: PathBuf,
    },
    TarExtract {
        archive: PathBuf,
        directory: PathBuf,
    },
    Spawn {
        argv: Vec<String>,
    },
    Shell {
        shell: String,
    },
    CargoFmt,
    CargoClippy,
    CargoDeny,
    CargoDoc,
    CargoBuild,
    CargoTest,
    CargoInstall,
    Deb,
    Custom(Custom),
}

impl RunnableAction {
    pub fn execute(&self, context: &mut Context, plan: &RunnablePlan) -> Result<(), ActionError> {
        debug!("Plan::execute: plan={self:#?}");
        debug!("Plan::execute: context={context:#?}");
        match self {
            Self::Dummy => Self::dummy(),
            Self::Pwd { sourcedir } => Self::pwd(sourcedir),
            Self::CargoFetch { sourcedir } => Self::cargo_fetch(context, sourcedir),
            Self::HttpGet {
                items,
                dependenciesdir,
            } => Self::http_get(items, dependenciesdir),
            Self::Rsync {
                artifactsdir,
                rsync_target,
            } => Self::rsync(artifactsdir, rsync_target.as_ref().map(|s| s.as_str())),
            Self::Dput {
                artifactsdir,
                dput_target,
            } => Self::dput_action(artifactsdir, dput_target.as_ref().map(|s| s.as_str())),
            Self::Mkdir { pathname } => Self::mkdir(pathname),
            Self::TarCreate { archive, directory } => Self::tar_create(archive, directory),
            Self::TarExtract { archive, directory } => Self::tar_extract(archive, directory),
            Self::Spawn { argv } => {
                let argv: Vec<&str> = argv.iter().map(|s| s.as_str()).collect();
                Self::spawn(context, &argv, context.source_dir())
            }
            Self::Shell { shell: snippet } => Self::shell(context, snippet),
            Self::CargoFmt => Self::cargo_fmt(context),
            Self::CargoClippy => Self::cargo_clippy(context),
            Self::CargoDeny => Self::cargo_deny(context),
            Self::CargoDoc => Self::cargo_doc(context),
            Self::CargoBuild => Self::cargo_build(context),
            Self::CargoTest => Self::cargo_test(context),
            Self::CargoInstall => Self::cargo_install(context, plan),
            Self::Deb => Self::deb(context),
            Self::Custom(x) => Self::custom(context, plan, x),
        }
    }

    pub fn from_trusted_action(
        action: &TrustedAction,
        project: &Project,
        state: &State,
        rsync_target: Option<&str>,
        dput_target: Option<&str>,
    ) -> Self {
        match action {
            TrustedAction::Dummy => Self::Dummy,
            TrustedAction::Pwd => Self::Pwd {
                sourcedir: project.source().into(),
            },
            TrustedAction::CargoFetch => Self::CargoFetch {
                sourcedir: project.source().into(),
            },
            TrustedAction::HttpGet { items } => {
                let items: Vec<Pair> = items.to_vec();
                Self::HttpGet {
                    items,
                    dependenciesdir: state.dependenciesdir(),
                }
            }
            TrustedAction::Rsync => Self::Rsync {
                artifactsdir: state.artifactsdir(),
                rsync_target: rsync_target.map(|s| s.into()),
            },
            TrustedAction::Dput => Self::Dput {
                artifactsdir: state.artifactsdir(),
                dput_target: dput_target.map(|s| s.into()),
            },
        }
    }

    pub fn from_unsafe_action(action: &UnsafeAction) -> Self {
        match action {
            UnsafeAction::Mkdir { pathname } => Self::Mkdir {
                pathname: pathname.clone(),
            },
            UnsafeAction::TarCreate { archive, directory } => Self::TarCreate {
                archive: archive.clone(),
                directory: directory.clone(),
            },
            UnsafeAction::TarExtract { archive, directory } => Self::TarExtract {
                archive: archive.clone(),
                directory: directory.clone(),
            },
            UnsafeAction::Spawn { argv } => Self::Spawn { argv: argv.clone() },
            UnsafeAction::Shell { shell } => Self::Shell {
                shell: shell.clone(),
            },
            UnsafeAction::CargoFmt => Self::CargoFmt,
            UnsafeAction::CargoClippy => Self::CargoClippy,
            UnsafeAction::CargoDeny => Self::CargoDeny,
            UnsafeAction::CargoDoc => Self::CargoDoc,
            UnsafeAction::CargoBuild => Self::CargoBuild,
            UnsafeAction::CargoTest => Self::CargoTest,
            UnsafeAction::CargoInstall => Self::CargoInstall,
            UnsafeAction::Deb => Self::Deb,
            UnsafeAction::Custom(x) => Self::Custom(x.clone()),
        }
    }

    fn dummy() -> Result<(), ActionError> {
        println!("dummy action");
        Ok(())
    }

    fn pwd(sourcedir: &Path) -> Result<(), ActionError> {
        info!("cwd: {}", sourcedir.display());
        Ok(())
    }

    #[allow(clippy::unwrap_used)]
    fn custom(
        context: &mut Context,
        plan: &RunnablePlan,
        custom: &Custom,
    ) -> Result<(), ActionError> {
        let source = PathBuf::from(
            plan.source_dir()
                .ok_or(ActionError::Missing("source_dir"))?,
        );
        let exe = Path::new(".ambient").join(&custom.name);
        let json = serde_json::to_string(&custom.args).map_err(ActionError::ArgsToJson)?;

        eprintln!("custom: source={}", source.display());
        eprintln!("custom: exe={exe:?} exists={}", exe.exists());
        let mut cmd = Command::new(exe);
        cmd.current_dir(&source);
        cmd.envs(context.env());
        for (key, value) in custom.args.iter() {
            let key = format!("AMBIENT_CI_{key}");
            let value = serde_json::to_string(value).map_err(ActionError::ArgsToJson)?;
            cmd.env(key, value);
        }
        let mut runner = CommandRunner::new(cmd);
        runner.feed_stdin(json.as_bytes());
        let output = runner
            .execute()
            .map_err(|err| ActionError::Custom(custom.name.to_string(), err))?;
        println!(
            "custom action {:?} exit code {:?}",
            custom.name,
            output.status.code()
        );
        Ok(())
    }

    fn rsync(artifactsdir: &Path, target: Option<&str>) -> Result<(), ActionError> {
        let target = target.ok_or(ActionError::RsyncTargetMissing)?;
        rsync_server(artifactsdir, target)?;
        Ok(())
    }

    fn dput_action(artifactsdir: &Path, dput_target: Option<&str>) -> Result<(), ActionError> {
        let target = dput_target.ok_or(ActionError::RsyncTargetMissing)?;
        let changes = changes_file(artifactsdir)?;
        dput(target, &changes)?;
        Ok(())
    }

    fn rust_toolchain_versions(context: &mut Context) -> Result<(), ActionError> {
        Self::spawn(context, &["cargo", "--version"], context.source_dir())?;
        Self::spawn(
            context,
            &["cargo", "clippy", "--version"],
            context.source_dir(),
        )?;
        Self::spawn(context, &["rustc", "--version"], context.source_dir())?;
        Ok(())
    }

    fn cargo_fetch(context: &mut Context, source_dir: &Path) -> Result<(), ActionError> {
        Self::rust_toolchain_versions(context)?;
        let tmp = tempdir().map_err(ActionError::TempDir)?;
        let dest = tmp.path();
        copy_partial_tree(source_dir, dest, |path| {
            path.ends_with("Cargo.toml") || path.ends_with("Cargo.lock")
        })
        .map_err(ActionError::CargoFetchCopy)?;
        mkdir(&dest.join("src"))?;
        create_file(&dest.join("src/lib.rs"))?;

        let lockfile = dest.join("Cargo.lock");
        let deny1 = dest.join("deny.toml");
        let deny2 = dest.join(".cargo/deny.toml");
        let deny = deny1.exists() || deny2.exists();
        if lockfile.exists() {
            Self::spawn(context, &["cargo", "fetch", "--locked"], dest.to_path_buf())?;
            if deny {
                Self::spawn(
                    context,
                    &["cargo", "deny", "--locked", "fetch"],
                    dest.to_path_buf(),
                )?;
            }
        } else {
            Self::spawn(context, &["cargo", "fetch"], dest.to_path_buf())?;
            if deny {
                Self::spawn(
                    context,
                    &["cargo", "deny", "--locked", "fetch"],
                    dest.to_path_buf(),
                )?;
            }
        }

        Ok(())
    }

    fn http_get(items: &[Pair], dependencies_dir: &Path) -> Result<(), ActionError> {
        info!("http_get: deps={}", dependencies_dir.display());
        for pair in items.iter() {
            let filename = dependencies_dir.join(&pair.filename);
            http_get_to_file(&pair.url, &filename)
                .map_err(|err| ActionError::Get(pair.url.clone(), err))?;
        }
        Ok(())
    }

    fn mkdir(pathname: &Path) -> Result<(), ActionError> {
        if !pathname.exists() {
            std::fs::create_dir(pathname).map_err(|e| ActionError::Mkdir(pathname.into(), e))?;
        }
        Ok(())
    }

    fn tar_create(archive: &Path, dirname: &Path) -> Result<(), ActionError> {
        VirtualDriveBuilder::default()
            .filename(archive)
            .root_directory(dirname)
            .create()
            .map_err(|e| ActionError::TarCreate(archive.into(), dirname.into(), e))?;
        Ok(())
    }

    fn tar_extract(archive: &Path, dirname: &Path) -> Result<(), ActionError> {
        let tar = VirtualDriveBuilder::default()
            .filename(archive)
            .root_directory(dirname)
            .open()
            .map_err(|e| ActionError::TarOpen(archive.into(), e))?;
        tar.extract_to(dirname)
            .map_err(|e| ActionError::TarExtract(archive.into(), dirname.into(), e))?;

        Ok(())
    }

    fn spawn(context: &mut Context, argv: &[&str], cwd: PathBuf) -> Result<(), ActionError> {
        println!("SPAWN: argv={argv:?}");
        println!("       cwd={} (exists? {})", cwd.display(), cwd.exists());

        let argv0 = if let Some(argv0) = argv.first() {
            argv0
        } else {
            return Err(ActionError::SpawnNoArgv0);
        };

        let mut cmd = Command::new(argv0);
        cmd.args(&argv[1..])
            .envs(context.env())
            .current_dir(&cwd)
            .stdin(Stdio::null())
            .current_dir(cwd);

        let runner = CommandRunner::new(cmd);
        match runner.execute() {
            Ok(_) => Ok(()),
            Err(err) => Err(ActionError::Execute(argv0.to_string(), err)),
        }
    }

    fn shell(context: &mut Context, snippet: &str) -> Result<(), ActionError> {
        let snippet = format!("set -xeuo pipefail\n{snippet}\n");
        Self::spawn(
            context,
            &["/bin/bash", "-c", &snippet],
            context.source_dir(),
        )
    }

    fn cargo_fmt(context: &mut Context) -> Result<(), ActionError> {
        Self::rust_toolchain_versions(context)?;
        Self::spawn(context, &["cargo", "fmt", "--check"], context.source_dir())
    }

    fn cargo_clippy(context: &mut Context) -> Result<(), ActionError> {
        Self::rust_toolchain_versions(context)?;
        Self::spawn(
            context,
            &[
                "cargo",
                "clippy",
                "--offline",
                "--locked",
                "--workspace",
                "--all-targets",
                "--no-deps",
                "--",
                "--deny",
                "warnings",
            ],
            context.source_dir(),
        )
    }

    fn cargo_deny(context: &mut Context) -> Result<(), ActionError> {
        Self::rust_toolchain_versions(context)?;
        Self::spawn(
            context,
            &[
                "cargo",
                "deny",
                "--offline",
                "--locked",
                "--workspace",
                "check",
            ],
            context.source_dir(),
        )
    }

    fn cargo_doc(context: &mut Context) -> Result<(), ActionError> {
        Self::rust_toolchain_versions(context)?;
        Self::spawn(
            context,
            &[
                "env",
                "RUSTDOCFLAGS=-D warnings",
                "cargo",
                "doc",
                "--workspace",
            ],
            context.source_dir(),
        )
    }

    fn cargo_build(context: &mut Context) -> Result<(), ActionError> {
        Self::rust_toolchain_versions(context)?;
        Self::spawn(
            context,
            &[
                "cargo",
                "build",
                "--offline",
                "--locked",
                "--workspace",
                "--all-targets",
            ],
            context.source_dir(),
        )
    }

    fn cargo_test(context: &mut Context) -> Result<(), ActionError> {
        Self::rust_toolchain_versions(context)?;
        Self::spawn(
            context,
            &["cargo", "test", "--offline", "--locked", "--workspace"],
            context.source_dir(),
        )
    }

    fn cargo_install(context: &mut Context, plan: &RunnablePlan) -> Result<(), ActionError> {
        Self::rust_toolchain_versions(context)?;
        Self::spawn(
            context,
            &[
                "cargo",
                "install",
                "--offline",
                "--locked",
                "--bins",
                "--path=.",
                "--root",
                plan.artifacts_dir()
                    .ok_or(ActionError::Missing("artifacts_dir"))?,
            ],
            context.source_dir(),
        )
    }

    fn deb(context: &mut Context) -> Result<(), ActionError> {
        let shell = format!(
            r#"#!/bin/bash
set -xeuo pipefail

echo "PATH at start: $PATH"
export PATH="/root/.cargo/bin:$PATH"
export CARGO_HOME=/workspace/deps
export DEBEMAIL=liw@liw.fi
export DEBFULLNAME="Lars Wirzenius"
/bin/env

command -v cargo
command -v rustc

cargo --version
rustc --version

# Get name and version of source package.
name="$(dpkg-parsechangelog -SSource)"
version="$(dpkg-parsechangelog -SVersion)"

# Get upstream version: everything before the last dash.
uv="$(echo "$version" | sed 's/-[^-]*$//')"

# Files that will be created.
arch="$(dpkg --print-architecture)"
orig="../${{name}}_${{uv}}.orig.tar.xz"
deb="../${{name}}_${{version}}_${{arch}}.deb"
changes="../${{name}}_${{version}}_${{arch}}.changes"

# Create "upstream tarball".
git archive HEAD | xz >"$orig"

# Build package.
dpkg-buildpackage -us -uc

# Dump some information to make it easier to visually verify
# everything looks OK. Also, test the package with the lintian tool.

ls -l ..
for x in ../*.deb; do dpkg -c "$x"; done
# FIXME: disabled while this prevents radicle-native-ci deb from being built.
# lintian -i --allow-root --fail-on warning ../*.changes

# Move files to artifacts directory.
mv ../*_* {}
        "#,
            qemu::ARTIFACTS_DIR
        );

        Self::spawn(context, &["/bin/bash", "-c", &shell], context.source_dir())
    }
}

#[derive(Debug, Clone, Serialize, Deserialize)]
#[serde(tag = "action")]
#[serde(rename_all = "snake_case")]
pub enum TrustedAction {
    Dummy,
    Pwd,
    CargoFetch,
    HttpGet { items: Vec<Pair> },
    Rsync,
    Dput,
}

impl TrustedAction {
    pub fn names() -> &'static [&'static str] {
        &["dummy", "pwd", "cargo_fetch", "rsync", "dput"]
    }
}

#[derive(Debug, Clone, Serialize, Deserialize)]
#[serde(tag = "action")]
#[serde(rename_all = "snake_case")]
pub enum UnsafeAction {
    Mkdir {
        pathname: PathBuf,
    },
    TarCreate {
        archive: PathBuf,
        directory: PathBuf,
    },
    TarExtract {
        archive: PathBuf,
        directory: PathBuf,
    },
    Spawn {
        argv: Vec<String>,
    },
    Shell {
        shell: String,
    },
    CargoFmt,
    CargoClippy,
    CargoDeny,
    CargoDoc,
    CargoBuild,
    CargoTest,
    CargoInstall,
    Deb,
    Custom(Custom),
}

impl UnsafeAction {
    pub fn names() -> &'static [&'static str] {
        &[
            "mkdir",
            "tar_create",
            "tar_extract",
            "spawn",
            "shell",
            "cargo_fmt",
            "cargo_clippy",
            "cargo_build",
            "cargo_test",
            "cargo_install",
            "deb",
            "custom",
        ]
    }

    pub fn mkdir<P: AsRef<Path>>(pathname: P) -> Self {
        Self::Mkdir {
            pathname: pathname.as_ref().into(),
        }
    }

    pub fn tar_create<P: AsRef<Path>>(archive: P, directory: P) -> Self {
        Self::TarCreate {
            archive: archive.as_ref().into(),
            directory: directory.as_ref().into(),
        }
    }

    pub fn tar_extract<P: AsRef<Path>>(archive: P, directory: P) -> Self {
        Self::TarExtract {
            archive: archive.as_ref().into(),
            directory: directory.as_ref().into(),
        }
    }

    pub fn shell(shell: &str) -> Self {
        Self::Shell {
            shell: shell.into(),
        }
    }

    pub fn spawn(argv: &[&str]) -> Self {
        Self::Spawn {
            argv: argv.iter().map(|s| s.to_string()).collect(),
        }
    }
}

/// A custom action.
#[derive(Debug, Clone, Eq, PartialEq, Serialize, Deserialize)]
pub struct Custom {
    name: String,

    #[serde(default)]
    args: HashMap<String, serde_norway::Value>,
}

#[derive(Debug, thiserror::Error)]
pub enum ActionError {
    #[error("failed to create a temporary directory")]
    TempDir(#[source] std::io::Error),

    #[error("failed to copy source tree to temporary location")]
    CargoFetchCopy(#[source] UtilError),

    #[error("failed to create directory {0}")]
    Mkdir(PathBuf, #[source] std::io::Error),

    #[error("failed to execute {0}")]
    Execute(String, #[source] CommandError),

    #[error("failed to create tar archive {0} from {1}")]
    TarCreate(PathBuf, PathBuf, #[source] VirtualDriveError),

    #[error("failed to open tar archive {0}")]
    TarOpen(PathBuf, #[source] VirtualDriveError),

    #[error("failed to extract tar archive {0} into {1}")]
    TarExtract(PathBuf, PathBuf, #[source] VirtualDriveError),

    #[error("failed to invoke command: empty argv")]
    SpawnNoArgv0,

    #[error("failed to invoke command: {0} in {1}")]
    SpawnInvoke(String, PathBuf, #[source] std::io::Error),

    #[error("command failed was killed by signal")]
    SpawnKilledBySignal(String),

    #[error("command failed: {0}")]
    SpawnFailed(String, i32),

    #[error("failed to remove directory {0}")]
    RemoveDir(PathBuf, #[source] std::io::Error),

    #[error("failed to remove file {0}")]
    RemoveFile(PathBuf, #[source] std::io::Error),

    #[error("failed to create symlink {0} -> {1}")]
    Symlink(PathBuf, PathBuf, #[source] std::io::Error),

    #[error(transparent)]
    Util(#[from] UtilError),

    #[error("for the rsync action you must specify an rsync target (config or command line)")]
    RsyncTargetMissing,

    #[error("for the dput action you must specify a dput target (config or command line)")]
    DputTargetMissing,

    #[error("failed to convert custom action into JSON")]
    ArgsToJson(#[source] serde_json::Error),

    #[error("failed to run custom action {0:?}")]
    Custom(String, #[source] CommandError),

    #[error("failed to retrieve URL {0:?} into file {1}")]
    Get(String, UtilError),

    #[error("runnable plan does not have field {0} set")]
    Missing(&'static str),
}

#[cfg(test)]
mod test {
    use super::*;
    use tempfile::tempdir;

    fn plan() -> RunnablePlan {
        let mut plan = RunnablePlan::default();
        plan.set_cache_dir("/tmp");
        plan.set_deps_dir("/tmp");
        plan.set_source_dir("/tmp");
        plan
    }

    #[test]
    fn mkdir_action() -> Result<(), Box<dyn std::error::Error>> {
        let tmp = tempdir()?;
        let path = tmp.path().join("testdir");
        let action = RunnableAction::from_unsafe_action(&UnsafeAction::mkdir(&path));
        let plan = plan();
        let mut context = Context::default();
        context.set_envs_from_plan(&plan)?;
        assert!(!path.exists());
        assert!(action.execute(&mut context, &plan).is_ok());
        assert!(path.exists());
        Ok(())
    }

    #[test]
    fn tar_create_action() -> Result<(), Box<dyn std::error::Error>> {
        let tmp = tempdir()?;
        let src = tmp.path().join("src");
        let tar = tmp.path().join("src.tar");

        std::fs::create_dir(&src)?;
        let action = RunnableAction::from_unsafe_action(&UnsafeAction::tar_create(&tar, &src));
        let plan = plan();
        let mut context = Context::default();
        context.set_envs_from_plan(&plan)?;

        assert!(!tar.exists());
        assert!(action.execute(&mut context, &plan).is_ok());
        assert!(tar.exists());
        Ok(())
    }

    #[test]
    fn tar_extract_action() -> Result<(), Box<dyn std::error::Error>> {
        let tmp = tempdir()?;
        let src = tmp.path().join("src");
        let tar = tmp.path().join("src.tar");
        let extracted = tmp.path().join("extracted");

        std::fs::create_dir(&src)?;
        std::fs::File::create(src.join("file.dat"))?;
        let plan = plan();
        let mut context = Context::default();
        context.set_envs_from_plan(&plan)?;

        RunnableAction::from_unsafe_action(&UnsafeAction::tar_create(&tar, &src))
            .execute(&mut context, &plan)?;

        let action =
            RunnableAction::from_unsafe_action(&UnsafeAction::tar_extract(&tar, &extracted));
        assert!(action.execute(&mut context, &plan).is_ok());
        assert!(extracted.join("file.dat").exists());
        Ok(())
    }

    #[test]
    fn spawn_action() -> Result<(), Box<dyn std::error::Error>> {
        let plan = plan();
        let mut context = Context::default();
        context.set_envs_from_plan(&plan)?;
        let src = context.source_dir();
        assert!(RunnableAction::spawn(&mut context, &["true"], src).is_ok());
        Ok(())
    }
}
