#![allow(clippy::result_large_err)]

use std::{
    collections::HashMap,
    ffi::OsString,
    path::{Path, PathBuf},
};

use clingwrap::runner::CommandError;
use log::debug;
use serde::{Deserialize, Serialize};

use crate::{action_impl::*, plan::RunnablePlan, util::UtilError, vdrive::VirtualDriveError};

/// A context for running an action.
#[derive(Debug, Default, Clone)]
pub struct Context {
    envs: HashMap<OsString, OsString>,
    source_dir: PathBuf,
    deps_dir: PathBuf,
    artifacts_dir: PathBuf,
}

impl Context {
    /// Create a new [`Context`].
    pub fn set_envs_from_plan(&mut self, plan: &RunnablePlan) -> Result<(), ActionError> {
        self.source_dir = plan
            .source_dir()
            .ok_or(ActionError::Missing("source_dir"))?
            .into();

        self.deps_dir = plan
            .deps_dir()
            .ok_or(ActionError::Missing("deps_dir"))?
            .into();

        self.artifacts_dir = plan
            .artifacts_dir()
            .ok_or(ActionError::Missing("artifacts_dir"))?
            .into();

        if let Some(path) = plan.cache_dir() {
            self.set_env("CARGO_TARGET_DIR", path);
        }

        if let Some(path) = plan.deps_dir() {
            self.set_env("CARGO_HOME", path);
        }

        let path = std::env::var("PATH").unwrap_or("/bin".into());
        self.set_env("PATH", &format!("/root/.cargo/bin:{path}"));

        Ok(())
    }

    /// Return all environment variables.
    pub fn env(&self) -> Vec<(OsString, OsString)> {
        self.envs
            .iter()
            .map(|(k, v)| (k.into(), v.into()))
            .collect()
    }

    /// Set environment variable for future execution of programs.
    pub fn set_env<S: Into<OsString>>(&mut self, name: S, value: S) {
        self.envs.insert(name.into(), value.into());
    }

    /// Return source directory for this context.
    pub fn source_dir(&self) -> &Path {
        &self.source_dir
    }

    /// Return dependencies directory for this context.
    pub fn deps_dir(&self) -> &Path {
        &self.deps_dir
    }

    /// Return artifacts directory for this context.
    pub fn artifacts_dir(&self) -> &Path {
        &self.artifacts_dir
    }
}

/// A pair of URL and basename, for an item in a `http_get` action.
#[derive(Debug, Clone, Serialize, Deserialize, Eq, PartialEq)]
pub struct Pair {
    url: String,
    filename: PathBuf,
}

impl Pair {
    pub fn url(&self) -> &str {
        &self.url
    }

    pub fn filename(&self) -> &Path {
        &self.filename
    }
}

#[derive(Debug, Clone, Serialize, Deserialize, Eq, PartialEq)]
#[serde(tag = "action")]
#[serde(rename_all = "snake_case")]
pub enum RunnableAction {
    Dummy(Dummy),
    Pwd(Pwd),
    CargoFetch(CargoFetch),
    HttpGet(HttpGet),
    Rsync(Rsync),
    Dput(Dput),
    Mkdir(Mkdir),
    TarCreate(TarCreate),
    TarExtract(TarExtract),
    Spawn(Spawn),
    Shell(Shell),
    CargoFmt(CargoFmt),
    CargoClippy(CargoClippy),
    CargoDeny(CargoDeny),
    CargoDoc(CargoDoc),
    CargoBuild(CargoBuild),
    CargoTest(CargoTest),
    CargoInstall(CargoInstall),
    Deb(Deb),
    Custom(Custom),
}

impl RunnableAction {
    pub fn execute(&self, context: &mut Context) -> Result<(), ActionError> {
        debug!("RunnableAction::execute: self={self:#?}");
        debug!("RunnableAction::execute: context={context:#?}");
        match self {
            Self::Dummy(x) => x.execute(context),
            Self::Pwd(x) => x.execute(context),
            Self::CargoFetch(x) => x.execute(context),
            Self::HttpGet(x) => x.execute(context),
            Self::Rsync(x) => x.execute(context),
            Self::Dput(x) => x.execute(context),
            Self::Mkdir(x) => x.execute(context),
            Self::TarCreate(x) => x.execute(context),
            Self::TarExtract(x) => x.execute(context),
            Self::Spawn(x) => x.execute(context),
            Self::Shell(x) => x.execute(context),
            Self::CargoFmt(x) => x.execute(context),
            Self::CargoClippy(x) => x.execute(context),
            Self::CargoDeny(x) => x.execute(context),
            Self::CargoDoc(x) => x.execute(context),
            Self::CargoBuild(x) => x.execute(context),
            Self::CargoTest(x) => x.execute(context),
            Self::CargoInstall(x) => x.execute(context),
            Self::Deb(x) => x.execute(context),
            Self::Custom(x) => x.execute(context),
        }
    }

    pub fn from_trusted_action(
        action: &TrustedAction,
        rsync_target: Option<&str>,
        dput_target: Option<&str>,
    ) -> Self {
        match action {
            TrustedAction::Dummy => Self::Dummy(Dummy),
            TrustedAction::Pwd => Self::Pwd(Pwd),
            TrustedAction::CargoFetch => Self::CargoFetch(CargoFetch),
            TrustedAction::HttpGet { items } => {
                let items: Vec<Pair> = items.to_vec();
                Self::HttpGet(HttpGet::new(items))
            }
            TrustedAction::Rsync => Self::Rsync(Rsync::new(rsync_target.map(|s| s.to_string()))),
            TrustedAction::Dput => Self::Dput(Dput::new(dput_target.map(|s| s.into()))),
        }
    }

    pub fn from_unsafe_action(action: &UnsafeAction) -> Self {
        match action {
            UnsafeAction::Mkdir { pathname } => Self::Mkdir(Mkdir::new(pathname.to_path_buf())),
            UnsafeAction::TarCreate { archive, directory } => {
                Self::TarCreate(TarCreate::new(archive.clone(), directory.clone()))
            }
            UnsafeAction::TarExtract { archive, directory } => {
                Self::TarExtract(TarExtract::new(archive.clone(), directory.clone()))
            }
            UnsafeAction::Spawn { argv } => Self::Spawn(Spawn::new(argv.clone())),
            UnsafeAction::Shell { shell } => Self::Shell(Shell::new(shell.clone())),
            UnsafeAction::CargoFmt => Self::CargoFmt(CargoFmt),
            UnsafeAction::CargoClippy => Self::CargoClippy(CargoClippy),
            UnsafeAction::CargoDeny => Self::CargoDeny(CargoDeny),
            UnsafeAction::CargoDoc => Self::CargoDoc(CargoDoc),
            UnsafeAction::CargoBuild => Self::CargoBuild(CargoBuild),
            UnsafeAction::CargoTest => Self::CargoTest(CargoTest),
            UnsafeAction::CargoInstall => Self::CargoInstall(CargoInstall),
            UnsafeAction::Deb => Self::Deb(Deb),
            UnsafeAction::Custom(x) => Self::Custom(x.clone()),
        }
    }
}

#[derive(Debug, Clone, Serialize, Deserialize)]
#[serde(tag = "action")]
#[serde(rename_all = "snake_case")]
pub enum TrustedAction {
    Dummy,
    Pwd,
    CargoFetch,
    HttpGet { items: Vec<Pair> },
    Rsync,
    Dput,
}

impl TrustedAction {
    pub fn names() -> &'static [&'static str] {
        &["dummy", "pwd", "cargo_fetch", "rsync", "dput"]
    }
}

#[derive(Debug, Clone, Serialize, Deserialize)]
#[serde(tag = "action")]
#[serde(rename_all = "snake_case")]
pub enum UnsafeAction {
    Mkdir {
        pathname: PathBuf,
    },
    TarCreate {
        archive: PathBuf,
        directory: PathBuf,
    },
    TarExtract {
        archive: PathBuf,
        directory: PathBuf,
    },
    Spawn {
        argv: Vec<String>,
    },
    Shell {
        shell: String,
    },
    CargoFmt,
    CargoClippy,
    CargoDeny,
    CargoDoc,
    CargoBuild,
    CargoTest,
    CargoInstall,
    Deb,
    Custom(Custom),
}

impl UnsafeAction {
    pub fn names() -> &'static [&'static str] {
        &[
            "mkdir",
            "tar_create",
            "tar_extract",
            "spawn",
            "shell",
            "cargo_fmt",
            "cargo_clippy",
            "cargo_build",
            "cargo_test",
            "cargo_install",
            "deb",
            "custom",
        ]
    }

    pub fn mkdir<P: AsRef<Path>>(pathname: P) -> Self {
        Self::Mkdir {
            pathname: pathname.as_ref().into(),
        }
    }

    pub fn tar_create<P: AsRef<Path>>(archive: P, directory: P) -> Self {
        Self::TarCreate {
            archive: archive.as_ref().into(),
            directory: directory.as_ref().into(),
        }
    }

    pub fn tar_extract<P: AsRef<Path>>(archive: P, directory: P) -> Self {
        Self::TarExtract {
            archive: archive.as_ref().into(),
            directory: directory.as_ref().into(),
        }
    }

    pub fn shell(shell: &str) -> Self {
        Self::Shell {
            shell: shell.into(),
        }
    }

    pub fn spawn(argv: &[&str]) -> Self {
        Self::Spawn {
            argv: argv.iter().map(|s| s.to_string()).collect(),
        }
    }
}

#[derive(Debug, thiserror::Error)]
pub enum ActionError {
    #[error("failed to create a temporary directory")]
    TempDir(#[source] std::io::Error),

    #[error("failed to make path absolute: {0}")]
    Canonicalize(PathBuf, #[source] std::io::Error),

    #[error("failed to copy source tree to temporary location")]
    CargoFetchCopy(#[source] UtilError),

    #[error("failed to create directory {0}")]
    Mkdir(PathBuf, #[source] std::io::Error),

    #[error("failed to execute {0}")]
    Execute(String, #[source] CommandError),

    #[error("failed to create tar archive {0} from {1}")]
    TarCreate(PathBuf, PathBuf, #[source] VirtualDriveError),

    #[error("failed to open tar archive {0}")]
    TarOpen(PathBuf, #[source] VirtualDriveError),

    #[error("failed to extract tar archive {0} into {1}")]
    TarExtract(PathBuf, PathBuf, #[source] VirtualDriveError),

    #[error("failed to invoke command: empty argv")]
    SpawnNoArgv0,

    #[error("failed to invoke command: {0} in {1}")]
    SpawnInvoke(String, PathBuf, #[source] std::io::Error),

    #[error("command failed was killed by signal")]
    SpawnKilledBySignal(String),

    #[error("command failed: {0}")]
    SpawnFailed(String, i32),

    #[error("failed to remove directory {0}")]
    RemoveDir(PathBuf, #[source] std::io::Error),

    #[error("failed to remove file {0}")]
    RemoveFile(PathBuf, #[source] std::io::Error),

    #[error("failed to create symlink {0} -> {1}")]
    Symlink(PathBuf, PathBuf, #[source] std::io::Error),

    #[error(transparent)]
    Util(#[from] UtilError),

    #[error("for the rsync action you must specify an rsync target (config or command line)")]
    RsyncTargetMissing,

    #[error("for the dput action you must specify a dput target (config or command line)")]
    DputTargetMissing,

    #[error("failed to convert custom action into JSON")]
    ArgsToJson(#[source] serde_json::Error),

    #[error("failed to run custom action {0:?}")]
    Custom(String, #[source] CommandError),

    #[error("failed to retrieve URL {0:?} into file {1}")]
    Get(String, UtilError),

    #[error("runnable plan does not have field {0} set")]
    Missing(&'static str),
}

#[cfg(test)]
mod test {
    use super::*;
    use tempfile::tempdir;

    fn plan() -> RunnablePlan {
        let mut plan = RunnablePlan::default();
        plan.set_cache_dir("/tmp");
        plan.set_deps_dir("/tmp");
        plan.set_source_dir("/tmp");
        plan.set_artifacts_dir("/tmp");
        plan
    }

    #[test]
    fn mkdir_action() -> Result<(), Box<dyn std::error::Error>> {
        let tmp = tempdir()?;
        let path = tmp.path().join("testdir");
        let action = RunnableAction::from_unsafe_action(&UnsafeAction::mkdir(&path));
        let plan = plan();
        let mut context = Context::default();
        context.set_envs_from_plan(&plan)?;
        assert!(!path.exists());
        assert!(action.execute(&mut context).is_ok());
        assert!(path.exists());
        Ok(())
    }

    #[test]
    fn tar_create_action() -> Result<(), Box<dyn std::error::Error>> {
        let tmp = tempdir()?;
        let src = tmp.path().join("src");
        let tar = tmp.path().join("src.tar");

        std::fs::create_dir(&src)?;
        let action = RunnableAction::from_unsafe_action(&UnsafeAction::tar_create(&tar, &src));
        let plan = plan();
        let mut context = Context::default();
        context.set_envs_from_plan(&plan)?;

        assert!(!tar.exists());
        assert!(action.execute(&mut context).is_ok());
        assert!(tar.exists());
        Ok(())
    }

    #[test]
    fn tar_extract_action() -> Result<(), Box<dyn std::error::Error>> {
        let tmp = tempdir()?;
        let src = tmp.path().join("src");
        let tar = tmp.path().join("src.tar");
        let extracted = tmp.path().join("extracted");

        std::fs::create_dir(&src)?;
        std::fs::File::create(src.join("file.dat"))?;
        let plan = plan();
        let mut context = Context::default();
        context.set_envs_from_plan(&plan)?;

        RunnableAction::from_unsafe_action(&UnsafeAction::tar_create(&tar, &src))
            .execute(&mut context)?;

        let action =
            RunnableAction::from_unsafe_action(&UnsafeAction::tar_extract(&tar, &extracted));
        assert!(action.execute(&mut context).is_ok());
        assert!(extracted.join("file.dat").exists());
        Ok(())
    }

    #[test]
    fn spawn_action() -> Result<(), Box<dyn std::error::Error>> {
        let plan = plan();
        let mut context = Context::default();
        context.set_envs_from_plan(&plan)?;
        let action = Spawn::new(vec!["true".into()]);
        assert!(action.execute(&context).is_ok());
        Ok(())
    }
}
