//! A type representing a filename without directories.

use std::{ffi::OsStr, path::PathBuf};

use serde::{Deserialize, Serialize};

use std::{fmt, ops::Deref, path::Path};

/// A data type for a filename that has no directory component.
#[derive(Debug, Clone, Eq, PartialEq, Serialize)]
pub struct Basename {
    path: PathBuf,
}

impl Basename {
    /// Create a new `Basename`.
    pub fn new(path: PathBuf) -> Self {
        Self { path }
    }

    /// The contained filename.
    pub fn path(&self) -> &Path {
        &self.path
    }
}

impl From<PathBuf> for Basename {
    fn from(s: PathBuf) -> Self {
        Self::new(s)
    }
}

impl From<&Path> for Basename {
    fn from(s: &Path) -> Self {
        Self::new(PathBuf::from(s))
    }
}

impl From<&OsStr> for Basename {
    fn from(s: &OsStr) -> Self {
        Self::new(PathBuf::from(s))
    }
}

impl From<&str> for Basename {
    fn from(s: &str) -> Self {
        Self::new(PathBuf::from(s))
    }
}

impl From<&String> for Basename {
    fn from(s: &String) -> Self {
        Self::new(PathBuf::from(s))
    }
}

impl From<String> for Basename {
    fn from(s: String) -> Self {
        Self::new(PathBuf::from(s))
    }
}

impl AsRef<Path> for Basename {
    fn as_ref(&self) -> &Path {
        self.path()
    }
}

impl Deref for Basename {
    type Target = Path;
    fn deref(&self) -> &Self::Target {
        self.path()
    }
}

impl<'de> Deserialize<'de> for Basename {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error>
    where
        D: serde::de::Deserializer<'de>,
    {
        struct Visitor;
        impl serde::de::Visitor<'_> for Visitor {
            type Value = Basename;
            fn expecting(&self, formatter: &mut fmt::Formatter) -> fmt::Result {
                formatter.write_str("a path that is only the base name")
            }
            fn visit_str<E>(self, value: &str) -> Result<Self::Value, E>
            where
                E: serde::de::Error,
            {
                let path = PathBuf::from(value);
                if path.parent() == Some(Path::new("")) {
                    Ok(Basename { path })
                } else {
                    Err(E::custom(BasenameError::Dirs))
                }
            }
        }

        deserializer.deserialize_str(Visitor)
    }
}

/// Possible errors from tilde expansion.
#[derive(Debug, thiserror::Error)]
pub enum BasenameError {
    #[error("filename contains directory components")]
    Dirs,
}

#[cfg(test)]
mod test {
    use super::*;

    #[derive(Deserialize)]
    struct Foo {
        basename: Basename,
    }

    #[test]
    fn basename() -> Result<(), Box<dyn std::error::Error>> {
        let thing = r#"{"basename": "bar"}"#;

        let x: Foo = serde_json::from_str(thing)?;
        assert_eq!(x.basename.path(), PathBuf::from("bar"));

        Ok(())
    }
}
