use std::{
    path::{Path, PathBuf},
    process::{Command, Stdio},
};

use log::{debug, info};
use serde::{Deserialize, Serialize};

use crate::{
    config::RunCommand,
    project::{Project, State},
    qemu,
    util::{changes_file, dput, rsync_server, UtilError},
    vdrive::{VirtualDriveBuilder, VirtualDriveError},
};

const RUST_ENVS: &[(&str, &str)] = &[
    ("CARGO_TARGET_DIR", qemu::CACHE_DIR),
    ("CARGO_HOME", qemu::DEPS_DIR),
];

#[derive(Debug, Clone, Serialize, Deserialize)]
#[serde(tag = "action")]
#[serde(rename_all = "snake_case")]
pub enum PreAction {
    Dummy,
    Pwd,
    CargoFetch,
}

impl PreAction {
    pub fn names() -> &'static [&'static str] {
        &["dummy", "pwd", "cargo_fetch"]
    }

    pub fn execute(&self, project: &Project, state: &State) -> Result<(), ActionError> {
        debug!("Plan::execute: {:#?}", self);
        match self {
            Self::Dummy => dummy(),
            Self::Pwd => pwd(project),
            Self::CargoFetch => cargo_fetch(project, state),
        }
    }
}

#[derive(Debug, Clone, Serialize, Deserialize)]
#[serde(tag = "action")]
#[serde(rename_all = "snake_case")]
pub enum PostAction {
    Dummy,
    Pwd,
    Rsync,
    Dput,
}

impl PostAction {
    pub fn names() -> &'static [&'static str] {
        &["dummy", "pwd"]
    }

    pub fn execute(
        &self,
        project: &Project,
        state: &State,
        run: &RunCommand,
    ) -> Result<(), ActionError> {
        debug!("Plan::execute: {:#?}", self);
        match self {
            Self::Dummy => dummy(),
            Self::Pwd => pwd(project),
            Self::Rsync => rsync(state, run),
            Self::Dput => dput_action(state, run),
        }
    }
}

fn dummy() -> Result<(), ActionError> {
    println!("dummy action");
    Ok(())
}

fn pwd(project: &Project) -> Result<(), ActionError> {
    info!("cwd: {}", project.source().display());
    Ok(())
}

fn rsync(state: &State, run: &RunCommand) -> Result<(), ActionError> {
    if let Some(target) = &run.target() {
        rsync_server(&state.artifactsdir(), target)?;
    } else {
        return Err(ActionError::RsyncTargetMissing);
    }
    Ok(())
}

fn dput_action(state: &State, run: &RunCommand) -> Result<(), ActionError> {
    if let Some(target) = &run.dput_target() {
        let changes = changes_file(&state.artifactsdir())?;
        dput(target, &changes)?;
    } else {
        return Err(ActionError::DputTargetMissing);
    }
    Ok(())
}

fn cargo_fetch(project: &Project, state: &State) -> Result<(), ActionError> {
    let cargo_home = state.dependenciesdir(); // FIXME: hardcoded pathname
    if !cargo_home.exists() {
        std::fs::create_dir(&cargo_home).map_err(|e| ActionError::Mkdir(cargo_home.clone(), e))?;
    }
    spawn_str_in(
        &[
            "env",
            &format!("CARGO_HOME={}", cargo_home.display()),
            "cargo",
            "fetch",
            "--locked",
        ],
        project.source(),
    )
}

#[derive(Debug, Clone, Serialize, Deserialize)]
#[serde(tag = "action")]
#[serde(rename_all = "snake_case")]
pub enum Action {
    Mkdir {
        pathname: PathBuf,
    },
    TarCreate {
        archive: PathBuf,
        directory: PathBuf,
    },
    TarExtract {
        archive: PathBuf,
        directory: PathBuf,
    },
    Spawn {
        argv: Vec<String>,
    },
    Shell {
        shell: String,
    },
    CargoFmt,
    CargoClippy,
    CargoBuild,
    CargoTest,
    CargoInstall,
    Deb,
}

impl Action {
    pub fn names() -> &'static [&'static str] {
        &[
            "mkdir",
            "tar_create",
            "tar_extract",
            "spawn",
            "shell",
            "cargo_fmt",
            "cargo_clippy",
            "cargo_build",
            "cargo_test",
            "cargo_install",
            "deb",
        ]
    }

    pub fn execute(&self) -> Result<(), ActionError> {
        debug!("Plan::execute: {:#?}", self);
        match self {
            Self::Mkdir { pathname } => mkdir(pathname),
            Self::TarCreate { archive, directory } => tar_create(archive, directory),
            Self::TarExtract { archive, directory } => tar_extract(archive, directory),
            Self::Spawn { argv } => spawn(argv, &[]),
            Self::Shell { shell: snippet } => shell(snippet),
            Self::CargoFmt => cargo_fmt(),
            Self::CargoClippy => cargo_clippy(),
            Self::CargoBuild => cargo_build(),
            Self::CargoTest => cargo_test(),
            Self::CargoInstall => cargo_install(),
            Self::Deb => deb(),
        }
    }

    pub fn mkdir<P: AsRef<Path>>(pathname: P) -> Self {
        Self::Mkdir {
            pathname: pathname.as_ref().into(),
        }
    }

    pub fn tar_create<P: AsRef<Path>>(archive: P, directory: P) -> Self {
        Self::TarCreate {
            archive: archive.as_ref().into(),
            directory: directory.as_ref().into(),
        }
    }

    pub fn tar_extract<P: AsRef<Path>>(archive: P, directory: P) -> Self {
        Self::TarExtract {
            archive: archive.as_ref().into(),
            directory: directory.as_ref().into(),
        }
    }

    pub fn spawn(argv: &[&str]) -> Self {
        Self::Spawn {
            argv: argv.iter().map(|s| s.to_string()).collect(),
        }
    }
}

fn mkdir(pathname: &Path) -> Result<(), ActionError> {
    std::fs::create_dir(pathname).map_err(|e| ActionError::Mkdir(pathname.into(), e))
}

fn tar_create(archive: &Path, dirname: &Path) -> Result<(), ActionError> {
    VirtualDriveBuilder::default()
        .filename(archive)
        .root_directory(dirname)
        .create(None)
        .map_err(|e| ActionError::TarCreate(archive.into(), dirname.into(), e))?;
    Ok(())
}

fn tar_extract(archive: &Path, dirname: &Path) -> Result<(), ActionError> {
    let tar = VirtualDriveBuilder::default()
        .filename(archive)
        .root_directory(dirname)
        .open()
        .map_err(|e| ActionError::TarOpen(archive.into(), e))?;
    tar.extract_to(dirname)
        .map_err(|e| ActionError::TarExtract(archive.into(), dirname.into(), e))?;

    Ok(())
}

fn spawn_in(argv: &[String], cwd: &Path, extra_env: &[(&str, &str)]) -> Result<(), ActionError> {
    let argv0 = if let Some(argv0) = argv.first() {
        argv0
    } else {
        return Err(ActionError::SpawnNoArgv0);
    };
    let argv_str = format!("{:?}", argv);

    let mut cmd = Command::new(argv0)
        .args(&argv[1..])
        .envs(extra_env.iter().copied())
        .stdin(Stdio::null())
        .current_dir(cwd)
        .spawn()
        .map_err(|e| ActionError::SpawnInvoke(argv_str.clone(), e))?;

    let exit = cmd.wait();
    let exit = exit.map_err(|e| ActionError::SpawnInvoke(argv_str.clone(), e))?;
    match exit.code() {
        Some(exit) => {
            if exit != 0 {
                return Err(ActionError::SpawnFailed(argv_str, exit));
            }
        }
        None => return Err(ActionError::SpawnKilledBySignal(argv_str)),
    }

    Ok(())
}

fn spawn_str_in(argv: &[&str], cwd: &Path) -> Result<(), ActionError> {
    let argv: Vec<String> = argv.iter().map(|s| s.to_string()).collect();
    spawn_in(&argv, cwd, &[])
}

fn spawn(argv: &[String], extra_env: &[(&str, &str)]) -> Result<(), ActionError> {
    spawn_in(argv, Path::new(qemu::SOURCE_DIR), extra_env)
}

fn shell(snippet: &str) -> Result<(), ActionError> {
    let snippet = format!("set -xeuo pipefail\n{}", snippet);
    spawn(&["/bin/bash".into(), "-c".into(), snippet], &[])
}

fn spawn_str(argv: &[&str], extra_env: &[(&str, &str)]) -> Result<(), ActionError> {
    let argv: Vec<String> = argv.iter().map(|s| s.to_string()).collect();
    spawn(&argv, extra_env)
}

fn prepend_cargo_bin_dir_to_path() -> String {
    let path = std::env::var("PATH").unwrap();
    format!("/root/.cargo/bin:{path}")
}

fn rust_envs(path: &str) -> Vec<(&str, &str)> {
    let mut env = Vec::from(RUST_ENVS);
    env.push(("PATH", path));
    env
}

fn spawn_cargo(args: &[&'static str]) -> Result<(), ActionError> {
    let path = prepend_cargo_bin_dir_to_path();
    spawn_str(args, &rust_envs(&path))
}

fn cargo_fmt() -> Result<(), ActionError> {
    spawn_cargo(&["cargo", "fmt", "--check"])
}

fn cargo_clippy() -> Result<(), ActionError> {
    spawn_cargo(&[
        "cargo",
        "clippy",
        "--offline",
        "--all",
        "--",
        "--deny",
        "clippy::all",
    ])
}

fn cargo_build() -> Result<(), ActionError> {
    spawn_cargo(&[
        "cargo",
        "build",
        "--offline",
        "--workspace",
        "--all-targets",
    ])
}

fn cargo_test() -> Result<(), ActionError> {
    spawn_cargo(&["cargo", "test", "--offline", "--workspace"])
}

fn cargo_install() -> Result<(), ActionError> {
    spawn_cargo(&[
        "cargo",
        "install",
        "--offline",
        "--bins",
        "--path=.",
        "--root",
        qemu::ARTIFACTS_DIR,
    ])
}

fn deb() -> Result<(), ActionError> {
    let shell = format!(
        r#"#!/bin/bash
set -xeuo pipefail

export PATH="/root/.cargo/bin:$PATH"
export CARGO_HOME=/workspace/deps
export DEBEMAIL=liw@liw.fi
export DEBFULLNAME="Lars Wirzenius"
env

command -v cargo
command -v rustc

cargo --version
rustc --version

# Get name and version of source package.
name="$(dpkg-parsechangelog -SSource)"
version="$(dpkg-parsechangelog -SVersion)"

# Get upstream version: everything before the last dash.
uv="$(echo "$version" | sed 's/-[^-]*$//')"

# Files that will be created.
arch="$(dpkg --print-architecture)"
orig="../${{name}}_${{uv}}.orig.tar.xz"
deb="../${{name}}_${{version}}_${{arch}}.deb"
changes="../${{name}}_${{version}}_${{arch}}.changes"

# Create "upstream tarball".
git archive HEAD | xz >"$orig"

# Build package.
dpkg-buildpackage -us -uc

# Dump some information to make it easier to visually verify
# everything looks OK. Also, test the package with the lintian tool.

ls -l ..
for x in ../*.deb; do dpkg -c "$x"; done
lintian -i --allow-root ../*.changes

# Move files to artifacts directory.
mv ../*_* {}
"#,
        qemu::ARTIFACTS_DIR
    );

    spawn_str(&["bash", "-c", &shell], RUST_ENVS)
}

#[derive(Debug, thiserror::Error)]
pub enum ActionError {
    #[error("failed to create directory {0}")]
    Mkdir(PathBuf, #[source] std::io::Error),

    #[error("failed to create tar archive {0} from {1}")]
    TarCreate(PathBuf, PathBuf, #[source] VirtualDriveError),

    #[error("failed to open tar archive {0}")]
    TarOpen(PathBuf, #[source] VirtualDriveError),

    #[error("failed to extract tar archive {0} into {1}")]
    TarExtract(PathBuf, PathBuf, #[source] VirtualDriveError),

    #[error("failed to invoke command: empty argv")]
    SpawnNoArgv0,

    #[error("failed to invoke command: {0}")]
    SpawnInvoke(String, #[source] std::io::Error),

    #[error("command failed was killed by signal")]
    SpawnKilledBySignal(String),

    #[error("command failed: {0}")]
    SpawnFailed(String, i32),

    #[error("failed to remove directory {0}")]
    RemoveDir(PathBuf, #[source] std::io::Error),

    #[error("failed to remove file {0}")]
    RemoveFile(PathBuf, #[source] std::io::Error),

    #[error("failed to create symlink {0} -> {1}")]
    Symlink(PathBuf, PathBuf, #[source] std::io::Error),

    #[error(transparent)]
    Util(#[from] UtilError),

    #[error("for the rsync action you must specify an rsync target (config or command line)")]
    RsyncTargetMissing,

    #[error("for the dput action you must specify a dput target (config or command line)")]
    DputTargetMissing,
}

#[cfg(test)]
mod test {
    use super::*;
    use tempfile::tempdir;

    #[test]
    fn mkdir_action() {
        let tmp = tempdir().unwrap();
        let path = tmp.path().join("testdir");
        let action = Action::mkdir(&path);
        assert!(!path.exists());
        assert!(action.execute().is_ok());
        assert!(path.exists());
    }

    #[test]
    fn tar_create_action() {
        let tmp = tempdir().unwrap();
        let src = tmp.path().join("src");
        let tar = tmp.path().join("src.tar");

        std::fs::create_dir(&src).unwrap();
        let action = Action::tar_create(&tar, &src);

        assert!(!tar.exists());
        assert!(action.execute().is_ok());
        assert!(tar.exists());
    }

    #[test]
    fn tar_extract_action() {
        let tmp = tempdir().unwrap();
        let src = tmp.path().join("src");
        let tar = tmp.path().join("src.tar");
        let extracted = tmp.path().join("extracted");

        std::fs::create_dir(&src).unwrap();
        std::fs::File::create(src.join("file.dat")).unwrap();
        Action::tar_create(&tar, &src).execute().unwrap();

        let action = Action::tar_extract(&tar, &extracted);
        assert!(action.execute().is_ok());
        assert!(extracted.join("file.dat").exists());
    }

    #[test]
    fn spawn_action() {
        // We can't use the Spawn action here, as it expects the
        // SOURCE_DIR to exist. However, spawn_in does the same thing,
        // except in a directory of our choosing.
        assert!(spawn_in(&["true".into()], Path::new("/"), &[]).is_ok());
    }
}
