use std::{
    fs::File,
    path::{Path, PathBuf},
};

use bytesize::GIB;
use directories::ProjectDirs;
use log::debug;
use serde::Deserialize;

use clap::{Parser, Subcommand};

use crate::util::expand_tilde;

const QUAL: &str = "liw.fi";
const ORG: &str = "Ambient CI";
const APP: &str = env!("CARGO_PKG_NAME");

const DEFAULT_TMP: &str = "/tmp";
const DEFAULT_MEMORY: u64 = GIB;

const DEFAULT_ARTIFACTS_MAX_SIZE: u64 = 10 * GIB;
const DEFAULT_CACHE_MAX_SIZE: u64 = 10 * GIB;

#[derive(Debug, Parser)]
#[clap(name = APP, version = env!("CARGO_PKG_VERSION"))]
pub struct Args {
    /// Configuration file to use instead of the default one.
    #[clap(long)]
    config: Option<PathBuf>,

    /// Operation
    #[clap(subcommand)]
    cmd: Command,
}

impl Args {
    pub fn cmd(&self) -> &Command {
        &self.cmd
    }
}

#[derive(Debug, Subcommand)]
pub enum Command {
    Run(RunCommand),
    Config(ConfigCommand),
    Projects(ProjectsCommand),
    Actions(ActionsCommand),
}

#[derive(Debug, Clone, Default, Parser)]
pub struct RunCommand {
    /// Project specification file. May contain any number of projects.
    projects: PathBuf,

    /// Which projects to run CI for, from the ones in the PROJECTS
    /// file. Default is all of them.
    chosen: Option<Vec<String>>,

    /// Build log.
    #[clap(long)]
    log: Option<PathBuf>,

    /// rsync target for publishing ikiwiki output.
    #[clap(long, alias = "rsync")]
    target: Option<String>,

    /// dput target for publishing .deb package.
    #[clap(long, alias = "dput")]
    dput_target: Option<String>,

    /// Path to `run-ci` binary to use to run CI in VM.
    #[clap(long)]
    run_ci: Option<PathBuf>,

    /// Only pretend to run CI.
    #[clap(long)]
    dry_run: bool,

    /// Run even if repository hasn't changed.
    #[clap(long)]
    force: bool,
}

impl RunCommand {
    fn with_config(&mut self, config: &StoredConfig) {
        if self.log.is_none() {
            self.log = config.log.clone();
        }
        if self.target.is_none() {
            self.target = config.target.clone();
        }
        if self.dput_target.is_none() {
            self.dput_target = config.dput_target.clone();
        }
        if self.run_ci.is_none() {
            self.run_ci = config.run_ci.clone();
        }
    }

    pub fn chosen(&self) -> Option<&[String]> {
        self.chosen.as_deref()
    }

    pub fn projects(&self) -> &Path {
        self.projects.as_path()
    }

    pub fn log(&self) -> Option<&Path> {
        self.log.as_deref()
    }

    pub fn target(&self) -> Option<&str> {
        self.target.as_deref()
    }

    pub fn dput_target(&self) -> Option<&str> {
        self.dput_target.as_deref()
    }

    pub fn run_ci(&self) -> Option<&Path> {
        self.run_ci.as_deref()
    }

    pub fn dry_run(&self) -> bool {
        self.dry_run
    }

    pub fn force(&self) -> bool {
        self.force
    }
}

#[derive(Debug, Parser, Clone)]
pub struct ConfigCommand {}

#[derive(Debug, Default, Deserialize)]
pub struct StoredConfig {
    tmpdir: Option<PathBuf>,
    log: Option<PathBuf>,
    state: Option<PathBuf>,
    target: Option<String>,
    dput_target: Option<String>,
    run_ci: Option<PathBuf>,
    cpus: Option<usize>,
    memory: Option<u64>,
    artifacts_max_size: Option<u64>,
    cache_max_size: Option<u64>,
}

impl StoredConfig {
    fn from_file(path: &Path) -> Result<Self, ConfigError> {
        let f = File::open(path).map_err(|e| ConfigError::Read(path.into(), e))?;
        let config: Self =
            serde_yaml::from_reader(f).map_err(|e| ConfigError::Yaml(path.into(), e))?;
        Ok(config)
    }

    pub fn tmpdir(&self) -> PathBuf {
        if let Some(path) = &self.tmpdir {
            path.into()
        } else if let Ok(path) = std::env::var("TMPDIR") {
            PathBuf::from(&path)
        } else {
            PathBuf::from(DEFAULT_TMP)
        }
    }

    pub fn state(&self) -> Option<&Path> {
        self.state.as_deref()
    }
}

// Some of the fields are for the "config" subcommand to write, and
// it's OK if they're not otherwise accessed.
#[allow(dead_code)]
#[derive(Debug)]
pub struct EffectiveConfig {
    config_file: PathBuf,
    config_file_read: bool,
    stored_config: StoredConfig,

    cmd: Command,
}

impl EffectiveConfig {
    pub fn config_file(&self) -> &Path {
        &self.config_file
    }

    pub fn config_file_read(&self) -> bool {
        self.config_file_read
    }

    pub fn stored_config(&self) -> &StoredConfig {
        &self.stored_config
    }

    pub fn cpus(&self) -> usize {
        self.stored_config.cpus.unwrap()
    }

    pub fn memory(&self) -> u64 {
        self.stored_config.memory.unwrap()
    }

    pub fn cmd(&self) -> &Command {
        &self.cmd
    }

    pub fn artfifacts_max_size(&self) -> u64 {
        self.stored_config
            .artifacts_max_size
            .unwrap_or(DEFAULT_ARTIFACTS_MAX_SIZE)
    }

    pub fn cache_max_size(&self) -> u64 {
        self.stored_config
            .cache_max_size
            .unwrap_or(DEFAULT_CACHE_MAX_SIZE)
    }
}

impl TryFrom<&Args> for EffectiveConfig {
    type Error = ConfigError;

    fn try_from(args: &Args) -> Result<Self, Self::Error> {
        let dirs = ProjectDirs::from(QUAL, ORG, APP).ok_or(ConfigError::ProjectDirs)?;
        let config_path = if let Some(config_path) = &args.config {
            config_path.to_path_buf()
        } else {
            dirs.config_dir().join("config.yaml")
        };

        let mut config_file_read = false;

        let mut stored_config = StoredConfig::default();
        if config_path.exists() {
            config_file_read = true;
            stored_config = StoredConfig::from_file(Path::new(&config_path))?;
            if let Some(path) = &stored_config.tmpdir {
                stored_config.tmpdir = Some(expand_tilde(path));
            }
            if let Some(path) = &stored_config.log {
                stored_config.log = Some(expand_tilde(path));
            }
            if let Some(path) = &stored_config.state {
                stored_config.state = Some(expand_tilde(path));
            } else {
                stored_config.state = dirs.state_dir().map(|p| p.into());
            }
            if let Some(tgt) = &stored_config.run_ci {
                stored_config.run_ci = Some(expand_tilde(tgt));
            }
            stored_config.cpus = Some(stored_config.cpus.unwrap_or(1));
            stored_config.memory = Some(stored_config.memory.unwrap_or(DEFAULT_MEMORY));
            debug!("{:#?}", stored_config);
        };

        let cmd = match &args.cmd {
            Command::Run(build) => {
                let mut build = build.clone();
                build.with_config(&stored_config);
                Command::Run(build)
            }
            Command::Config(x) => Command::Config(x.clone()),
            Command::Projects(x) => Command::Projects(x.clone()),
            Command::Actions(x) => Command::Actions(x.clone()),
        };

        Ok(Self {
            config_file: config_path,
            config_file_read,
            stored_config,
            cmd,
        })
    }
}

#[derive(Debug, thiserror::Error)]
pub enum ConfigError {
    #[error("failed to find home directory, while looking for configuration file")]
    ProjectDirs,

    #[error("failed to read configuration file {0}")]
    Read(PathBuf, #[source] std::io::Error),

    #[error("failed to parse configuration file as YAML: {0}")]
    Yaml(PathBuf, #[source] serde_yaml::Error),
}

#[derive(Debug, Parser, Clone)]
pub struct ProjectsCommand {
    /// Name of YAML file with projects.
    projects: PathBuf,
}

impl ProjectsCommand {
    pub fn projects(&self) -> &Path {
        &self.projects
    }
}

#[derive(Debug, Parser, Clone)]
pub struct ActionsCommand {}
