//! Run QEMU.

use std::{
    fs::{copy, File},
    path::{Path, PathBuf},
    process::Command,
};

use bytesize::MIB;
use log::{debug, error, info, trace};
use tempfile::tempdir_in;

use crate::vdrive::{VirtualDrive, VirtualDriveBuilder, VirtualDriveError};

const OVMF_FD: &str = "/usr/share/ovmf/OVMF.fd";

pub const RUN_CI_DRIVE: &str = "/dev/vdb";
pub const SOURCE_DRIVE: &str = "/dev/vdc";
pub const ARTIFACT_DRIVE: &str = "/dev/vdd";
pub const CACHE_DRIVE: &str = "/dev/vde";
pub const DEPS_DRIVE: &str = "/dev/vdf";

pub const WORKSPACE_DIR: &str = "/workspace";
pub const SOURCE_DIR: &str = "/workspace/src";
pub const DEPS_DIR: &str = "/workspace/deps";
pub const CACHE_DIR: &str = "/workspace/cache";
pub const ARTIFACTS_DIR: &str = "/workspace/artifacts";

/// A QEMU runner.
#[derive(Debug, Default)]
pub struct Qemu {
    image: PathBuf,
    tmpdir: PathBuf,
    log: Option<PathBuf>,
    run_ci: PathBuf,
    source: PathBuf,
    artifact: Option<PathBuf>,
    artifact_max_xize: u64,
    dependencies: Option<PathBuf>,
    cache: Option<PathBuf>,
    cache_max_size: u64,
    cpus: usize,
    memory: u64,
}

impl Qemu {
    /// Create a new runner.
    pub fn new(image: &Path, tmpdir: &Path, cpus: usize, memory: u64) -> Self {
        Self {
            image: image.into(),
            tmpdir: tmpdir.into(),
            source: PathBuf::from("."),
            cpus,
            memory,
            ..Default::default()
        }
    }

    /// Set log file.
    pub fn with_log(mut self, log: &Path) -> Self {
        self.log = Some(log.into());
        self
    }

    /// Set directory with run-ci program.
    pub fn with_run_ci(mut self, dirname: &Path) -> Self {
        self.run_ci = dirname.into();
        self
    }

    /// Set source directory.
    pub fn with_source(mut self, source: &Path) -> Self {
        self.source = source.into();
        self
    }

    /// Set output artifact filename.
    pub fn with_artifact(mut self, artifact: &Option<PathBuf>) -> Self {
        self.artifact = artifact.clone();
        self
    }

    /// Set maximum length of output artifact.
    pub fn with_artifact_max_size(mut self, size: u64) -> Self {
        self.artifact_max_xize = size;
        self
    }

    /// Set directory where dependencies are.
    pub fn with_dependencies(mut self, dirname: &Option<PathBuf>) -> Self {
        self.dependencies = dirname.clone();
        self
    }

    /// Set directory where project build cache is.
    pub fn with_cache(mut self, dirname: &Option<PathBuf>) -> Self {
        self.cache = dirname.clone();
        self
    }

    /// Set maximum length of cache.
    pub fn with_cache_max_size(mut self, size: u64) -> Self {
        self.cache_max_size = size;
        self
    }

    /// Run QEMU in the specified way.
    pub fn run(&self) -> Result<i32, QemuError> {
        debug!("run QEMU");
        trace!("{:#?}", self);
        let tmp = tempdir_in(&self.tmpdir).map_err(QemuError::TempDir)?;
        let empty = tempdir_in(&self.tmpdir).map_err(QemuError::TempDir)?;

        let image = tmp.path().join("vm.qcow2");
        let vars = tmp.path().join("vars.fd");
        let console_log = tmp.path().join("console.log");

        debug!("copy image and vars");
        copy(&self.image, &image).map_err(|e| QemuError::Copy(self.image.clone(), e))?;
        copy(OVMF_FD, &vars).map_err(|e| QemuError::Copy(OVMF_FD.into(), e))?;

        debug!("create run-ci drive from {}", self.run_ci.display());
        let run_ci_drive = Self::create_tar(tmp.path().join("run-ci.tar"), &self.run_ci)?;

        let artifact_drive = if let Some(filename) = &self.artifact {
            debug!("create artifact drive at {}", filename.display());
            Self::create_tar_with_size(
                filename.to_path_buf(),
                empty.path(),
                self.artifact_max_xize,
            )?
        } else {
            debug!("create empty artifact drive");
            Self::create_tar_with_size(tmp.path().join("artifacts.tar"), empty.path(), 0)?
        };

        let cache_drive = if let Some(dirname) = &self.cache {
            debug!("create cache drive at {}", dirname.display());
            Self::create_tar_with_size(tmp.path().join("cache.tar"), dirname, self.cache_max_size)?
        } else {
            debug!("no cache drive");
            Self::create_tar_with_size(
                tmp.path().join("cache.tar"),
                empty.path(),
                self.cache_max_size,
            )?
        };

        let deps_drive = if let Some(dirname) = &self.dependencies {
            debug!("create dependencies drive at {}", dirname.display());
            Self::create_tar(tmp.path().join("deps.tar"), dirname)?
        } else {
            debug!("no dependencies drive");
            Self::create_tar(tmp.path().join("deps.tar"), empty.path())?
        };

        debug!("create source drive");
        let tarball = tmp.path().join("src.tar");
        info!("source drive: tarball={}", tarball.display());
        info!("source drive: root={}", self.source.display());
        let source_drive = Self::create_tar(tarball, &self.source)?;

        debug!("set console log file");
        Self::create_file(&Some(console_log.clone()))?;

        debug!("set build log file");
        let build_log = Self::create_file(&self.log)?;

        let cpus = format!("cpus={}", self.cpus);
        let memory = format!("{}", self.memory / MIB);
        let args = QemuArgs::default()
            .with_valued_arg("-m", &memory)
            .with_valued_arg("-smp", &cpus)
            .with_valued_arg("-display", "none")
            .with_valued_arg("-serial", &format!("file:/{}", console_log.display())) // ttyS0
            .with_valued_arg("-serial", &format!("file:{}", build_log.display())) // ttyS1
            .with_ipflash(0, OVMF_FD, true)
            .with_ipflash(1, vars.to_str().unwrap(), false)
            .with_qcow2(image.to_str().unwrap())
            .with_raw(run_ci_drive.filename(), true)
            .with_raw(source_drive.filename(), true)
            .with_raw(artifact_drive.filename(), false)
            .with_raw(cache_drive.filename(), false)
            .with_raw(deps_drive.filename(), true)
            .with_arg("-nodefaults");

        debug!("spawn QEMU");
        let child = Command::new("kvm")
            .args(args.iter())
            .spawn()
            .map_err(QemuError::Invoke)?;

        debug!("wait for QEMU to finish");
        let output = child.wait_with_output().map_err(QemuError::Run)?;
        if output.status.success() {
            debug!("QEMU OK");
            if let Some(dirname) = &self.cache {
                if dirname.exists() {
                    debug!("remove old cache");
                    std::fs::remove_dir_all(dirname)
                        .map_err(|e| QemuError::RemoveCache(dirname.into(), e))?
                }
                debug!("extract cache");
                cache_drive
                    .extract_to(dirname)
                    .map_err(|e| QemuError::ExtractCache(dirname.into(), e))?;
            }

            let log = Self::build_log(&build_log)?;
            let exit = Self::exit_code(&log)?;
            Ok(exit)
        } else {
            let exit = output.status.code().unwrap_or(255);
            let out = String::from_utf8_lossy(&output.stdout).to_string();
            let err = String::from_utf8_lossy(&output.stderr).to_string();
            let log = std::fs::read(&console_log).map_err(QemuError::TemporaryLog)?;
            let log = String::from_utf8_lossy(&log).to_string();
            error!(
                "QEMU failed: exit={}\nstdout: {:?}\nstderr: {:?}\nVM console: {:?}",
                exit, out, err, log
            );
            Ok(exit)
        }
    }

    fn create_file(filename: &Option<PathBuf>) -> Result<PathBuf, QemuError> {
        let filename = if let Some(filename) = filename {
            filename
        } else {
            Path::new("/dev/null")
        };
        File::create(filename).map_err(|e| QemuError::Log(filename.into(), e))?;
        Ok(filename.into())
    }

    fn build_log(filename: &Path) -> Result<String, QemuError> {
        const BEGIN: &str = "====================== BEGIN ======================";

        let log = std::fs::read(filename).map_err(|e| QemuError::ReadLog(filename.into(), e))?;
        let log = String::from_utf8_lossy(&log);

        if let Some((_, log)) = log.split_once(BEGIN) {
            return Ok(log.into());
        }

        Ok("".into())
    }

    fn exit_code(log: &str) -> Result<i32, QemuError> {
        const EXIT: &str = "\nEXIT CODE: ";
        if let Some((_, rest)) = log.split_once(EXIT) {
            trace!("log has {:?}", EXIT);
            if let Some((exit, _)) = rest.split_once('\n') {
                let exit = exit.trim();
                trace!("log has newline after exit: {:?}", exit);
                if let Ok(exit) = exit.parse::<i32>() {
                    trace!("log exit coded parses ok: {}", exit);
                    return Ok(exit);
                }
                debug!("log exit does not parse");
            }
        }
        Err(QemuError::NoExit)
    }

    fn create_tar(tar_filename: PathBuf, dirname: &Path) -> Result<VirtualDrive, QemuError> {
        assert!(!tar_filename.starts_with(dirname));
        let tar = VirtualDriveBuilder::default()
            .filename(&tar_filename)
            .root_directory(dirname)
            .create(None)
            .map_err(|e| QemuError::Tar(dirname.into(), e))?;
        Ok(tar)
    }

    fn create_tar_with_size(
        tar_filename: PathBuf,
        dirname: &Path,
        size: u64,
    ) -> Result<VirtualDrive, QemuError> {
        let tar = VirtualDriveBuilder::default()
            .filename(&tar_filename)
            .root_directory(dirname)
            .size(size)
            .create(None)
            .map_err(|e| QemuError::Tar(dirname.into(), e))?;
        Ok(tar)
    }
}

#[derive(Debug, Default)]
struct QemuArgs {
    args: Vec<String>,
}

impl QemuArgs {
    fn with_arg(mut self, arg: &str) -> Self {
        self.args.push(arg.into());
        self
    }

    fn with_valued_arg(mut self, arg: &str, value: &str) -> Self {
        self.args.push(arg.into());
        self.args.push(value.into());
        self
    }

    fn with_ipflash(mut self, unit: usize, path: &str, readonly: bool) -> Self {
        self.args.push("-drive".into());
        self.args.push(format!(
            "if=pflash,format=raw,unit={},file={}{}",
            unit,
            path,
            if readonly { ",readonly=on" } else { "" },
        ));
        self
    }

    fn with_qcow2(mut self, path: &str) -> Self {
        self.args.push("-drive".into());
        self.args
            .push(format!("format=qcow2,if=virtio,file={}", path));
        self
    }

    fn with_raw(mut self, path: &Path, readonly: bool) -> Self {
        self.args.push("-drive".into());
        self.args.push(format!(
            "format=raw,if=virtio,file={}{}",
            path.display(),
            if readonly { ",readonly=on" } else { "" },
        ));
        self
    }

    fn iter(&self) -> impl Iterator<Item = &str> {
        self.args.iter().map(|s| s.as_str())
    }
}

/// Possible errors from running Qemu.
#[allow(missing_docs)]
#[derive(Debug, thiserror::Error)]
pub enum QemuError {
    #[error("failed to invoke QEMU")]
    Invoke(#[source] std::io::Error),

    #[error("QEMU failed")]
    Run(#[source] std::io::Error),

    #[error("failed to run QEMU (kvm): exit code {0}, stderr:\n{1}\nconsole: {2}")]
    Kvm(i32, String, String),

    #[error("failed to create a temporary directory")]
    TempDir(#[source] std::io::Error),

    #[error("failed to copy to temporary directory: {0}")]
    Copy(PathBuf, #[source] std::io::Error),

    #[error("failed to write temporary file")]
    Write(#[source] std::io::Error),

    #[error("failed to read log file {0}")]
    ReadLog(PathBuf, #[source] std::io::Error),

    #[error("failed to open log file for writing")]
    Log(PathBuf, #[source] std::io::Error),

    #[error("failed to create a tar archive from {0}")]
    Tar(PathBuf, #[source] VirtualDriveError),

    #[error("failed to remove cache (so it can be replaced): {0}")]
    RemoveCache(PathBuf, #[source] std::io::Error),

    #[error("failed to extract cache drive to {0}")]
    ExtractCache(PathBuf, #[source] VirtualDriveError),

    #[error("failed to read temporary file for logging")]
    TemporaryLog(#[source] std::io::Error),

    #[error("build log lacks exit code of run")]
    NoExit,
}

#[cfg(test)]
mod test {
    use super::Qemu;
    use bytesize::GIB;
    use std::path::Path;

    #[test]
    fn sets_image() {
        let path = Path::new("/my/image.qcow2");
        let tmp = Path::new("/tmp");
        let qemu = Qemu::new(path, tmp, 1, GIB);
        assert_eq!(&qemu.image, &path);
    }

    #[test]
    fn sets_log() {
        let path = Path::new("/my/image.qcow2");
        let log = Path::new("/my/log");
        let tmp = Path::new("/tmp");
        let qemu = Qemu::new(path, tmp, 1, GIB).with_log(log);
        assert_eq!(qemu.log, Some(log.into()));
    }
}
