use std::{
    collections::HashSet,
    path::{Path, PathBuf},
};

use log::{debug, info};
use serde::Serialize;
use tempfile::tempdir_in;

use crate::{
    action::{Action, ActionError, PostAction, PreAction},
    config::{Command, EffectiveConfig, ProjectsCommand, RunCommand},
    git::{git_head, git_is_clean, is_git, GitError},
    plan::{Plan, PlanError},
    project::{Project, ProjectError, Projects, State},
    qemu::{self, Qemu, QemuError},
    util::{mkdir, mkdir_child, recreate_dir, UtilError},
    vdrive::{VirtualDriveBuilder, VirtualDriveError},
};

pub fn run(config: &EffectiveConfig) -> Result<(), RunError> {
    match config.cmd() {
        Command::Run(x) => cmd_run(config, x)?,
        Command::Config(_) => cmd_config(config)?,
        Command::Projects(x) => cmd_projects(x)?,
        Command::Actions(_) => cmd_actions(),
    }
    Ok(())
}

fn cmd_run(config: &EffectiveConfig, run: &RunCommand) -> Result<(), RunError> {
    let projects = Projects::from_file(run.projects())?;
    let log = if let Some(log) = run.log() {
        log
    } else {
        Path::new("/dev/null")
    };

    let statedir = config.stored_config().state().ok_or(RunError::NoState)?;
    if !statedir.exists() {
        mkdir(statedir)?;
    }

    for (name, project) in chosen(run, &projects) {
        let (do_run, mut state) = should_run(run, statedir, name, project)?;

        if do_run {
            info!("{}: running CI on {}", name, project.source().display());

            let artifactsdir = state.artifactsdir();
            recreate_dir(&artifactsdir)?;

            let cachedir = state.cachedir();
            if !cachedir.exists() {
                mkdir(&cachedir)?;
            }

            let dependencies = state.dependenciesdir();
            if !dependencies.exists() {
                mkdir(&dependencies)?;
            }

            debug!("Executing pre-plan steps");
            for action in project.pre_plan() {
                action.execute(project, &state)?;
            }

            debug!("Executing CI run");
            let tmp = tempdir_in(config.stored_config().tmpdir()).map_err(RunError::TempDir)?;
            let artifact = tmp.path().join("artifact.tar");

            let run_ci_dir = mkdir_child(tmp.path(), "run-ci")?;
            assert!(run.run_ci().is_some());
            let run_ci_bin = &run.run_ci().unwrap();
            let run_ci_exe = run_ci_dir.join("run-ci");

            debug!(
                "copying {} to {}",
                run_ci_bin.display(),
                run_ci_exe.display()
            );
            std::fs::copy(run_ci_bin, &run_ci_exe)
                .map_err(|e| RunError::Copy(run_ci_bin.into(), run_ci_exe, e))?;

            let mut plan = Plan::default();
            prelude(&mut plan);
            for action in project.plan().iter() {
                plan.push(action.clone());
            }
            epilog(&mut plan);
            debug!("plan: {:#?}", plan);
            let plan_filename = run_ci_dir.join("plan.yaml");
            plan.to_file(&plan_filename).map_err(RunError::Plan)?;

            let qemu = Qemu::new(
                project.image(),
                &config.stored_config().tmpdir(),
                config.cpus(),
                config.memory(),
            )
            .with_run_ci(&run_ci_dir)
            .with_source(project.source())
            .with_artifact(&Some(artifact.clone()))
            .with_artifact_max_size(
                project
                    .artifact_max_size()
                    .unwrap_or(config.artfifacts_max_size()),
            )
            .with_dependencies(&Some(dependencies.clone()))
            .with_cache(&Some(cachedir.clone()))
            .with_cache_max_size(project.cache_max_size().unwrap_or(config.cache_max_size()))
            .with_log(log);
            let exit = qemu.run()?;
            debug!("QEMU exit code {}", exit);
            if exit != 0 {
                return Err(RunError::QemuFailed(exit));
            }

            let vdrive = VirtualDriveBuilder::default().filename(&artifact).open()?;
            vdrive.extract_to(&artifactsdir)?;

            debug!("Executing post-plan steps");
            for action in project.post_plan() {
                action.execute(project, &state, run)?;
            }

            if is_git(project.source()) {
                let head = git_head(project.source())?;
                state.set_latest_commot(Some(&head));
            } else {
                state.set_latest_commot(None);
            };
            state.write_to_file()?;
        }
    }

    Ok(())
}

fn should_run(
    run: &RunCommand,
    statedir: &Path,
    name: &str,
    project: &Project,
) -> Result<(bool, State), RunError> {
    let state = State::from_file(statedir, name)?;
    let head = if is_git(project.source()) {
        Some(git_head(project.source())?)
    } else {
        None
    };
    debug!("latest commit: {:?}", state.latest_commit());
    debug!("HEAD commit:   {:?}", head);

    let mut do_run = false;

    if !is_git(project.source()) {
        debug!("not a git repository: {}", project.source().display());
        do_run = true;
    } else if !git_is_clean(project.source()) {
        debug!(
            "git repository is not clean: {}",
            project.source().display()
        );
        do_run = true;
    } else if state.latest_commit() != head.as_deref() {
        debug!("git repository has changed");
        do_run = true;
    }

    if run.force() && run.dry_run() {
        debug!("both --dry-run and --force used, --dry run wins");
        do_run = false;
    } else if run.force() {
        debug!("--force used");
        do_run = true;
    }

    if do_run {
        if run.dry_run() {
            info!("would run CI on {} but --dry-run used", name);
        } else {
            info!("running CI on {}", name);
        }
    } else {
        info!("skipping {}", name);
    }

    Ok((do_run, state))
}

fn chosen<'a>(run: &'a RunCommand, projects: &'a Projects) -> Vec<(&'a str, &'a Project)> {
    let set: HashSet<&str> = match run.chosen() {
        Some(v) if !v.is_empty() => v.iter().map(|s| s.as_str()).collect(),
        _ => projects.iter().map(|(k, _)| k).collect(),
    };
    let mut projects: Vec<(&'a str, &'a Project)> =
        projects.iter().filter(|(k, _)| set.contains(k)).collect();
    projects.sort_by(|(a_name, _), (b_name, _)| a_name.cmp(b_name));
    projects
}

fn prelude(plan: &mut Plan) {
    plan.push(Action::mkdir(Path::new(qemu::WORKSPACE_DIR)));
    plan.push(Action::mkdir(Path::new(qemu::ARTIFACTS_DIR)));

    plan.push(Action::tar_extract(
        Path::new(qemu::SOURCE_DRIVE),
        Path::new(qemu::SOURCE_DIR),
    ));

    plan.push(Action::tar_extract(
        Path::new(qemu::DEPS_DRIVE),
        Path::new(qemu::DEPS_DIR),
    ));

    plan.push(Action::tar_extract(
        Path::new(qemu::CACHE_DRIVE),
        Path::new(qemu::CACHE_DIR),
    ));

    plan.push(Action::spawn(&[
        "find",
        "/workspace",
        "-maxdepth",
        "2",
        "-ls",
    ]));
}

fn epilog(plan: &mut Plan) {
    plan.push(Action::tar_create(
        Path::new(qemu::CACHE_DRIVE),
        Path::new(qemu::CACHE_DIR),
    ));
    plan.push(Action::tar_create(
        Path::new(qemu::ARTIFACT_DRIVE),
        Path::new(qemu::ARTIFACTS_DIR),
    ));
}

fn cmd_config(config: &EffectiveConfig) -> Result<(), RunError> {
    println!("{:#?}", config);
    Ok(())
}

fn cmd_projects(projcts: &ProjectsCommand) -> Result<(), RunError> {
    let projects = Projects::from_file(projcts.projects())?;
    let mut names: Vec<&str> = projects.iter().map(|(name, _)| name).collect();
    names.sort();
    for name in names.iter() {
        println!("{}", name);
    }
    Ok(())
}

fn cmd_actions() {
    #[derive(Serialize)]
    struct Actions {
        pre_actions: Vec<&'static str>,
        actions: Vec<&'static str>,
        post_actions: Vec<&'static str>,
    }

    let mut actions = Actions {
        pre_actions: PreAction::names().to_vec(),
        actions: Action::names().to_vec(),
        post_actions: PostAction::names().to_vec(),
    };

    actions.pre_actions.sort();
    actions.actions.sort();
    actions.post_actions.sort();

    print!("{}", serde_yaml::to_string(&actions).unwrap());
}

#[derive(Debug, thiserror::Error)]
pub enum RunError {
    #[error(transparent)]
    Project(#[from] ProjectError),

    #[error(transparent)]
    Util(#[from] UtilError),

    #[error("failed to create temporary directory for running CI build")]
    TempDir(#[source] std::io::Error),

    #[error("failed to read ikikwiki setup file {0}")]
    ReadSetup(PathBuf, #[source] std::io::Error),

    #[error("failed to parse ikiwiki setup file as YAML: {0}")]
    SetupYaml(PathBuf, #[source] serde_yaml::Error),

    #[error("failed to figure out parent directory of {0}")]
    Parent(PathBuf),

    #[error("failed to make filename absolute: {0}")]
    Canonicalize(PathBuf, #[source] std::io::Error),

    #[error("failed to run ambient-run")]
    AmbientRunInvoke(#[source] std::io::Error),

    #[error("ambient-run failed:\n{0}")]
    AmbientRun(String),

    #[error("ambient-run didn't create the artifact tar archive {0}")]
    NoArtifact(PathBuf),

    #[error("failed to create temporary directory for unpacking artifacts")]
    CreateStaging(PathBuf, #[source] std::io::Error),

    #[error("failed to open artifact tar archive {0}")]
    OpenArtifacts(PathBuf, #[source] std::io::Error),

    #[error("failed to extract artifact archive {0} to {0}")]
    Extract(PathBuf, PathBuf, #[source] std::io::Error),

    #[error(transparent)]
    Qemu(#[from] QemuError),

    #[error("QEMU failed")]
    QemuFailed(i32),

    #[error("failed to copy {0} to {1}")]
    Copy(PathBuf, PathBuf, #[source] std::io::Error),

    #[error(transparent)]
    Plan(#[from] PlanError),

    #[error(transparent)]
    Action(#[from] ActionError),

    #[error(transparent)]
    Git(#[from] GitError),

    #[error(transparent)]
    VDrive(#[from] VirtualDriveError),

    #[error("no state directory specified")]
    NoState,
}
