use std::path::{Path, PathBuf};

use serde::{Deserialize, Serialize};

use crate::action::UnsafeAction;

#[derive(Debug, Clone, Default, Serialize, Deserialize)]
pub struct Plan {
    steps: Vec<UnsafeAction>,
}

impl Plan {
    pub fn from_file(filename: &Path) -> Result<Self, PlanError> {
        let plan = std::fs::read(filename).map_err(|e| PlanError::PlanOpen(filename.into(), e))?;
        let plan =
            serde_yaml::from_slice(&plan).map_err(|e| PlanError::PlanParse(filename.into(), e))?;
        Ok(plan)
    }

    pub fn to_file(&self, filename: &Path) -> Result<(), PlanError> {
        let plan = serde_yaml::to_string(&self).map_err(PlanError::PlanSerialize)?;
        std::fs::write(filename, plan).map_err(|e| PlanError::PlanWrite(filename.into(), e))?;
        Ok(())
    }

    pub fn push(&mut self, action: UnsafeAction) {
        self.steps.push(action);
    }

    pub fn iter(&self) -> impl Iterator<Item = &UnsafeAction> {
        self.steps.iter()
    }
}

#[derive(Debug, thiserror::Error)]
pub enum PlanError {
    #[error("failed to read CI plan file: {0}")]
    PlanOpen(PathBuf, #[source] std::io::Error),

    #[error("failed to parse CI plan file as YAML: {0}")]
    PlanParse(PathBuf, #[source] serde_yaml::Error),

    #[error("failed to serialize CI plan as YAML")]
    PlanSerialize(#[source] serde_yaml::Error),

    #[error("failed to write CI plan file: {0}")]
    PlanWrite(PathBuf, #[source] std::io::Error),
}
