//! Run QEMU.

use std::{
    fs::{copy, File},
    path::{Path, PathBuf},
    process::Command,
};

use bytesize::MIB;
use log::{debug, error, trace};
use tempfile::tempdir_in;

use crate::vdrive::{VirtualDrive, VirtualDriveError};

const OVMF_FD: &str = "/usr/share/ovmf/OVMF.fd";

pub const RUN_CI_DRIVE: &str = "/dev/vdb";
pub const SOURCE_DRIVE: &str = "/dev/vdc";
pub const ARTIFACT_DRIVE: &str = "/dev/vdd";
pub const CACHE_DRIVE: &str = "/dev/vde";
pub const DEPS_DRIVE: &str = "/dev/vdf";

pub const WORKSPACE_DIR: &str = "/workspace";
pub const SOURCE_DIR: &str = "/workspace/src";
pub const DEPS_DIR: &str = "/workspace/deps";
pub const CACHE_DIR: &str = "/workspace/cache";
pub const ARTIFACTS_DIR: &str = "/workspace/artifacts";

/// Build a QEMU runner.
#[derive(Debug)]
pub struct QemuBuilder<'a> {
    tmpdir: PathBuf,
    image: Option<PathBuf>,
    log: Option<PathBuf>,
    run_ci: Option<&'a VirtualDrive>,
    source: Option<&'a VirtualDrive>,
    artifact: Option<&'a VirtualDrive>,
    dependencies: Option<&'a VirtualDrive>,
    cache: Option<&'a VirtualDrive>,
    cpus: Option<usize>,
    memory: Option<u64>,
}

impl<'a> QemuBuilder<'a> {
    /// Create a new runner.
    pub fn new(tmpdir: &Path) -> Self {
        Self {
            image: None,
            tmpdir: tmpdir.into(),
            source: None,
            log: None,
            run_ci: None,
            artifact: None,
            dependencies: None,
            cache: None,
            cpus: None,
            memory: None,
        }
    }

    /// Set base image to use.
    pub fn with_image(mut self, image: &Path) -> Self {
        self.image = Some(image.into());
        self
    }

    /// Set number of virtual CPUs to use.
    pub fn with_cpus(mut self, cpus: usize) -> Self {
        self.cpus = Some(cpus);
        self
    }

    /// Set amount of memory to use.
    pub fn with_memory(mut self, memory: u64) -> Self {
        self.memory = Some(memory);
        self
    }

    /// Set log file.
    pub fn with_log(mut self, log: &Path) -> Self {
        self.log = Some(log.into());
        self
    }

    /// Set directory with run-ci program.
    pub fn with_run_ci(mut self, drive: &'a VirtualDrive) -> Self {
        self.run_ci = Some(drive);
        self
    }

    /// Set source directory.
    pub fn with_source(mut self, source: &'a VirtualDrive) -> Self {
        self.source = Some(source);
        self
    }

    /// Set output artifact filename.
    pub fn with_artifact(mut self, artifact: &'a VirtualDrive) -> Self {
        self.artifact = Some(artifact);
        self
    }

    /// Set directory where dependencies are.
    pub fn with_dependencies(mut self, drive: &'a VirtualDrive) -> Self {
        self.dependencies = Some(drive);
        self
    }

    /// Set directory where project build cache is.
    pub fn with_cache(mut self, drive: &'a VirtualDrive) -> Self {
        self.cache = Some(drive);
        self
    }

    /// Create a [`Qemu`], or panic if something is wrong.
    pub fn build(self) -> Result<Qemu<'a>, QemuError> {
        Ok(Qemu {
            image: self.image.ok_or_else(|| QemuError::Missing("image"))?,
            tmpdir: self.tmpdir,
            log: self.log.ok_or_else(|| QemuError::Missing("log"))?,
            run_ci: self.run_ci,
            source: self.source,
            artifact: self.artifact,
            dependencies: self.dependencies,
            cache: self.cache,
            cpus: self.cpus.unwrap(),
            memory: self.memory.unwrap(),
        })
    }
}

/// A QEMU runner.
#[derive(Debug)]
pub struct Qemu<'a> {
    image: PathBuf,
    tmpdir: PathBuf,
    log: PathBuf,
    run_ci: Option<&'a VirtualDrive>,
    source: Option<&'a VirtualDrive>,
    artifact: Option<&'a VirtualDrive>,
    dependencies: Option<&'a VirtualDrive>,
    cache: Option<&'a VirtualDrive>,
    cpus: usize,
    memory: u64,
}

impl<'a> Qemu<'a> {
    /// Run QEMU in the specified way.
    pub fn run(&self) -> Result<i32, QemuError> {
        debug!("run QEMU");
        trace!("{:#?}", self);
        let tmp = tempdir_in(&self.tmpdir).map_err(QemuError::TempDir)?;

        let image = tmp.path().join("vm.qcow2");
        let vars = tmp.path().join("vars.fd");
        let console_log = tmp.path().join("console.log");

        debug!("create copy-on-write image and UEFI vars file");
        create_cow_image(&self.image, &image)?;
        copy(OVMF_FD, &vars).map_err(|e| QemuError::Copy(OVMF_FD.into(), e))?;

        let source = self.source.unwrap();
        let artifact_drive = self.artifact.unwrap();
        let deps_drive = self.dependencies.unwrap();
        let cache_drive = self.cache.unwrap();
        let run_ci_drive = self.run_ci.unwrap();

        debug!("set console log file");
        Self::create_file(&console_log)?;

        debug!("set build log file");
        let build_log = Self::create_file(&self.log)?;

        let cpus = format!("cpus={}", self.cpus);
        let memory = format!("{}", self.memory / MIB);
        let args = QemuArgs::default()
            .with_valued_arg("-m", &memory)
            .with_valued_arg("-smp", &cpus)
            .with_valued_arg("-display", "none")
            .with_valued_arg("-serial", &format!("file:/{}", console_log.display())) // ttyS0
            .with_valued_arg("-serial", &format!("file:{}", build_log.display())) // ttyS1
            .with_ipflash(0, OVMF_FD, true)
            .with_ipflash(1, vars.to_str().unwrap(), false)
            .with_qcow2(image.to_str().unwrap())
            .with_raw(run_ci_drive.filename(), true)
            .with_raw(source.filename(), true)
            .with_raw(artifact_drive.filename(), false)
            .with_raw(cache_drive.filename(), false)
            .with_raw(deps_drive.filename(), true)
            .with_arg("-nodefaults");

        debug!("spawn QEMU");
        let child = Command::new("kvm")
            .args(args.iter())
            .spawn()
            .map_err(QemuError::Invoke)?;

        debug!("wait for QEMU to finish");
        let output = child.wait_with_output().map_err(QemuError::Run)?;
        if output.status.success() {
            debug!("QEMU OK");
            let log = Self::build_log(&build_log)?;
            let exit = Self::exit_code(&log)?;
            Ok(exit)
        } else {
            let exit = output.status.code().unwrap_or(255);
            let out = String::from_utf8_lossy(&output.stdout).to_string();
            let err = String::from_utf8_lossy(&output.stderr).to_string();
            let log = std::fs::read(&console_log).map_err(QemuError::TemporaryLog)?;
            let log = String::from_utf8_lossy(&log).to_string();
            error!(
                "QEMU failed: exit={}\nstdout: {:?}\nstderr: {:?}\nVM console: {:?}",
                exit, out, err, log
            );
            Ok(exit)
        }
    }

    fn create_file(filename: &Path) -> Result<PathBuf, QemuError> {
        File::create(filename).map_err(|e| QemuError::Log(filename.into(), e))?;
        Ok(filename.into())
    }

    fn build_log(filename: &Path) -> Result<String, QemuError> {
        const BEGIN: &str = "====================== BEGIN ======================";

        let log = std::fs::read(filename).map_err(|e| QemuError::ReadLog(filename.into(), e))?;
        let log = String::from_utf8_lossy(&log);

        if let Some((_, log)) = log.split_once(BEGIN) {
            return Ok(log.into());
        }

        Ok("".into())
    }

    fn exit_code(log: &str) -> Result<i32, QemuError> {
        const EXIT: &str = "\nEXIT CODE: ";
        if let Some((_, rest)) = log.split_once(EXIT) {
            trace!("log has {:?}", EXIT);
            if let Some((exit, _)) = rest.split_once('\n') {
                let exit = exit.trim();
                trace!("log has newline after exit: {:?}", exit);
                if let Ok(exit) = exit.parse::<i32>() {
                    trace!("log exit coded parses ok: {}", exit);
                    return Ok(exit);
                }
                debug!("log exit does not parse");
            }
        }
        Err(QemuError::NoExit)
    }
}

fn create_cow_image(backing_file: &Path, new_file: &Path) -> Result<(), QemuError> {
    let output = Command::new("qemu-img")
        .arg("create")
        .arg("-b")
        .arg(backing_file)
        .args(["-F", "qcow2"])
        .args(["-f", "qcow2"])
        .arg(new_file)
        .output()
        .map_err(QemuError::Run)?;
    if !output.status.success() {
        let exit = output.status.code().unwrap_or(255);
        let out = String::from_utf8_lossy(&output.stdout).to_string();
        let err = String::from_utf8_lossy(&output.stderr).to_string();
        error!(
            "qemu-img failed: exit={}\nstdout: {:?}\nstderr: {:?}",
            exit, out, err,
        );
        Err(QemuError::QemuImg(exit, err))
    } else {
        Ok(())
    }
}

#[derive(Debug, Default)]
struct QemuArgs {
    args: Vec<String>,
}

impl QemuArgs {
    fn with_arg(mut self, arg: &str) -> Self {
        self.args.push(arg.into());
        self
    }

    fn with_valued_arg(mut self, arg: &str, value: &str) -> Self {
        self.args.push(arg.into());
        self.args.push(value.into());
        self
    }

    fn with_ipflash(mut self, unit: usize, path: &str, readonly: bool) -> Self {
        self.args.push("-drive".into());
        self.args.push(format!(
            "if=pflash,format=raw,unit={},file={}{}",
            unit,
            path,
            if readonly { ",readonly=on" } else { "" },
        ));
        self
    }

    fn with_qcow2(mut self, path: &str) -> Self {
        self.args.push("-drive".into());
        self.args
            .push(format!("format=qcow2,if=virtio,file={}", path));
        self
    }

    fn with_raw(mut self, path: &Path, readonly: bool) -> Self {
        self.args.push("-drive".into());
        self.args.push(format!(
            "format=raw,if=virtio,file={}{}",
            path.display(),
            if readonly { ",readonly=on" } else { "" },
        ));
        self
    }

    fn iter(&self) -> impl Iterator<Item = &str> {
        self.args.iter().map(|s| s.as_str())
    }
}

/// Possible errors from running Qemu.
#[allow(missing_docs)]
#[derive(Debug, thiserror::Error)]
pub enum QemuError {
    #[error("missing field in QemuBuilder: {0}")]
    Missing(&'static str),

    #[error("failed to invoke QEMU")]
    Invoke(#[source] std::io::Error),

    #[error("QEMU failed")]
    Run(#[source] std::io::Error),

    #[error("qemu-img failed: exit code {0}: stderr:\n{1}")]
    QemuImg(i32, String),

    #[error("failed to run QEMU (kvm): exit code {0}, stderr:\n{1}\nconsole: {2}")]
    Kvm(i32, String, String),

    #[error("failed to create a temporary directory")]
    TempDir(#[source] std::io::Error),

    #[error("failed to copy to temporary directory: {0}")]
    Copy(PathBuf, #[source] std::io::Error),

    #[error("failed to write temporary file")]
    Write(#[source] std::io::Error),

    #[error("failed to read log file {0}")]
    ReadLog(PathBuf, #[source] std::io::Error),

    #[error("failed to open log file for writing")]
    Log(PathBuf, #[source] std::io::Error),

    #[error("failed to create a tar archive from {0}")]
    Tar(PathBuf, #[source] VirtualDriveError),

    #[error("failed to remove cache (so it can be replaced): {0}")]
    RemoveCache(PathBuf, #[source] std::io::Error),

    #[error("failed to extract cache drive to {0}")]
    ExtractCache(PathBuf, #[source] VirtualDriveError),

    #[error("failed to read temporary file for logging")]
    TemporaryLog(#[source] std::io::Error),

    #[error("build log lacks exit code of run")]
    NoExit,
}

#[cfg(test)]
mod test {
    use super::QemuBuilder;
    use bytesize::GIB;
    use std::path::Path;

    #[test]
    fn sets_image() {
        let path = Path::new("/my/image.qcow2");
        let log = Path::new("/tmp/log");
        let tmp = Path::new("/tmp");
        let qemu = QemuBuilder::new(tmp)
            .with_image(path)
            .with_log(log)
            .with_cpus(1)
            .with_memory(GIB)
            .build()
            .unwrap();
        assert_eq!(&qemu.image, &path);
    }

    #[test]
    fn sets_log() {
        let path = Path::new("/my/image.qcow2");
        let log = Path::new("/my/log");
        let tmp = Path::new("/tmp");
        let qemu = QemuBuilder::new(tmp)
            .with_image(path)
            .with_cpus(1)
            .with_memory(GIB)
            .with_log(log)
            .build()
            .unwrap();
        assert_eq!(qemu.log, log);
    }
}
