use std::{
    collections::HashSet,
    path::{Path, PathBuf},
};

use log::{debug, info};
use serde::Serialize;
use tempfile::tempdir_in;

use crate::{
    action::{ActionError, TrustedAction, UnsafeAction},
    config::{Command, EffectiveConfig, ProjectsCommand, RunCommand},
    git::{git_head, git_is_clean, is_git, GitError},
    plan::{Plan, PlanError},
    project::{Project, ProjectError, Projects, State},
    qemu::{self, QemuBuilder, QemuError},
    util::{mkdir, mkdir_child, recreate_dir, UtilError},
    vdrive::{VirtualDrive, VirtualDriveBuilder, VirtualDriveError},
};

pub fn run(config: &EffectiveConfig) -> Result<(), RunError> {
    match config.cmd() {
        Command::Run(x) => cmd_run(config, x)?,
        Command::Config(_) => cmd_config(config)?,
        Command::Projects(x) => cmd_projects(x)?,
        Command::Actions(_) => cmd_actions(),
    }
    Ok(())
}

fn cmd_run(config: &EffectiveConfig, run: &RunCommand) -> Result<(), RunError> {
    let projects = Projects::from_file(run.projects())?;
    let log = if let Some(log) = run.log() {
        log
    } else {
        Path::new("/dev/null")
    };

    let statedir = config.stored_config().state().ok_or(RunError::NoState)?;
    if !statedir.exists() {
        debug!(
            "Create state directory shared by all projects {}",
            statedir.display()
        );
        mkdir(statedir).map_err(|e| RunError::MkdirState(statedir.into(), e))?;
    }

    for (name, project) in chosen(run, &projects) {
        let (do_run, mut state) = should_run(run, statedir, name, project)?;

        if do_run {
            info!("project {}: running CI", name);

            let plan = construct_unsafe_plan(project)?;

            let tmp = tempdir_in(config.stored_config().tmpdir()).map_err(RunError::TempDir)?;

            let source_drive = create_tar(tmp.path().join("src.tar"), project.source())?;

            let artifactsdir = state.artifactsdir();
            recreate_dir(&artifactsdir)?;
            let artifact_drive = create_tar_with_size(
                tmp.path().join("artifacts.tar"),
                &artifactsdir,
                project
                    .artifact_max_size()
                    .unwrap_or(config.artfifacts_max_size()),
            )?;

            let dependencies = state.dependenciesdir();
            if !dependencies.exists() {
                mkdir(&dependencies)
                    .map_err(|e| RunError::MkdirProjectSubState(dependencies.clone(), e))?;
            }
            let deps_drive = create_tar(tmp.path().join("deps.tar"), &dependencies)?;

            let cachedir = state.cachedir();
            if !cachedir.exists() {
                mkdir(&cachedir)
                    .map_err(|e| RunError::MkdirProjectSubState(cachedir.clone(), e))?;
            }
            let cache_drive = create_tar_with_size(
                tmp.path().join("cache.tar"),
                &cachedir,
                project.cache_max_size().unwrap_or(config.cache_max_size()),
            )?;

            let run_ci_drive = {
                assert!(run.run_ci().is_some());
                let bin = &run.run_ci().unwrap();

                let dirname =
                    mkdir_child(tmp.path(), "run-ci").map_err(RunError::MkdirProjectRunCi)?;
                let bin2 = dirname.join("run-ci");

                debug!("copying {} to {}", bin.display(), bin2.display());
                std::fs::copy(bin, &bin2).map_err(|e| RunError::Copy(bin.into(), bin2, e))?;

                let plan_filename = dirname.join("plan.yaml");
                std::fs::write(&plan_filename, plan.as_bytes())
                    .map_err(|e| RunError::PlanWrite(plan_filename.clone(), e))?;

                create_tar(tmp.path().join("run-ci.tar"), &dirname)?
            };

            debug!("Executing pre-plan steps");
            for action in project.pre_plan() {
                action.execute(project, &state, run)?;
            }

            run_in_qemu(
                config,
                project,
                &run_ci_drive,
                &source_drive,
                &deps_drive,
                &cache_drive,
                &artifact_drive,
                log,
            )?;

            artifact_drive.extract_to(&artifactsdir)?;

            debug!("remove old cache");
            std::fs::remove_dir_all(&cachedir)
                .map_err(|e| QemuError::RemoveCache(cachedir.clone(), e))?;
            debug!("extract cache");
            cache_drive
                .extract_to(&cachedir)
                .map_err(|e| QemuError::ExtractCache(cachedir.clone(), e))?;

            debug!("Executing post-plan steps");
            for action in project.post_plan() {
                action.execute(project, &state, run)?;
            }

            if is_git(project.source()) {
                let head = git_head(project.source())?;
                state.set_latest_commot(Some(&head));
            } else {
                state.set_latest_commot(None);
            };
            state.write_to_file()?;
        } else {
            info!("project {}: NOT running CI", name);
        }
    }

    Ok(())
}

fn create_tar(tar_filename: PathBuf, dirname: &Path) -> Result<VirtualDrive, QemuError> {
    assert!(!tar_filename.starts_with(dirname));
    debug!("create virtual drive {}", tar_filename.display());
    let tar = VirtualDriveBuilder::default()
        .filename(&tar_filename)
        .root_directory(dirname)
        .create(None)
        .map_err(|e| QemuError::Tar(dirname.into(), e))?;
    Ok(tar)
}

fn create_tar_with_size(
    tar_filename: PathBuf,
    dirname: &Path,
    size: u64,
) -> Result<VirtualDrive, QemuError> {
    let tar = VirtualDriveBuilder::default()
        .filename(&tar_filename)
        .root_directory(dirname)
        .size(size)
        .create(None)
        .map_err(|e| QemuError::Tar(dirname.into(), e))?;
    Ok(tar)
}

#[allow(clippy::too_many_arguments)]
fn run_in_qemu(
    config: &EffectiveConfig,
    project: &Project,
    run_ci_drive: &VirtualDrive,
    source: &VirtualDrive,
    dependencies: &VirtualDrive,
    cache_drive: &VirtualDrive,
    artifacts: &VirtualDrive,
    log: &Path,
) -> Result<(), RunError> {
    debug!("Executing CI run");

    let qemu = QemuBuilder::new(&config.stored_config().tmpdir())
        .with_image(project.image())
        .with_cpus(config.cpus())
        .with_memory(config.memory())
        .with_run_ci(run_ci_drive)
        .with_source(source)
        .with_artifact(artifacts)
        .with_dependencies(dependencies)
        .with_cache(cache_drive)
        .with_log(log)
        .build()?;
    let exit = qemu.run()?;
    debug!("QEMU exit code {}", exit);
    if exit != 0 {
        return Err(RunError::QemuFailed(exit));
    }

    Ok(())
}

fn construct_unsafe_plan(project: &Project) -> Result<String, RunError> {
    let mut plan = Plan::default();
    prelude(&mut plan);
    for action in project.plan().iter() {
        plan.push(action.clone());
    }
    epilog(&mut plan);
    debug!("plan: {:#?}", plan);

    serde_yaml::to_string(&plan).map_err(RunError::PlanSerialize)
}

fn should_run(
    run: &RunCommand,
    statedir: &Path,
    name: &str,
    project: &Project,
) -> Result<(bool, State), RunError> {
    let mut decision = Decision::default();

    let state = State::from_file(statedir, name)?;
    if let Some(latest_commit) = state.latest_commit() {
        debug!("latest commit: {:?}", latest_commit);
        decision.latest_commit(latest_commit);
    } else {
        debug!("no latest commit stored");
        // No need to set latest commit to anything, the default is OK.
    }

    let is_git = is_git(project.source());
    if is_git {
        debug!("is a git repository");
        decision.is_git();
    } else {
        debug!("is not a git repository");
        decision.is_not_git();
    }

    if git_is_clean(project.source()) {
        debug!("git repository is clean");
        decision.is_clean();
    } else {
        debug!("git repository is dirty");
        decision.is_dirty();
    }

    if is_git {
        let head = git_head(project.source())?;
        debug!("current (HEAD) commit: {head}");
        decision.current_commit(&head);
    } else {
        debug!("no current commit due to not git repository");
        // No need to set current commit to anything, the default is OK.
    }

    if run.dry_run() {
        debug!("dry run requested");
        decision.dry_run();
    } else {
        debug!("no dry run requested");
        decision.no_dry_run();
    }

    if run.force() {
        debug!("forced run requested");
        decision.force();
    } else {
        debug!("no forced run requested");
        decision.no_force();
    }

    let do_run = decision.should_run() == ShouldRun::Run;
    debug!("run? {do_run}");

    Ok((do_run, state))
}

fn chosen<'a>(run: &'a RunCommand, projects: &'a Projects) -> Vec<(&'a str, &'a Project)> {
    let set: HashSet<&str> = match run.chosen() {
        Some(v) if !v.is_empty() => v.iter().map(|s| s.as_str()).collect(),
        _ => projects.iter().map(|(k, _)| k).collect(),
    };
    let mut projects: Vec<(&'a str, &'a Project)> =
        projects.iter().filter(|(k, _)| set.contains(k)).collect();
    projects.sort_by(|(a_name, _), (b_name, _)| a_name.cmp(b_name));
    projects
}

fn prelude(plan: &mut Plan) {
    plan.push(UnsafeAction::mkdir(Path::new(qemu::WORKSPACE_DIR)));
    plan.push(UnsafeAction::mkdir(Path::new(qemu::ARTIFACTS_DIR)));

    plan.push(UnsafeAction::tar_extract(
        Path::new(qemu::SOURCE_DRIVE),
        Path::new(qemu::SOURCE_DIR),
    ));

    plan.push(UnsafeAction::tar_extract(
        Path::new(qemu::DEPS_DRIVE),
        Path::new(qemu::DEPS_DIR),
    ));

    plan.push(UnsafeAction::tar_extract(
        Path::new(qemu::CACHE_DRIVE),
        Path::new(qemu::CACHE_DIR),
    ));

    plan.push(UnsafeAction::spawn(&[
        "find",
        "/workspace",
        "-maxdepth",
        "2",
        "-ls",
    ]));
}

fn epilog(plan: &mut Plan) {
    plan.push(UnsafeAction::tar_create(
        Path::new(qemu::CACHE_DRIVE),
        Path::new(qemu::CACHE_DIR),
    ));
    plan.push(UnsafeAction::tar_create(
        Path::new(qemu::ARTIFACT_DRIVE),
        Path::new(qemu::ARTIFACTS_DIR),
    ));
}

fn cmd_config(config: &EffectiveConfig) -> Result<(), RunError> {
    println!("{:#?}", config);
    Ok(())
}

fn cmd_projects(projcts: &ProjectsCommand) -> Result<(), RunError> {
    let projects = Projects::from_file(projcts.projects())?;
    let mut names: Vec<&str> = projects.iter().map(|(name, _)| name).collect();
    names.sort();
    for name in names.iter() {
        println!("{}", name);
    }
    Ok(())
}

fn cmd_actions() {
    #[derive(Serialize)]
    struct Actions {
        pre_actions: Vec<&'static str>,
        actions: Vec<&'static str>,
        post_actions: Vec<&'static str>,
    }

    let mut actions = Actions {
        pre_actions: TrustedAction::names().to_vec(),
        actions: UnsafeAction::names().to_vec(),
        post_actions: TrustedAction::names().to_vec(),
    };

    actions.pre_actions.sort();
    actions.actions.sort();
    actions.post_actions.sort();

    print!("{}", serde_yaml::to_string(&actions).unwrap());
}

#[derive(Debug, thiserror::Error)]
pub enum RunError {
    #[error(transparent)]
    Project(#[from] ProjectError),

    #[error(transparent)]
    Util(#[from] UtilError),

    #[error("failed to create temporary directory for running CI build")]
    TempDir(#[source] std::io::Error),

    #[error("failed to read ikikwiki setup file {0}")]
    ReadSetup(PathBuf, #[source] std::io::Error),

    #[error("failed to parse ikiwiki setup file as YAML: {0}")]
    SetupYaml(PathBuf, #[source] serde_yaml::Error),

    #[error("failed to figure out parent directory of {0}")]
    Parent(PathBuf),

    #[error("failed to create a general state directory")]
    MkdirState(PathBuf, #[source] UtilError),

    #[error("failed to create a project state directory")]
    MkdirProjectState(PathBuf, #[source] UtilError),

    #[error("failed to create a project state sub-directory")]
    MkdirProjectSubState(PathBuf, #[source] UtilError),

    #[error("failed to create a temporary directory for action runner")]
    MkdirProjectRunCi(#[source] UtilError),

    #[error("failed to make filename absolute: {0}")]
    Canonicalize(PathBuf, #[source] std::io::Error),

    #[error("failed to run ambient-run")]
    AmbientRunInvoke(#[source] std::io::Error),

    #[error("ambient-run failed:\n{0}")]
    AmbientRun(String),

    #[error("ambient-run didn't create the artifact tar archive {0}")]
    NoArtifact(PathBuf),

    #[error("failed to create temporary directory for unpacking artifacts")]
    CreateStaging(PathBuf, #[source] std::io::Error),

    #[error("failed to open artifact tar archive {0}")]
    OpenArtifacts(PathBuf, #[source] std::io::Error),

    #[error("failed to extract artifact archive {0} to {0}")]
    Extract(PathBuf, PathBuf, #[source] std::io::Error),

    #[error(transparent)]
    Qemu(#[from] QemuError),

    #[error("QEMU failed")]
    QemuFailed(i32),

    #[error("failed to copy {0} to {1}")]
    Copy(PathBuf, PathBuf, #[source] std::io::Error),

    #[error(transparent)]
    Plan(#[from] PlanError),

    #[error(transparent)]
    Action(#[from] ActionError),

    #[error(transparent)]
    Git(#[from] GitError),

    #[error(transparent)]
    VDrive(#[from] VirtualDriveError),

    #[error("no state directory specified")]
    NoState,

    #[error("failed to parse CI plan file as YAML: {0}")]
    PlanParse(PathBuf, #[source] serde_yaml::Error),

    #[error("failed to serialize CI plan as YAML")]
    PlanSerialize(#[source] serde_yaml::Error),

    #[error("failed to write CI plan file: {0}")]
    PlanWrite(PathBuf, #[source] std::io::Error),
}

#[derive(Debug, Default)]
struct Decision {
    dry_run: Option<bool>,
    force_run: Option<bool>,
    is_git: Option<bool>,
    latest_commit: Option<String>,
    current_commit: Option<String>,
    source_is_dirty: Option<bool>,
}

impl Decision {
    fn dry_run(&mut self) {
        self.dry_run = Some(true);
    }

    fn no_dry_run(&mut self) {
        self.dry_run = Some(false);
    }

    fn force(&mut self) {
        self.force_run = Some(true);
    }

    fn no_force(&mut self) {
        self.force_run = Some(false);
    }

    fn is_git(&mut self) {
        self.is_git = Some(true);
    }

    fn is_not_git(&mut self) {
        self.is_git = Some(false);
    }

    fn latest_commit(&mut self, commit: &str) {
        self.latest_commit = Some(commit.into());
    }

    fn current_commit(&mut self, commit: &str) {
        self.current_commit = Some(commit.into());
    }

    fn is_clean(&mut self) {
        self.source_is_dirty = Some(false);
    }

    fn is_dirty(&mut self) {
        self.source_is_dirty = Some(true);
    }

    fn should_run(&self) -> ShouldRun {
        let dry_run = self.dry_run.unwrap();
        let force = self.force_run.unwrap();
        let is_git = self.is_git.unwrap();
        let dirty = self.source_is_dirty.unwrap();

        if dry_run {
            Self::log("dry run");
            ShouldRun::DontRun
        } else if force {
            Self::log("force");
            ShouldRun::Run
        } else if !is_git {
            Self::log("not git");
            ShouldRun::Run
        } else if dirty {
            Self::log("dirty");
            ShouldRun::Run
        } else if self.current_commit == self.latest_commit {
            Self::log("commits are equal");
            ShouldRun::DontRun
        } else {
            Self::log("nothing prevents run");
            ShouldRun::Run
        }
    }

    #[allow(unused_variables)]
    fn log(msg: &str) {
        #[cfg(test)]
        println!("{}", msg);
    }
}

// Use a custom enum to avoid confusing reader with interpreting
// boolean values, specially in tests.
#[derive(Debug, Eq, PartialEq, Copy, Clone)]
enum ShouldRun {
    Run,
    DontRun,
}

#[cfg(test)]
mod test_run_decision {
    use super::{Decision, ShouldRun};

    #[test]
    fn is_not_git() {
        let mut d = Decision::default();
        d.no_dry_run();
        d.no_force();
        d.is_not_git();
        d.is_clean();
        assert_eq!(d.should_run(), ShouldRun::Run);
    }

    #[test]
    fn unchanged() {
        let mut d = Decision::default();
        d.no_dry_run();
        d.no_force();
        d.is_git();
        d.is_clean();
        d.latest_commit("abcd");
        d.current_commit("abcd");
        assert_eq!(d.should_run(), ShouldRun::DontRun);
    }

    #[test]
    fn unchanged_with_force() {
        let mut d = Decision::default();
        d.no_dry_run();
        d.force();
        d.is_git();
        d.is_clean();
        d.latest_commit("abcd");
        d.current_commit("abcd");
        assert_eq!(d.should_run(), ShouldRun::Run);
    }

    #[test]
    fn unchanged_commit_but_dirty() {
        let mut d = Decision::default();
        d.no_dry_run();
        d.no_force();
        d.is_git();
        d.is_dirty();
        d.latest_commit("abcd");
        d.current_commit("abcd");
        assert_eq!(d.should_run(), ShouldRun::Run);
    }

    #[test]
    fn commit_changed() {
        let mut d = Decision::default();
        d.no_dry_run();
        d.no_force();
        d.is_git();
        d.is_clean();
        d.latest_commit("abcd");
        d.current_commit("efgh");
        assert_eq!(d.should_run(), ShouldRun::Run);
    }

    #[test]
    fn dry_run_for_unchanged() {
        let mut d = Decision::default();
        d.dry_run();
        d.no_force();
        d.is_git();
        d.is_clean();
        d.latest_commit("abcd");
        d.current_commit("abcd");
        assert_eq!(d.should_run(), ShouldRun::DontRun);
    }

    #[test]
    fn dry_run_for_unchanged_but_dirty() {
        let mut d = Decision::default();
        d.dry_run();
        d.no_force();
        d.is_git();
        d.is_dirty();
        d.latest_commit("abcd");
        d.current_commit("efgh");
        assert_eq!(d.should_run(), ShouldRun::DontRun);
    }

    #[test]
    fn dry_run_for_commit_changed() {
        let mut d = Decision::default();
        d.dry_run();
        d.no_force();
        d.is_git();
        d.is_clean();
        d.latest_commit("abcd");
        d.current_commit("efgh");
        assert_eq!(d.should_run(), ShouldRun::DontRun);
    }

    #[test]
    fn dry_run_for_unchanged_with_force() {
        let mut d = Decision::default();
        d.dry_run();
        d.no_force();
        d.is_git();
        d.is_clean();
        d.latest_commit("abcd");
        d.current_commit("abcd");
        assert_eq!(d.should_run(), ShouldRun::DontRun);
    }

    #[test]
    fn dry_run_for_unchanged_but_dirty_with_force() {
        let mut d = Decision::default();
        d.dry_run();
        d.no_force();
        d.is_git();
        d.is_dirty();
        d.latest_commit("abcd");
        d.current_commit("efgh");
        assert_eq!(d.should_run(), ShouldRun::DontRun);
    }

    #[test]
    fn dry_run_for_commit_changed_with_force() {
        let mut d = Decision::default();
        d.dry_run();
        d.no_force();
        d.is_git();
        d.is_clean();
        d.latest_commit("abcd");
        d.current_commit("efgh");
        assert_eq!(d.should_run(), ShouldRun::DontRun);
    }
}
