//! Virtual drive handling for ambient-run.

use std::{
    fs::File,
    path::{Path, PathBuf},
};

use log::{debug, trace};

/// A virtual drive.
#[derive(Debug)]
pub struct VirtualDrive {
    filename: PathBuf,
}

impl VirtualDrive {
    /// Path to file containing virtual drive.
    pub fn filename(&self) -> &Path {
        self.filename.as_path()
    }

    /// List files in the virtual drive.
    pub fn list(&self) -> Result<Vec<PathBuf>, VirtualDriveError> {
        let file = File::open(&self.filename)
            .map_err(|e| VirtualDriveError::Open(self.filename.clone(), e))?;
        let mut archive = tar::Archive::new(file);
        let entries = archive
            .entries()
            .map_err(|e| VirtualDriveError::List(self.filename.clone(), e))?;
        let mut filenames = vec![];
        for maybe_entry in entries {
            let entry =
                maybe_entry.map_err(|e| VirtualDriveError::List(self.filename.clone(), e))?;
            let path = entry
                .path()
                .map_err(|e| VirtualDriveError::List(self.filename.clone(), e))?;
            let path = path.to_path_buf();
            filenames.push(path);
        }
        Ok(filenames)
    }

    /// Extract files in the virtual drive into a directory. Create
    /// the directory if it doesn't exist.
    pub fn extract_to(&self, dirname: &Path) -> Result<(), VirtualDriveError> {
        debug!(
            "extracting {} to {}",
            self.filename.display(),
            dirname.display()
        );
        if !dirname.exists() {
            std::fs::create_dir(dirname)
                .map_err(|e| VirtualDriveError::Extract(dirname.into(), e))?;
        }
        let file = File::open(&self.filename)
            .map_err(|e| VirtualDriveError::Open(self.filename.clone(), e))?;
        let mut archive = tar::Archive::new(file);
        let entries = archive
            .entries()
            .map_err(|e| VirtualDriveError::Open(self.filename.clone(), e))?;
        for entry in entries {
            let mut entry = entry.map_err(|e| {
                VirtualDriveError::ExtractEntry(self.filename.clone(), dirname.into(), e)
            })?;
            let path = entry.path().map_err(|e| {
                VirtualDriveError::ExtractEntry(self.filename.clone(), dirname.into(), e)
            })?;
            debug!("extract {}", path.display());
            entry.unpack_in(dirname).map_err(|e| {
                VirtualDriveError::ExtractEntry(self.filename.clone(), dirname.into(), e)
            })?;
        }
        debug!("extraction OK");
        Ok(())
    }
}

/// Builder for [`VirtualDrive`].
#[derive(Debug, Default)]
pub struct VirtualDriveBuilder {
    filename: Option<PathBuf>,
    root: Option<PathBuf>,
    size: Option<u64>,
}

impl VirtualDriveBuilder {
    /// Set filename for virtual drive.
    pub fn filename(mut self, filename: &Path) -> Self {
        self.filename = Some(filename.into());
        self
    }

    /// Set directory of tree to copy into virtual drive.
    pub fn root_directory(mut self, dirname: &Path) -> Self {
        self.root = Some(dirname.into());
        self
    }

    /// Set size of new drive. This is important when the build VM
    /// writes to the drive.
    pub fn size(mut self, size: u64) -> Self {
        self.size = Some(size);
        self
    }

    /// Create a virtual drive.
    pub fn create(self, extra: Option<(&Path, &str)>) -> Result<VirtualDrive, VirtualDriveError> {
        trace!("creating virtual drive (tar archive): {:#?}", self);

        let filename = self.filename.expect("filename has been set");
        trace!(
            "tar archive to be created: {}; exists? {}",
            filename.display(),
            filename.exists()
        );

        trace!("create archive file {}", filename.display());
        let archive =
            File::create(&filename).map_err(|e| VirtualDriveError::Create(filename.clone(), e))?;

        if let Some(size) = self.size {
            trace!("restrict file {} to {} bytes", filename.display(), size);
            archive
                .set_len(size)
                .map_err(|e| VirtualDriveError::Create(filename.clone(), e))?;
        }

        let mut builder = tar::Builder::new(archive);
        if let Some(root) = self.root {
            trace!("directory {} exists? {}", root.display(), root.exists());
            trace!("add contents of {} as .", root.display());
            builder
                .append_dir_all(".", &root)
                .map_err(|e| VirtualDriveError::CreateTar(root.clone(), e))?;
            trace!("added {} to archive", root.display());
        }

        if let Some((path, name)) = extra {
            trace!("add extra file {} as {}", path.display(), name);
            builder
                .append_path_with_name(path, name)
                .map_err(|e| VirtualDriveError::CreateTar(path.to_path_buf(), e))?;
        }

        trace!("finish creating {}", filename.display());
        builder
            .finish()
            .map_err(|e| VirtualDriveError::CreateTar(filename.clone(), e))?;

        debug!("created virtual drive {}", filename.display());
        Ok(VirtualDrive { filename })
    }

    /// Open an existing virtual drive.
    pub fn open(self) -> Result<VirtualDrive, VirtualDriveError> {
        let filename = self.filename.expect("filename has been set");
        Ok(VirtualDrive { filename })
    }
}

/// Errors that may be returned from [`VirtualDrive`] use.
#[allow(missing_docs)]
#[derive(Debug, thiserror::Error)]
pub enum VirtualDriveError {
    #[error("failed to create virtual drive {0}")]
    Create(PathBuf, #[source] std::io::Error),

    #[error("failed to create tar archive for virtual drive from {0}")]
    CreateTar(PathBuf, #[source] std::io::Error),

    #[error("failed to open virtual drive {0}")]
    Open(PathBuf, #[source] std::io::Error),

    #[error("failed to list files in virtual drive {0}")]
    List(PathBuf, #[source] std::io::Error),

    #[error("failed to create directory {0}")]
    Extract(PathBuf, #[source] std::io::Error),

    #[error("failed to extract {0} to {1}")]
    ExtractEntry(PathBuf, PathBuf, #[source] std::io::Error),
}
