use std::{
    collections::HashMap,
    fs::{read, write},
    path::{Path, PathBuf},
};

use log::debug;
use serde::{Deserialize, Serialize};

use crate::{
    action::{TrustedAction, UnsafeAction},
    util::expand_tilde,
};

#[derive(Debug, Deserialize, Clone)]
#[serde(deny_unknown_fields)]
pub struct Projects {
    projects: HashMap<String, Project>,
}

impl Projects {
    pub fn from_file(filename: &Path) -> Result<Self, ProjectError> {
        let dirname = if let Some(parent) = filename.parent() {
            parent.to_path_buf()
        } else {
            return Err(ProjectError::Parent(filename.into()));
        };

        let yaml = read(filename).map_err(|e| ProjectError::Read(filename.into(), e))?;
        let mut projects: Self =
            serde_yaml::from_slice(&yaml).map_err(|e| ProjectError::Yaml(filename.into(), e))?;

        for (_, p) in projects.projects.iter_mut() {
            p.expand_tilde(&dirname)?;
        }

        debug!("projects from file {}: {:#?}", filename.display(), projects);
        Ok(projects)
    }

    pub fn iter(&self) -> impl Iterator<Item = (&str, &Project)> {
        self.projects.iter().map(|(k, v)| (k.as_str(), v))
    }
}

#[derive(Debug, Deserialize, Clone)]
#[serde(deny_unknown_fields)]
pub struct Project {
    source: PathBuf,
    image: PathBuf,
    pre_plan: Option<Vec<TrustedAction>>,
    plan: Option<Vec<UnsafeAction>>,
    post_plan: Option<Vec<TrustedAction>>,
    artifact_max_size: Option<u64>,
    cache_max_size: Option<u64>,
}

impl Project {
    fn expand_tilde(&mut self, basedir: &Path) -> Result<(), ProjectError> {
        self.source = Self::abspath(basedir.join(expand_tilde(&self.source)))?;
        self.image = Self::abspath(basedir.join(expand_tilde(&self.image)))?;
        Ok(())
    }

    pub fn from_file(filename: &Path) -> Result<Self, ProjectError> {
        let dirname = if let Some(parent) = filename.parent() {
            parent.to_path_buf()
        } else {
            return Err(ProjectError::Parent(filename.into()));
        };

        let yaml = read(filename).map_err(|e| ProjectError::Read(filename.into(), e))?;
        let mut project: Project =
            serde_yaml::from_slice(&yaml).map_err(|e| ProjectError::Yaml(filename.into(), e))?;

        project.expand_tilde(&dirname)?;

        debug!("project from file {}: {:#?}", filename.display(), project);
        Ok(project)
    }

    fn abspath(path: PathBuf) -> Result<PathBuf, ProjectError> {
        path.canonicalize()
            .map_err(|e| ProjectError::Canonicalize(path, e))
    }

    pub fn source(&self) -> &Path {
        &self.source
    }

    pub fn image(&self) -> &Path {
        &self.image
    }

    pub fn artifact_max_size(&self) -> Option<u64> {
        self.artifact_max_size
    }

    pub fn cache_max_size(&self) -> Option<u64> {
        self.cache_max_size
    }

    pub fn pre_plan(&self) -> &[TrustedAction] {
        if let Some(plan) = &self.pre_plan {
            plan.as_slice()
        } else {
            &[]
        }
    }

    pub fn plan(&self) -> &[UnsafeAction] {
        if let Some(plan) = &self.plan {
            plan.as_slice()
        } else {
            &[]
        }
    }

    pub fn post_plan(&self) -> &[TrustedAction] {
        if let Some(plan) = &self.post_plan {
            plan.as_slice()
        } else {
            &[]
        }
    }
}

/// Persistent project state.
#[derive(Debug, Clone, Deserialize, Serialize)]
#[allow(dead_code)]
pub struct State {
    // File where this state is stored, if it's stored.
    #[serde(skip)]
    filename: PathBuf,

    // Where persistent state is stored for this project.
    #[serde(skip)]
    statedir: PathBuf,

    /// Latest commit that CI has run on, if any.
    latest_commit: Option<String>,
}

impl State {
    /// Load state for a project from a file, if it's present. If it's
    /// not present, return an empty state.
    pub fn from_file(statedir: &Path, project: &str) -> Result<Self, ProjectError> {
        let statedir = statedir.join(project);
        let filename = statedir.join("meta.yaml");
        debug!("load project state from {}", filename.display());
        if filename.exists() {
            let yaml = read(&filename).map_err(|e| ProjectError::ReadState(filename.clone(), e))?;
            let mut state: Self = serde_yaml::from_slice(&yaml)
                .map_err(|e| ProjectError::ParseState(filename.clone(), e))?;
            state.filename = filename;
            state.statedir = statedir;
            Ok(state)
        } else {
            Ok(Self {
                filename,
                statedir,
                latest_commit: None,
            })
        }
    }

    /// Write project state.
    pub fn write_to_file(&self) -> Result<(), ProjectError> {
        debug!("write project state to {}", self.filename.display());
        let yaml = serde_yaml::to_string(&self)
            .map_err(|e| ProjectError::SerializeState(self.clone(), e))?;
        if !self.statedir.exists() {
            std::fs::create_dir(&self.statedir)
                .map_err(|e| ProjectError::CreateState(self.statedir.clone(), e))?;
        }
        write(&self.filename, yaml)
            .map_err(|e| ProjectError::WriteState(self.filename.clone(), e))?;
        Ok(())
    }

    /// Return state directory.
    pub fn statedir(&self) -> &Path {
        &self.statedir
    }

    /// Return artifacts directory for project.
    pub fn artifactsdir(&self) -> PathBuf {
        self.statedir.join("artifacts")
    }

    /// Return cache directory for project.
    pub fn cachedir(&self) -> PathBuf {
        self.statedir.join("cache")
    }

    /// Return dependencies directory for a project.
    pub fn dependenciesdir(&self) -> PathBuf {
        self.statedir.join("dependencies")
    }

    /// Return latest commit that CI has run on.
    pub fn latest_commit(&self) -> Option<&str> {
        self.latest_commit.as_deref()
    }

    /// Set latest commit.
    pub fn set_latest_commot(&mut self, commit: Option<&str>) {
        self.latest_commit = commit.map(|s| s.into());
    }
}

#[derive(Debug, thiserror::Error)]
pub enum ProjectError {
    #[error("failed to determine directory containing project file {0}")]
    Parent(PathBuf),

    #[error("failed to make filename absolute: {0}")]
    Canonicalize(PathBuf, #[source] std::io::Error),

    #[error("failed top read project file {0}")]
    Read(PathBuf, #[source] std::io::Error),

    #[error("failed to parse project file as YAML: {0}")]
    Yaml(PathBuf, #[source] serde_yaml::Error),

    #[error("failed to serialize project state as YAML: {0:#?}")]
    SerializeState(State, #[source] serde_yaml::Error),

    #[error("failed to write project state to file {0}")]
    WriteState(PathBuf, #[source] std::io::Error),

    #[error("failed to read project state from file {0}")]
    ReadState(PathBuf, #[source] std::io::Error),

    #[error("failed to parse project state file as YAML: {0}")]
    ParseState(PathBuf, #[source] serde_yaml::Error),

    #[error("failed to create project state directory {0}")]
    CreateState(PathBuf, #[source] std::io::Error),
}
