use std::{
    fs::create_dir_all,
    path::{Path, PathBuf},
    process::Command,
};

use log::{debug, info};
use walkdir::WalkDir;

pub fn expand_tilde(path: &Path) -> PathBuf {
    if path.starts_with("~/") {
        if let Some(home) = std::env::var_os("HOME") {
            let mut expanded = PathBuf::from(home);
            for comp in path.components().skip(1) {
                expanded.push(comp);
            }
            expanded
        } else {
            path.to_path_buf()
        }
    } else {
        path.to_path_buf()
    }
}

pub fn mkdir(dirname: &Path) -> Result<(), UtilError> {
    create_dir_all(dirname).map_err(|e| UtilError::CreateDir(dirname.into(), e))?;
    Ok(())
}

pub fn mkdir_child(parent: &Path, subdir: &str) -> Result<PathBuf, UtilError> {
    let pathname = parent.join(subdir);
    create_dir_all(&pathname).map_err(|e| UtilError::CreateDir(pathname.clone(), e))?;
    Ok(pathname)
}

pub fn recreate_dir(dirname: &Path) -> Result<(), UtilError> {
    if dirname.exists() {
        std::fs::remove_dir_all(dirname).map_err(|e| UtilError::RemoveDir(dirname.into(), e))?;
    }
    mkdir(dirname)?;
    Ok(())
}

pub fn rsync_dir(src: &Path, dest: &Path) -> Result<(), UtilError> {
    let src = src.join(".");
    let dest = dest.join(".");
    info!("rsync {} -> {}", src.display(), dest.display());
    let output = Command::new("rsync")
        .arg("-a")
        .arg("--del")
        .arg(&src)
        .arg(&dest)
        .output()
        .map_err(|e| UtilError::RsyncInvoke(src.clone(), dest.display().to_string(), e))?;

    if output.status.code() != Some(0) {
        let stderr = String::from_utf8_lossy(&output.stderr);
        return Err(UtilError::Rsync(
            src.clone(),
            dest.display().to_string(),
            stderr.into(),
        ));
    }

    Ok(())
}

pub fn rsync_server(src: &Path, target: &str) -> Result<(), UtilError> {
    let src = src.join(".");
    let target = format!("{}/.", target);
    info!("rsync {} -> {}", src.display(), target);
    let output = Command::new("rsync")
        .arg("-a")
        .arg("--del")
        .arg(&src)
        .arg(&target)
        .output()
        .map_err(|e| UtilError::RsyncInvoke(src.clone(), target.clone(), e))?;

    if output.status.code() != Some(0) {
        let stderr = String::from_utf8_lossy(&output.stderr);
        return Err(UtilError::Rsync(src.clone(), target.clone(), stderr.into()));
    }

    Ok(())
}

pub fn dput(dest: &str, changes: &Path) -> Result<(), UtilError> {
    info!("dput {} {}", dest, changes.display());
    let output = Command::new("dput")
        .arg(dest)
        .arg(changes)
        .output()
        .map_err(|e| UtilError::DputInvoke(dest.into(), changes.to_path_buf(), e))?;

    if output.status.code() != Some(0) {
        let stderr = String::from_utf8_lossy(&output.stderr);
        return Err(UtilError::Dput(
            dest.into(),
            changes.to_path_buf(),
            stderr.into(),
        ));
    }

    Ok(())
}

pub fn changes_file(dir: &Path) -> Result<PathBuf, UtilError> {
    let mut result = None;
    for entry in WalkDir::new(dir).into_iter() {
        let path = entry.map_err(UtilError::WalkDir)?.path().to_path_buf();
        debug!("found {}", path.display());
        if path.display().to_string().ends_with(".changes") {
            if result.is_some() {
                return Err(UtilError::ManyChanges);
            }
            result = Some(path);
        }
    }

    if let Some(path) = result {
        Ok(path)
    } else {
        Err(UtilError::NoChanges)
    }
}

#[derive(Debug, thiserror::Error)]
pub enum UtilError {
    #[error("failed to create directory {0}")]
    CreateDir(PathBuf, #[source] std::io::Error),

    #[error("failed to run ambient-run")]
    AmbientRunInvoke(#[source] std::io::Error),

    #[error("ambient-run failed:\n{0}")]
    AmbientRun(String),

    #[error("failed to invoke dput to upload {1} to {0}")]
    DputInvoke(String, PathBuf, #[source] std::io::Error),

    #[error("dput failed to upload {1} to {0}:\n{2}")]
    Dput(String, PathBuf, String),

    #[error("failed to invoke rsync to synchronize {0} to {1}")]
    RsyncInvoke(PathBuf, String, #[source] std::io::Error),

    #[error("rsync failed to synchronize {0} to {1}:\n{2}")]
    Rsync(PathBuf, String, String),

    #[error("failed to create project YAML file {0}")]
    CreateProject(PathBuf, #[source] std::io::Error),

    #[error("failed to generate project YAML {0}")]
    Yaml(PathBuf, #[source] serde_yaml::Error),

    #[error("ambient-run didn't create the artifact tar archive {0}")]
    NoArtifact(PathBuf),

    #[error("failed to open artifact tar archive {0}")]
    OpenArtifacts(PathBuf, #[source] std::io::Error),

    #[error("failed to extract artifact archive {0} to {0}")]
    ExtractArtifacts(PathBuf, PathBuf, #[source] std::io::Error),
    //
    #[error("failed to list contents of upload directory")]
    WalkDir(#[source] walkdir::Error),

    #[error("no *.changes file built for deb project")]
    NoChanges,

    #[error("more than one *.changes file built for deb project")]
    ManyChanges,

    #[error("failed to remove directory {0}")]
    RemoveDir(PathBuf, #[source] std::io::Error),
}
