use std::path::{Path, PathBuf};

use clap::Parser;
use log::debug;

use ambient_driver::{
    action::UnsafeAction,
    cloud_init::{CloudInitError, LocalDataStoreBuilder},
    plan::RunnablePlan,
    qemu::{self, QemuError, QemuRunner},
    run::{create_cloud_init_iso, create_executor_vdrive, create_source_vdrive},
    util::{cat_text_file, mkdir, UtilError},
    vdrive::VirtualDriveError,
};
use tempfile::tempdir;

use super::{AmbientDriverError, Config, Leaf};

const SECRET: &str = "xyzzy";

/// Verify that a virtual machine image is acceptable for use with
/// ambient-driver. This is done by booting a VM using the image, and
/// making sure the VM run the provided executable and then shuts down.
///
/// If the verification doesn't finish within a reasonable time, the
/// image is not acceptable.
#[derive(Debug, Parser)]
pub struct VerifyImage {
    /// The virtual machine image to test.
    #[clap(long)]
    filename: PathBuf,

    /// Location of the ambient-execute-plan binary.
    #[clap(long)]
    executor: PathBuf,

    /// Run log goes to this file.
    #[clap(long)]
    run_log: Option<PathBuf>,

    /// Console log goes to this file.
    #[clap(long)]
    console_log: Option<PathBuf>,
}

impl Leaf for VerifyImage {
    fn run(&self, config: &Config) -> Result<(), AmbientDriverError> {
        debug!("verify image {}", self.filename.display());

        let image = self
            .filename
            .canonicalize()
            .map_err(|err| ImageError::Canonicalize(self.filename.clone(), err))?;
        if image != self.filename {
            debug!("using canonical path name for image: {}", image.display());
        }

        if image.exists() {
            debug!("image {} exists, good", image.display());
        } else {
            return Err(ImageError::NoSuchFile(image.clone()).into());
        }

        let tmp = tempdir().map_err(ImageError::TempDir)?;
        let src = tmp.path().join("src");
        mkdir(&src).map_err(ImageError::Util)?;

        debug!("create executable drive");
        let mut plan = RunnablePlan::default();
        plan.push_unsafe_actions(
            [
                UnsafeAction::mkdir(Path::new(qemu::WORKSPACE_DIR)),
                UnsafeAction::mkdir(Path::new(qemu::SOURCE_DIR)),
                UnsafeAction::shell(&format!("echo {SECRET}")),
            ]
            .iter(),
        );
        plan.set_executor_drive(qemu::EXECUTOR_DRIVE);
        plan.set_source_drive(qemu::SOURCE_DRIVE);
        plan.set_source_dir(qemu::SOURCE_DIR);
        let executor_drive =
            create_executor_vdrive(&tmp, &plan, &self.executor).map_err(ImageError::CreateDrive)?;
        let source_drive = create_source_vdrive(&tmp, &src).map_err(ImageError::CreateDrive)?;

        let run_log = self
            .run_log
            .clone()
            .unwrap_or_else(|| tmp.path().join("run.log"));
        let console_log = self
            .console_log
            .clone()
            .unwrap_or_else(|| tmp.path().join("console.log"));

        let ds = create_cloud_init_iso().map_err(ImageError::CreateDrive)?;

        debug!("run QEMU with image {}", image.display());
        let res = QemuRunner::default()
            .config(config)
            .image(&image)
            .excutor(&executor_drive)
            .source(&source_drive)
            .cloud_init(&ds)
            .console_log(&console_log)
            .run_log(&run_log)
            .run()
            .map_err(ImageError::Qemu);
        debug!("QEMU finished: {res:?}");

        let log = cat_text_file(&run_log).map_err(ImageError::Util)?;
        if !log.contains(SECRET) {
            return Err(ImageError::NotAcceptable(image.clone()).into());
        }
        debug!("run log contains expected string {SECRET:?}");
        println!("image {} is acceptable to Ambient", image.display());

        Ok(res?)
    }
}

/// Create a cloud-init local data store ISO file.
#[derive(Debug, Parser)]
pub struct CloudInit {
    /// The name of the ISO file.
    #[clap(long)]
    filename: PathBuf,

    /// Commands to run via cloud-init.
    #[clap(long)]
    runcmd: Vec<String>,
}

impl Leaf for CloudInit {
    fn run(&self, _config: &Config) -> Result<(), AmbientDriverError> {
        let mut ds = LocalDataStoreBuilder::default()
            .with_hostname("ambient")
            .with_bootcmd("echo xyzzy2")
            .with_bootcmd("echo more bootcmd");
        for runcmd in self.runcmd.iter() {
            ds = ds.with_runcmd(runcmd);
        }
        let ds = ds.build().map_err(ImageError::CloudInit)?;
        ds.iso(&self.filename).map_err(ImageError::CloudInit)?;
        Ok(())
    }
}

#[derive(Debug, thiserror::Error)]
pub enum ImageError {
    #[error(transparent)]
    Qemu(#[from] QemuError),

    #[error("failed to create a temporary directory")]
    TempDir(#[source] std::io::Error),

    #[error(transparent)]
    Util(#[from] UtilError),

    #[error(transparent)]
    VDrive(#[from] VirtualDriveError),

    #[error(transparent)]
    CloudInit(#[from] CloudInitError),

    #[error("image {0} does not exist")]
    NoSuchFile(PathBuf),

    #[error("image {0} is NOT an acceptable image for Ambient")]
    NotAcceptable(PathBuf),

    #[error("failed to make image filename into a canonical path: {0}")]
    Canonicalize(PathBuf, #[source] std::io::Error),

    #[error(transparent)]
    CreateDrive(#[from] ambient_driver::run::RunError),
}
