use std::path::PathBuf;

use clap::Parser;

use super::{AmbientDriverError, Config, Leaf};
use ambient_driver::project::{ProjectError, State};

/// Show available actions to the user.
#[derive(Debug, Parser)]
pub struct Log {
    /// Which project log should be written out?
    project: String,
}

impl Leaf for Log {
    fn run(&self, config: &Config) -> Result<(), AmbientDriverError> {
        let statedir = config.state();

        if !statedir.exists() {
            return Err(LogError::NoStateDir(self.project.clone(), statedir.into()))?;
        }

        let state = State::from_file(statedir, &self.project).map_err(LogError::Project)?;
        let data = state.read_run_log().map_err(LogError::Project)?;
        let log = String::from_utf8_lossy(&data).to_string();

        print!("{log}");

        Ok(())
    }
}

#[derive(Debug, thiserror::Error)]
pub enum LogError {
    #[error("state directory for project {0} does not exist: {1}")]
    NoStateDir(String, PathBuf),

    #[error(transparent)]
    Project(#[from] ProjectError),
}
