use std::path::PathBuf;

use clap::Parser;

use super::{AmbientDriverError, Config, Leaf};
use ambient_driver::run::cmd_run;

/// Run CI on projects.
#[derive(Debug, Parser)]
pub struct Run {
    /// Project specification file. May contain any number of projects.
    #[clap(long)]
    projects: Option<PathBuf>,

    /// Which projects to run CI for, from the ones in the PROJECTS
    /// file. Default is all of them.
    chosen: Option<Vec<String>>,

    /// Build log.
    #[clap(long)]
    log: Option<PathBuf>,

    /// rsync target for publishing ikiwiki output.
    #[clap(long, alias = "rsync")]
    target: Option<String>,

    /// dput target for publishing .deb package.
    #[clap(long, alias = "dput")]
    dput_target: Option<String>,

    /// Path to `ambient-execute-plan` binary to use to run CI in VM.
    #[clap(long)]
    executor: Option<PathBuf>,

    /// Only pretend to run CI.
    #[clap(long)]
    dry_run: bool,

    /// Run even if repository hasn't changed.
    #[clap(long)]
    force: bool,
}

impl Leaf for Run {
    fn run(&self, config: &Config) -> Result<(), AmbientDriverError> {
        let executor = if let Some(executor) = &self.executor {
            executor
        } else if let Some(executor) = config.executor() {
            executor
        } else {
            return Err(RunError::NoRunCi)?;
        };

        let rsync_target = if let Some(s) = &self.target {
            Some(s.as_str())
        } else {
            config.target()
        };

        let dput_target = if let Some(s) = &self.dput_target {
            Some(s.as_str())
        } else {
            config.dput_target()
        };

        let projects = if let Some(filename) = &self.projects {
            filename.to_path_buf()
        } else {
            config.projects().into()
        };

        cmd_run(
            config,
            &projects,
            self.chosen.as_deref(),
            self.dry_run,
            self.force,
            rsync_target,
            dput_target,
            executor,
        )
        .map_err(RunError::Run)?;

        Ok(())
    }
}

#[derive(Debug, thiserror::Error)]
pub enum RunError {
    #[error(transparent)]
    Run(#[from] ambient_driver::run::RunError),

    #[error("you must set path to ambient-execute-plan program with option or in configuration")]
    NoRunCi,
}
