use std::{
    fs::File,
    path::{Path, PathBuf},
};

use byte_unit::Byte;
use directories::ProjectDirs;
use log::trace;
use serde::{Deserialize, Serialize};

use crate::tildepathbuf::TildePathBuf;

const QUAL: &str = "liw.fi";
const ORG: &str = "Ambient CI";
const APP: &str = env!("CARGO_PKG_NAME");

#[derive(Debug, Deserialize)]
#[serde(deny_unknown_fields, default)]
struct StoredConfig {
    tmpdir: Option<TildePathBuf>,
    projects: Option<TildePathBuf>,
    state: Option<TildePathBuf>,
    target: Option<String>,
    dput_target: Option<String>,
    executor: Option<TildePathBuf>,
    cpus: Option<usize>,
    memory: Option<Byte>,
    artifacts_max_size: Option<Byte>,
    cache_max_size: Option<Byte>,
}

impl Default for StoredConfig {
    fn default() -> Self {
        let dirs = ProjectDirs::from(QUAL, ORG, APP).expect("have home directory");
        Self {
            tmpdir: Some(PathBuf::from("/tmp").into()),
            projects: Some(dirs.config_dir().join("projects.yaml").into()),
            state: dirs.state_dir().map(|x| x.into()),
            target: None,
            dput_target: None,
            executor: None,
            cpus: Some(1),
            memory: Some(Byte::GIBIBYTE),
            artifacts_max_size: Byte::MEBIBYTE.multiply(10),
            cache_max_size: Byte::GIBIBYTE.multiply(10),
        }
    }
}

#[derive(Debug, Serialize, Deserialize)]
#[serde(deny_unknown_fields)]
pub struct Config {
    tmpdir: PathBuf,
    projects: PathBuf,
    state: PathBuf,
    target: Option<String>,
    dput_target: Option<String>,
    executor: Option<PathBuf>,
    cpus: usize,
    memory: Byte,
    artifacts_max_size: Byte,
    cache_max_size: Byte,
}

impl Config {
    pub fn new() -> Result<Self, ConfigError> {
        Self::try_from(StoredConfig::default())
    }

    pub fn from_yaml_file(filename: &Path) -> Result<Self, ConfigError> {
        let f = File::open(filename).map_err(|e| ConfigError::Read(filename.into(), e))?;
        let stored: StoredConfig =
            serde_yml::from_reader(f).map_err(|e| ConfigError::Yaml(filename.into(), e))?;
        let config = Config::try_from(stored)?;
        trace!(
            "Loaded configuration from file {}:\n{config:#?}",
            filename.display()
        );
        Ok(config)
    }

    pub fn tmpdir(&self) -> &Path {
        &self.tmpdir
    }

    pub fn projects(&self) -> &Path {
        &self.projects
    }

    pub fn state(&self) -> &Path {
        &self.state
    }

    pub fn target(&self) -> Option<&str> {
        self.target.as_deref()
    }

    pub fn dput_target(&self) -> Option<&str> {
        self.dput_target.as_deref()
    }

    pub fn executor(&self) -> Option<&Path> {
        self.executor.as_deref()
    }

    pub fn cpus(&self) -> usize {
        self.cpus
    }

    pub fn memory(&self) -> Byte {
        self.memory
    }

    pub fn artifacts_max_size(&self) -> u64 {
        self.artifacts_max_size.as_u64()
    }

    pub fn cache_max_size(&self) -> u64 {
        self.cache_max_size.as_u64()
    }
}

impl TryFrom<StoredConfig> for Config {
    type Error = ConfigError;
    fn try_from(stored: StoredConfig) -> Result<Self, Self::Error> {
        Ok(Self {
            tmpdir: stored
                .tmpdir
                .ok_or(ConfigError::Missing("tmpdir"))?
                .path()
                .into(),
            projects: stored
                .projects
                .ok_or(ConfigError::Missing("projects"))?
                .path()
                .into(),
            state: stored
                .state
                .ok_or(ConfigError::Missing("state"))?
                .path()
                .into(),
            target: stored.target,
            dput_target: stored.dput_target,
            executor: stored.executor.map(|path| path.path().into()),
            cpus: stored.cpus.ok_or(ConfigError::Missing("cpus"))?,
            memory: stored.memory.ok_or(ConfigError::Missing("memory"))?,
            artifacts_max_size: stored
                .artifacts_max_size
                .ok_or(ConfigError::Missing("artifacts_max_size"))?,
            cache_max_size: stored
                .cache_max_size
                .ok_or(ConfigError::Missing("cache_max_size"))?,
        })
    }
}

#[derive(Debug, thiserror::Error)]
pub enum ConfigError {
    #[error("failed to find home directory, while looking for configuration file")]
    ProjectDirs,

    #[error("failed to read configuration file {0}")]
    Read(PathBuf, #[source] std::io::Error),

    #[error("failed to parse configuration file as YAML: {0}")]
    Yaml(PathBuf, #[source] serde_yml::Error),

    #[error("programming error: stored config field {0} is missing")]
    Missing(&'static str),
}
