//! Virtual drive handling for ambient-run.

use std::{
    fs::File,
    path::{Path, PathBuf},
};

use log::{debug, trace};

use crate::util::{tar_create, tar_extract};

/// A virtual drive.
#[derive(Debug, Clone)]
pub struct VirtualDrive {
    filename: PathBuf,
}

impl VirtualDrive {
    /// Path to file containing virtual drive.
    pub fn filename(&self) -> &Path {
        self.filename.as_path()
    }

    /// Extract files in the virtual drive into a directory. Create
    /// the directory if it doesn't exist.
    pub fn extract_to(&self, dirname: &Path) -> Result<(), VirtualDriveError> {
        debug!(
            "extracting {} to {}",
            self.filename.display(),
            dirname.display()
        );
        if !dirname.exists() {
            std::fs::create_dir(dirname)
                .map_err(|e| VirtualDriveError::Extract(dirname.into(), e))?;
        }
        tar_extract(&self.filename, dirname)?;
        debug!("extraction OK");
        Ok(())
    }
}

/// Builder for [`VirtualDrive`].
#[derive(Debug, Default)]
pub struct VirtualDriveBuilder {
    filename: Option<PathBuf>,
    root: Option<PathBuf>,
    size: Option<u64>,
}

impl VirtualDriveBuilder {
    /// Set filename for virtual drive.
    pub fn filename(mut self, filename: &Path) -> Self {
        self.filename = Some(filename.into());
        self
    }

    /// Set directory of tree to copy into virtual drive.
    pub fn root_directory(mut self, dirname: &Path) -> Self {
        self.root = Some(dirname.into());
        self
    }

    /// Set size of new drive. This is important when the build VM
    /// writes to the drive.
    pub fn size(mut self, size: u64) -> Self {
        self.size = Some(size);
        self
    }

    /// Create a virtual drive.
    pub fn create(self) -> Result<VirtualDrive, VirtualDriveError> {
        trace!("creating virtual drive (tar archive): {:#?}", self);

        let filename = self.filename.expect("filename has been set");
        trace!(
            "tar archive to be created: {}; exists? {}",
            filename.display(),
            filename.exists()
        );

        trace!("create archive file {}", filename.display());

        // Create the file, either empty or to the desired size. If we
        // don't have self.root set, the file created (and maybe
        // truncated) will be.
        {
            let archive = File::create(&filename)
                .map_err(|e| VirtualDriveError::Create(filename.clone(), e))?;
            if let Some(size) = self.size {
                trace!("restrict file {} to {} bytes", filename.display(), size);
                archive
                    .set_len(size)
                    .map_err(|e| VirtualDriveError::Create(filename.clone(), e))?;
            }
        }

        if let Some(root) = self.root {
            trace!("directory {} exists? {}", root.display(), root.exists());
            trace!("add contents of {} as .", root.display());
            tar_create(&filename, &root)?;
        }

        debug!("created virtual drive {}", filename.display());
        Ok(VirtualDrive { filename })
    }

    /// Open an existing virtual drive.
    pub fn open(self) -> Result<VirtualDrive, VirtualDriveError> {
        let filename = self.filename.expect("filename has been set");
        Ok(VirtualDrive { filename })
    }
}

/// Errors that may be returned from [`VirtualDrive`] use.
#[allow(missing_docs)]
#[derive(Debug, thiserror::Error)]
pub enum VirtualDriveError {
    #[error("failed to create virtual drive {0}")]
    Create(PathBuf, #[source] std::io::Error),

    #[error("failed to create tar archive for virtual drive from {0}")]
    CreateTar(PathBuf, #[source] std::io::Error),

    #[error("failed to open virtual drive {0}")]
    Open(PathBuf, #[source] std::io::Error),

    #[error("failed to list files in virtual drive {0}")]
    List(PathBuf, #[source] std::io::Error),

    #[error("failed to create directory {0}")]
    Extract(PathBuf, #[source] std::io::Error),

    #[error("failed to extract {0} to {1}")]
    ExtractEntry(PathBuf, PathBuf, #[source] std::io::Error),

    #[error(transparent)]
    Util(#[from] crate::util::UtilError),
}
