use std::{
    path::{Path, PathBuf},
    process::{Command, Stdio},
};

use log::{debug, info};
use serde::{Deserialize, Serialize};

use crate::{
    plan::RunnablePlan,
    project::{Project, State},
    qemu,
    util::{changes_file, dput, rsync_server, UtilError},
    vdrive::{VirtualDriveBuilder, VirtualDriveError},
};

#[derive(Debug, Clone, Serialize, Deserialize, Eq, PartialEq)]
#[serde(tag = "action")]
#[serde(rename_all = "snake_case")]
pub enum RunnableAction {
    Dummy,
    Pwd {
        sourcedir: PathBuf,
    },
    CargoFetch {
        sourcedir: PathBuf,
        dependenciesdir: PathBuf,
    },
    Rsync {
        artifactsdir: PathBuf,
        rsync_target: Option<String>,
    },
    Dput {
        artifactsdir: PathBuf,
        dput_target: Option<String>,
    },
    Mkdir {
        pathname: PathBuf,
    },
    TarCreate {
        archive: PathBuf,
        directory: PathBuf,
    },
    TarExtract {
        archive: PathBuf,
        directory: PathBuf,
    },
    Spawn {
        argv: Vec<String>,
    },
    Shell {
        shell: String,
    },
    CargoFmt,
    CargoClippy,
    CargoBuild,
    CargoTest,
    CargoInstall,
    Deb,
}

impl RunnableAction {
    pub fn execute(&self, plan: &RunnablePlan) -> Result<(), ActionError> {
        debug!("Plan::execute: {:#?}", self);
        match self {
            Self::Dummy => Self::dummy(),
            Self::Pwd { sourcedir } => Self::pwd(sourcedir),
            Self::CargoFetch {
                sourcedir,
                dependenciesdir,
            } => Self::cargo_fetch(sourcedir, dependenciesdir),
            Self::Rsync {
                artifactsdir,
                rsync_target,
            } => Self::rsync(artifactsdir, rsync_target.as_ref().map(|s| s.as_str())),
            Self::Dput {
                artifactsdir,
                dput_target,
            } => Self::dput_action(artifactsdir, dput_target.as_ref().map(|s| s.as_str())),
            Self::Mkdir { pathname } => Self::mkdir(pathname),
            Self::TarCreate { archive, directory } => Self::tar_create(archive, directory),
            Self::TarExtract { archive, directory } => Self::tar_extract(archive, directory),
            Self::Spawn { argv } => Self::spawn(plan, argv, &[]),
            Self::Shell { shell: snippet } => Self::shell(plan, snippet),
            Self::CargoFmt => Self::cargo_fmt(plan),
            Self::CargoClippy => Self::cargo_clippy(plan),
            Self::CargoBuild => Self::cargo_build(plan),
            Self::CargoTest => Self::cargo_test(plan),
            Self::CargoInstall => Self::cargo_install(plan),
            Self::Deb => Self::deb(plan),
        }
    }

    pub fn from_trusted_action(
        action: &TrustedAction,
        project: &Project,
        state: &State,
        rsync_target: Option<&str>,
        dput_target: Option<&str>,
    ) -> Self {
        match action {
            TrustedAction::Dummy => Self::Dummy,
            TrustedAction::Pwd => Self::Pwd {
                sourcedir: project.source().into(),
            },
            TrustedAction::CargoFetch => Self::CargoFetch {
                sourcedir: project.source().into(),
                dependenciesdir: state.dependenciesdir(),
            },
            TrustedAction::Rsync => Self::Rsync {
                artifactsdir: state.artifactsdir(),
                rsync_target: rsync_target.map(|s| s.into()),
            },
            TrustedAction::Dput => Self::Dput {
                artifactsdir: state.artifactsdir(),
                dput_target: dput_target.map(|s| s.into()),
            },
        }
    }

    pub fn from_unsafe_action(action: &UnsafeAction) -> Self {
        match action {
            UnsafeAction::Mkdir { pathname } => Self::Mkdir {
                pathname: pathname.clone(),
            },
            UnsafeAction::TarCreate { archive, directory } => Self::TarCreate {
                archive: archive.clone(),
                directory: directory.clone(),
            },
            UnsafeAction::TarExtract { archive, directory } => Self::TarExtract {
                archive: archive.clone(),
                directory: directory.clone(),
            },
            UnsafeAction::Spawn { argv } => Self::Spawn { argv: argv.clone() },
            UnsafeAction::Shell { shell } => Self::Shell {
                shell: shell.clone(),
            },
            UnsafeAction::CargoFmt => Self::CargoFmt,
            UnsafeAction::CargoClippy => Self::CargoClippy,
            UnsafeAction::CargoBuild => Self::CargoBuild,
            UnsafeAction::CargoTest => Self::CargoTest,
            UnsafeAction::CargoInstall => Self::CargoInstall,
            UnsafeAction::Deb => Self::Deb,
        }
    }

    fn dummy() -> Result<(), ActionError> {
        println!("dummy action");
        Ok(())
    }

    fn pwd(sourcedir: &Path) -> Result<(), ActionError> {
        info!("cwd: {}", sourcedir.display());
        Ok(())
    }

    fn rsync(artifactsdir: &Path, target: Option<&str>) -> Result<(), ActionError> {
        if let Some(target) = target {
            rsync_server(artifactsdir, target)?;
        } else {
            return Err(ActionError::RsyncTargetMissing);
        }
        Ok(())
    }

    fn dput_action(artifactsdir: &Path, dput_target: Option<&str>) -> Result<(), ActionError> {
        if let Some(target) = dput_target {
            let changes = changes_file(artifactsdir)?;
            dput(target, &changes)?;
        } else {
            return Err(ActionError::DputTargetMissing);
        }
        Ok(())
    }

    fn cargo_fetch(source_dir: &Path, dependencies_dir: &Path) -> Result<(), ActionError> {
        Self::spawn_str_in(
            &[
                "env",
                &format!("CARGO_HOME={}", dependencies_dir.display()),
                "cargo",
                "fetch",
                "--locked",
                "-vvv",
            ],
            source_dir,
        )
    }

    fn mkdir(pathname: &Path) -> Result<(), ActionError> {
        if !pathname.exists() {
            std::fs::create_dir(pathname).map_err(|e| ActionError::Mkdir(pathname.into(), e))?;
        }
        Ok(())
    }

    fn tar_create(archive: &Path, dirname: &Path) -> Result<(), ActionError> {
        VirtualDriveBuilder::default()
            .filename(archive)
            .root_directory(dirname)
            .create()
            .map_err(|e| ActionError::TarCreate(archive.into(), dirname.into(), e))?;
        Ok(())
    }

    fn tar_extract(archive: &Path, dirname: &Path) -> Result<(), ActionError> {
        let tar = VirtualDriveBuilder::default()
            .filename(archive)
            .root_directory(dirname)
            .open()
            .map_err(|e| ActionError::TarOpen(archive.into(), e))?;
        tar.extract_to(dirname)
            .map_err(|e| ActionError::TarExtract(archive.into(), dirname.into(), e))?;

        Ok(())
    }

    fn spawn_in(
        argv: &[String],
        cwd: &Path,
        extra_env: &[(&str, String)],
    ) -> Result<(), ActionError> {
        println!("SPAWN: argv={argv:?}");
        println!("       cwd={} (exists? {})", cwd.display(), cwd.exists());
        println!("       extra_env={extra_env:?}");

        let argv0 = if let Some(argv0) = argv.first() {
            argv0
        } else {
            return Err(ActionError::SpawnNoArgv0);
        };
        let argv_str = format!("{:?}", argv);

        let mut cmd = Command::new(argv0);
        cmd.args(&argv[1..])
            .envs(extra_env.iter().cloned())
            .stdin(Stdio::null())
            .current_dir(cwd);

        let mut cmd = cmd
            .spawn()
            .map_err(|e| ActionError::SpawnInvoke(argv_str.clone(), cwd.into(), e))?;

        let exit = cmd.wait();
        let exit = exit.map_err(|e| ActionError::SpawnInvoke(argv_str.clone(), cwd.into(), e))?;
        match exit.code() {
            Some(exit) => {
                if exit != 0 {
                    return Err(ActionError::SpawnFailed(argv_str, exit));
                }
            }
            None => return Err(ActionError::SpawnKilledBySignal(argv_str)),
        }

        Ok(())
    }

    fn spawn_str_in(argv: &[&str], cwd: &Path) -> Result<(), ActionError> {
        let argv: Vec<String> = argv.iter().map(|s| s.to_string()).collect();
        Self::spawn_in(&argv, cwd, &[])
    }

    fn spawn(
        plan: &RunnablePlan,
        argv: &[String],
        extra_env: &[(&str, String)],
    ) -> Result<(), ActionError> {
        Self::spawn_in(
            argv,
            Path::new(
                plan.source_dir()
                    .ok_or(ActionError::Missing("source_dir"))?,
            ),
            extra_env,
        )
    }

    fn shell(plan: &RunnablePlan, snippet: &str) -> Result<(), ActionError> {
        let snippet = format!("set -xeuo pipefail\n{}\n", snippet);
        Self::spawn(plan, &["/bin/bash".into(), "-c".into(), snippet], &[])
    }

    fn spawn_str(
        plan: &RunnablePlan,
        argv: &[&str],
        extra_env: &[(&str, String)],
    ) -> Result<(), ActionError> {
        let argv: Vec<String> = argv.iter().map(|s| s.to_string()).collect();
        Self::spawn(plan, &argv, extra_env)
    }

    fn prepend_cargo_bin_dir_to_path() -> String {
        let path = std::env::var("PATH").unwrap_or("/bin".into());
        format!("/root/.cargo/bin:{path}")
    }

    fn rust_envs(
        plan: &RunnablePlan,
        path: &str,
    ) -> Result<Vec<(&'static str, String)>, ActionError> {
        Ok(vec![
            (
                "CARGO_TARGET_DIR",
                plan.cache_dir()
                    .ok_or(ActionError::Missing("cache_dir"))?
                    .into(),
            ),
            (
                "CARGO_HOME",
                plan.deps_dir()
                    .ok_or(ActionError::Missing("deps_dir"))?
                    .into(),
            ),
            ("PATH", path.into()),
        ])
    }

    fn spawn_cargo(plan: &RunnablePlan, args: &[&str]) -> Result<(), ActionError> {
        let path = Self::prepend_cargo_bin_dir_to_path();
        Self::spawn_str(plan, args, &Self::rust_envs(plan, &path)?)
    }

    fn cargo_fmt(plan: &RunnablePlan) -> Result<(), ActionError> {
        Self::spawn_cargo(plan, &["cargo", "fmt", "--check"])
    }

    fn cargo_clippy(plan: &RunnablePlan) -> Result<(), ActionError> {
        Self::spawn_cargo(
            plan,
            &[
                "cargo",
                "clippy",
                "--offline",
                "--locked",
                "--workspace",
                "--all-targets",
                "-vvv",
                "--",
                "--deny",
                "warnings",
            ],
        )
    }

    fn cargo_build(plan: &RunnablePlan) -> Result<(), ActionError> {
        Self::spawn_cargo(
            plan,
            &[
                "cargo",
                "build",
                "--offline",
                "--locked",
                "--workspace",
                "--all-targets",
                "-vvv",
            ],
        )
    }

    fn cargo_test(plan: &RunnablePlan) -> Result<(), ActionError> {
        Self::spawn_cargo(
            plan,
            &[
                "cargo",
                "test",
                "--offline",
                "--locked",
                "--workspace",
                "-vvv",
            ],
        )
    }

    fn cargo_install(plan: &RunnablePlan) -> Result<(), ActionError> {
        Self::spawn_cargo(
            plan,
            &[
                "cargo",
                "install",
                "-vvv",
                "--offline",
                "--locked",
                "--bins",
                "--path=.",
                "--root",
                plan.artifacts_dir()
                    .ok_or(ActionError::Missing("artifacts_dir"))?,
            ],
        )
    }

    fn deb(plan: &RunnablePlan) -> Result<(), ActionError> {
        let shell = format!(
            r#"#!/bin/bash
set -xeuo pipefail

echo "PATH at start: $PATH"
export PATH="/root/.cargo/bin:$PATH"
export CARGO_HOME=/workspace/deps
export DEBEMAIL=liw@liw.fi
export DEBFULLNAME="Lars Wirzenius"
/bin/env

command -v cargo
command -v rustc

cargo --version
rustc --version

# Get name and version of source package.
name="$(dpkg-parsechangelog -SSource)"
version="$(dpkg-parsechangelog -SVersion)"

# Get upstream version: everything before the last dash.
uv="$(echo "$version" | sed 's/-[^-]*$//')"

# Files that will be created.
arch="$(dpkg --print-architecture)"
orig="../${{name}}_${{uv}}.orig.tar.xz"
deb="../${{name}}_${{version}}_${{arch}}.deb"
changes="../${{name}}_${{version}}_${{arch}}.changes"

# Create "upstream tarball".
git archive HEAD | xz >"$orig"

# Build package.
dpkg-buildpackage -us -uc

# Dump some information to make it easier to visually verify
# everything looks OK. Also, test the package with the lintian tool.

ls -l ..
for x in ../*.deb; do dpkg -c "$x"; done
# FIXME: disabled while this prevents radicle-native-ci deb from being built.
# lintian -i --allow-root --fail-on warning ../*.changes

# Move files to artifacts directory.
mv ../*_* {}
        "#,
            qemu::ARTIFACTS_DIR
        );

        let path = Self::prepend_cargo_bin_dir_to_path();

        Self::spawn_str(
            plan,
            &["/bin/bash", "-c", &shell],
            &Self::rust_envs(plan, &path)?,
        )
    }
}

#[derive(Debug, Clone, Serialize, Deserialize)]
#[serde(tag = "action")]
#[serde(rename_all = "snake_case")]
pub enum TrustedAction {
    Dummy,
    Pwd,
    CargoFetch,
    Rsync,
    Dput,
}

impl TrustedAction {
    pub fn names() -> &'static [&'static str] {
        &["dummy", "pwd", "cargo_fetch", "rsync", "dput"]
    }
}

#[derive(Debug, Clone, Serialize, Deserialize)]
#[serde(tag = "action")]
#[serde(rename_all = "snake_case")]
pub enum UnsafeAction {
    Mkdir {
        pathname: PathBuf,
    },
    TarCreate {
        archive: PathBuf,
        directory: PathBuf,
    },
    TarExtract {
        archive: PathBuf,
        directory: PathBuf,
    },
    Spawn {
        argv: Vec<String>,
    },
    Shell {
        shell: String,
    },
    CargoFmt,
    CargoClippy,
    CargoBuild,
    CargoTest,
    CargoInstall,
    Deb,
}

impl UnsafeAction {
    pub fn names() -> &'static [&'static str] {
        &[
            "mkdir",
            "tar_create",
            "tar_extract",
            "spawn",
            "shell",
            "cargo_fmt",
            "cargo_clippy",
            "cargo_build",
            "cargo_test",
            "cargo_install",
            "deb",
        ]
    }

    pub fn mkdir<P: AsRef<Path>>(pathname: P) -> Self {
        Self::Mkdir {
            pathname: pathname.as_ref().into(),
        }
    }

    pub fn tar_create<P: AsRef<Path>>(archive: P, directory: P) -> Self {
        Self::TarCreate {
            archive: archive.as_ref().into(),
            directory: directory.as_ref().into(),
        }
    }

    pub fn tar_extract<P: AsRef<Path>>(archive: P, directory: P) -> Self {
        Self::TarExtract {
            archive: archive.as_ref().into(),
            directory: directory.as_ref().into(),
        }
    }

    pub fn shell(shell: &str) -> Self {
        Self::Shell {
            shell: shell.into(),
        }
    }

    pub fn spawn(argv: &[&str]) -> Self {
        Self::Spawn {
            argv: argv.iter().map(|s| s.to_string()).collect(),
        }
    }
}

#[derive(Debug, thiserror::Error)]
pub enum ActionError {
    #[error("failed to create directory {0}")]
    Mkdir(PathBuf, #[source] std::io::Error),

    #[error("failed to create tar archive {0} from {1}")]
    TarCreate(PathBuf, PathBuf, #[source] VirtualDriveError),

    #[error("failed to open tar archive {0}")]
    TarOpen(PathBuf, #[source] VirtualDriveError),

    #[error("failed to extract tar archive {0} into {1}")]
    TarExtract(PathBuf, PathBuf, #[source] VirtualDriveError),

    #[error("failed to invoke command: empty argv")]
    SpawnNoArgv0,

    #[error("failed to invoke command: {0} in {1}")]
    SpawnInvoke(String, PathBuf, #[source] std::io::Error),

    #[error("command failed was killed by signal")]
    SpawnKilledBySignal(String),

    #[error("command failed: {0}")]
    SpawnFailed(String, i32),

    #[error("failed to remove directory {0}")]
    RemoveDir(PathBuf, #[source] std::io::Error),

    #[error("failed to remove file {0}")]
    RemoveFile(PathBuf, #[source] std::io::Error),

    #[error("failed to create symlink {0} -> {1}")]
    Symlink(PathBuf, PathBuf, #[source] std::io::Error),

    #[error(transparent)]
    Util(#[from] UtilError),

    #[error("for the rsync action you must specify an rsync target (config or command line)")]
    RsyncTargetMissing,

    #[error("for the dput action you must specify a dput target (config or command line)")]
    DputTargetMissing,

    #[error("runnable plan does not have field {0} set")]
    Missing(&'static str),
}

#[cfg(test)]
mod test {
    use super::*;
    use tempfile::tempdir;

    #[test]
    fn mkdir_action() -> Result<(), Box<dyn std::error::Error>> {
        let tmp = tempdir()?;
        let path = tmp.path().join("testdir");
        let action = RunnableAction::from_unsafe_action(&UnsafeAction::mkdir(&path));
        assert!(!path.exists());
        assert!(action.execute(&RunnablePlan::default()).is_ok());
        assert!(path.exists());
        Ok(())
    }

    #[test]
    fn tar_create_action() -> Result<(), Box<dyn std::error::Error>> {
        let tmp = tempdir()?;
        let src = tmp.path().join("src");
        let tar = tmp.path().join("src.tar");

        std::fs::create_dir(&src)?;
        let action = RunnableAction::from_unsafe_action(&UnsafeAction::tar_create(&tar, &src));

        assert!(!tar.exists());
        assert!(action.execute(&RunnablePlan::default()).is_ok());
        assert!(tar.exists());
        Ok(())
    }

    #[test]
    fn tar_extract_action() -> Result<(), Box<dyn std::error::Error>> {
        let tmp = tempdir()?;
        let src = tmp.path().join("src");
        let tar = tmp.path().join("src.tar");
        let extracted = tmp.path().join("extracted");

        std::fs::create_dir(&src)?;
        std::fs::File::create(src.join("file.dat"))?;
        RunnableAction::from_unsafe_action(&UnsafeAction::tar_create(&tar, &src))
            .execute(&RunnablePlan::default())?;

        let action =
            RunnableAction::from_unsafe_action(&UnsafeAction::tar_extract(&tar, &extracted));
        assert!(action.execute(&RunnablePlan::default()).is_ok());
        assert!(extracted.join("file.dat").exists());
        Ok(())
    }

    #[test]
    fn spawn_action() {
        // We can't use the Spawn action here, as it expects the
        // SOURCE_DIR to exist. However, spawn_in does the same thing,
        // except in a directory of our choosing.
        assert!(RunnableAction::spawn_in(&["true".into()], Path::new("/"), &[]).is_ok());
    }
}
