use std::path::PathBuf;

use clap::Parser;
use directories::ProjectDirs;
use log::{info, LevelFilter};

use ambient_driver::config2::{Config, ConfigError};

mod cmd;
use cmd::{AmbientDriverError, Leaf};

const QUAL: &str = "liw.fi";
const ORG: &str = "Ambient CI";
const APP: &str = env!("CARGO_PKG_NAME");

fn main() {
    if let Err(e) = fallible_main() {
        eprintln!("ERROR: {}", e);
        let mut source = e.source();
        while let Some(src) = source {
            eprintln!("caused by: {}", src);
            source = src.source();
        }
        std::process::exit(1);
    }
}

fn fallible_main() -> anyhow::Result<()> {
    setup_env_logger("AMBIENT_LOG", LevelFilter::Warn);
    info!("ambient-driver starts");
    let args = Args::parse();
    let config = args.config()?;
    match &args.cmd {
        Command::Actions(x) => x.run(&config)?,
        Command::Config(x) => x.run(&config)?,
        Command::Image(x) => x.run(&config)?,
        Command::Log(x) => x.run(&config)?,
        Command::Projects(x) => x.run(&config)?,
        Command::Run(x) => x.run(&config)?,
    }
    info!("ambient-driver ends successfully");
    Ok(())
}

fn setup_env_logger(var_name: &str, level_filter: LevelFilter) {
    if std::env::var(var_name).is_ok() {
        env_logger::init_from_env(var_name);
    } else {
        env_logger::builder().filter_level(level_filter).init();
    }
}

#[derive(Debug, Parser)]
#[clap(name = APP, version = env!("CARGO_PKG_VERSION"))]
pub struct Args {
    /// Configuration file to use instead of the default one.
    #[clap(long)]
    config: Option<PathBuf>,

    /// Don't load default configuration file.
    #[clap(long)]
    no_config: bool,

    /// Operation
    #[clap(subcommand)]
    cmd: Command,
}

impl Args {
    fn config(&self) -> Result<Config, ConfigError> {
        if self.no_config {
            Ok(Config::new()?)
        } else if let Some(filename) = &self.config {
            Config::from_yaml_file(filename)
        } else {
            let dirs = ProjectDirs::from(QUAL, ORG, APP).ok_or(ConfigError::ProjectDirs)?;
            let filename = dirs.config_dir().join("config.yaml");
            if filename.exists() {
                Config::from_yaml_file(&filename)
            } else {
                Ok(Config::new()?)
            }
        }
    }
}

#[derive(Debug, Parser)]
enum Command {
    Actions(cmd::actions::Actions),
    Image(ImageCmd),
    Config(cmd::config::ConfigCmd),
    Log(cmd::log::Log),
    Projects(cmd::projects::ProjectsCmd),
    Run(cmd::run::Run),
}

#[derive(Debug, Parser)]
pub struct ImageCmd {
    #[clap(subcommand)]
    cmd: ImageSubCmd,
}

impl ImageCmd {
    fn run(&self, config: &Config) -> Result<(), AmbientDriverError> {
        match &self.cmd {
            ImageSubCmd::CloudInit(x) => x.run(config)?,
            ImageSubCmd::Import(x) => x.run(config)?,
            ImageSubCmd::List(x) => x.run(config)?,
            ImageSubCmd::Remove(x) => x.run(config)?,
            ImageSubCmd::Show(x) => x.run(config)?,
            ImageSubCmd::Verify(x) => x.run(config)?,
        }
        Ok(())
    }
}

#[derive(Debug, Parser)]
enum ImageSubCmd {
    CloudInit(cmd::image::CloudInit),
    Import(cmd::image::ImportImage),
    List(cmd::image::ListImages),
    Remove(cmd::image::RemoveImages),
    Show(cmd::image::ShowImage),
    Verify(cmd::image::VerifyImage),
}
