//! Run QEMU.

use std::{
    fs::{copy, File},
    path::{Path, PathBuf},
    process::Command,
};

use bytesize::MIB;
use log::{debug, error, trace};
use tempfile::{tempdir_in, TempDir};

use crate::{
    cloud_init::LocalDataStore,
    config2::Config,
    vdrive::{VirtualDrive, VirtualDriveError},
};

const OVMF_FD: &str = "/usr/share/ovmf/OVMF.fd";

pub const EXECUTOR_DRIVE: &str = "/dev/vdb";
pub const SOURCE_DRIVE: &str = "/dev/vdc";
pub const ARTIFACT_DRIVE: &str = "/dev/vdd";
pub const CACHE_DRIVE: &str = "/dev/vde";
pub const DEPS_DRIVE: &str = "/dev/vdf";

pub const WORKSPACE_DIR: &str = "/workspace";
pub const SOURCE_DIR: &str = "/workspace/src";
pub const DEPS_DIR: &str = "/workspace/deps";
pub const CACHE_DIR: &str = "/workspace/cache";
pub const ARTIFACTS_DIR: &str = "/workspace/artifacts";

#[derive(Default)]
pub struct QemuRunner<'a> {
    config: Option<&'a Config>,
    image: Option<PathBuf>,
    cloud_init: Option<LocalDataStore>,
    executor: Option<&'a VirtualDrive>,
    source: Option<&'a VirtualDrive>,
    dependencies: Option<&'a VirtualDrive>,
    cache: Option<&'a VirtualDrive>,
    artifacts: Option<&'a VirtualDrive>,
    console_log: Option<PathBuf>,
    run_log: Option<PathBuf>,
    network: bool,
}

impl<'a> QemuRunner<'a> {
    pub fn config(mut self, value: &'a Config) -> Self {
        self.config = Some(value);
        self
    }

    pub fn image(mut self, filename: &Path) -> Self {
        self.image = Some(filename.into());
        self
    }

    pub fn cloud_init(mut self, ds: &LocalDataStore) -> Self {
        self.cloud_init = Some(ds.clone());
        self
    }

    pub fn excutor(mut self, value: &'a VirtualDrive) -> Self {
        self.executor = Some(value);
        self
    }

    pub fn source(mut self, value: &'a VirtualDrive) -> Self {
        self.source = Some(value);
        self
    }

    pub fn dependencies(mut self, value: &'a VirtualDrive) -> Self {
        self.dependencies = Some(value);
        self
    }

    pub fn cache(mut self, value: &'a VirtualDrive) -> Self {
        self.cache = Some(value);
        self
    }

    pub fn artifacts(mut self, value: &'a VirtualDrive) -> Self {
        self.artifacts = Some(value);
        self
    }

    pub fn console_log(mut self, value: &'a Path) -> Self {
        self.console_log = Some(value.into());
        self
    }

    pub fn run_log(mut self, value: &'a Path) -> Self {
        self.run_log = Some(value.into());
        self
    }

    pub fn network(mut self, network: bool) -> Self {
        self.network = network;
        self
    }

    pub fn run(&self) -> Result<(), QemuError> {
        let config = self.config.ok_or(QemuError::Missing("config"))?;
        let image = self.image.clone().ok_or(QemuError::Missing("image"))?;
        let cloud_init = self
            .cloud_init
            .clone()
            .ok_or(QemuError::Missing("cloud_init"))?;
        let executor_drive = self.executor.ok_or(QemuError::Missing("executor_drive"))?;

        let run_log = self.run_log.clone().ok_or(QemuError::Missing("run_log"))?;

        let qemu = Qemu {
            image,
            tmpdir: tempdir_in(config.tmpdir()).map_err(QemuError::TempDir)?,
            console_log: self.console_log.clone(),
            run_log,
            cloud_init,
            executor: executor_drive.clone(),
            source: self.source,
            artifact: self.artifacts,
            dependencies: self.dependencies,
            cache: self.cache,
            cpus: config.cpus(),
            memory: config.memory().as_u64(),
            network: self.network,
        };
        let exit = qemu.run()?;
        debug!("QEMU exit code {}", exit);

        if exit != 0 {
            return Err(QemuError::QemuFailed(exit));
        }

        Ok(())
    }
}

/// A QEMU runner.
#[derive(Debug)]
struct Qemu<'a> {
    image: PathBuf,
    cloud_init: LocalDataStore,
    tmpdir: TempDir,
    console_log: Option<PathBuf>,
    run_log: PathBuf,
    executor: VirtualDrive,
    source: Option<&'a VirtualDrive>,
    artifact: Option<&'a VirtualDrive>,
    dependencies: Option<&'a VirtualDrive>,
    cache: Option<&'a VirtualDrive>,
    cpus: usize,
    memory: u64,
    network: bool,
}

impl Qemu<'_> {
    /// Run QEMU in the specified way.
    fn run(&self) -> Result<i32, QemuError> {
        debug!("run QEMU");
        trace!("{:#?}", self);
        let tmp = tempdir_in(&self.tmpdir).map_err(QemuError::TempDir)?;

        let image = tmp.path().join("vm.qcow2");
        let iso = tmp.path().join("cloud_init.iso");
        let vars = tmp.path().join("vars.fd");

        let console_log = self
            .console_log
            .clone()
            .unwrap_or(tmp.path().join("console.log"));

        debug!("create copy-on-write image and UEFI vars file");
        create_cow_image(&self.image, &image)?;
        copy(OVMF_FD, &vars).map_err(|e| QemuError::Copy(OVMF_FD.into(), e))?;

        debug!("create cloud-init ISO file");
        self.cloud_init
            .iso(&iso)
            .map_err(|err| QemuError::Iso(iso.clone(), err))?;

        debug!("set console log file to {}", console_log.display());
        Self::create_file(&console_log)?;

        debug!("set run file to {}", self.run_log.display());

        let run_log = Self::create_file(&self.run_log)?;

        let cpus = format!("cpus={}", self.cpus);
        let memory = format!("{}", self.memory / MIB);
        let mut args = QemuArgs::default()
            .with_valued_arg("-m", &memory)
            .with_valued_arg("-smp", &cpus)
            .with_valued_arg("-display", "none")
            .with_valued_arg("-serial", &format!("file:{}", console_log.display())) // ttyS0
            .with_valued_arg("-serial", &format!("file:{}", run_log.display())) // ttyS1
            .with_ipflash(0, OVMF_FD, true)
            .with_ipflash(1, &vars.to_string_lossy(), false)
            .with_qcow2(&image.to_string_lossy())
            .with_raw(self.executor.filename(), true)
            .with_valued_arg("-cdrom", &iso.display().to_string());

        if let Some(drive) = self.source {
            args = args.with_raw(drive.filename(), true);
        }
        if let Some(drive) = self.artifact {
            args = args.with_raw(drive.filename(), false);
        }
        if let Some(drive) = self.cache {
            args = args.with_raw(drive.filename(), false);
        }
        if let Some(drive) = self.dependencies {
            args = args.with_raw(drive.filename(), true);
        }
        if self.network {
            args = args.with_valued_arg("-nic", "user,model=virtio");
        }
        args = args.with_arg("-nodefaults");

        debug!("spawn QEMU: {args:#?}");
        let child = Command::new("kvm")
            .args(args.iter())
            .spawn()
            .map_err(QemuError::Invoke)?;

        debug!("wait for QEMU to finish");
        let output = child.wait_with_output().map_err(QemuError::Run)?;
        if output.status.success() {
            debug!("QEMU finished OK");
            let run_log = Self::run_log(&run_log)?;
            let exit = Self::exit_code(&run_log)?;
            Ok(exit)
        } else {
            debug!("QEMU failed");
            let exit = output.status.code().unwrap_or(255);
            let out = String::from_utf8_lossy(&output.stdout).to_string();
            let err = String::from_utf8_lossy(&output.stderr).to_string();
            let _run_log = Self::run_log(&run_log)?;
            let console_log = std::fs::read(&console_log).map_err(QemuError::TemporaryLog)?;
            let console_log = String::from_utf8_lossy(&console_log).to_string();
            error!(
                "QEMU failed: exit={}\nstdout: {:?}\nstderr: {:?}\nVM console: {:?}",
                exit, out, err, console_log
            );
            Ok(exit)
        }
    }

    fn create_file(filename: &Path) -> Result<PathBuf, QemuError> {
        File::create(filename).map_err(|e| QemuError::Log(filename.into(), e))?;
        Ok(filename.into())
    }

    fn run_log(filename: &Path) -> Result<String, QemuError> {
        const BEGIN: &str = "====================== BEGIN ======================";

        let log = std::fs::read(filename).map_err(|e| QemuError::ReadLog(filename.into(), e))?;
        let log = String::from_utf8_lossy(&log);
        trace!("run log from QEMU:\n{log}");

        if let Some((_, log)) = log.split_once(BEGIN) {
            return Ok(log.into());
        }

        Ok("".into())
    }

    fn exit_code(log: &str) -> Result<i32, QemuError> {
        const EXIT: &str = "\nEXIT CODE: ";
        if let Some((_, rest)) = log.split_once(EXIT) {
            trace!("log has {:?}", EXIT);
            if let Some((exit, _)) = rest.split_once('\n') {
                let exit = exit.trim();
                trace!("log has newline after exit: {:?}", exit);
                if let Ok(exit) = exit.parse::<i32>() {
                    trace!("log exit coded parses ok: {}", exit);
                    return Ok(exit);
                }
                debug!("log exit does not parse");
            }
        }
        Err(QemuError::NoExit)
    }
}

fn create_cow_image(backing_file: &Path, new_file: &Path) -> Result<(), QemuError> {
    debug!(
        "qemu-img create {} backing on {}",
        new_file.display(),
        backing_file.display()
    );
    let output = Command::new("qemu-img")
        .arg("create")
        .arg("-b")
        .arg(backing_file)
        .args(["-F", "qcow2"])
        .args(["-f", "qcow2"])
        .arg(new_file)
        .output()
        .map_err(QemuError::Run)?;
    if !output.status.success() {
        let exit = output.status.code().unwrap_or(255);
        let out = String::from_utf8_lossy(&output.stdout).to_string();
        let err = String::from_utf8_lossy(&output.stderr).to_string();
        error!(
            "qemu-img failed: exit={}\nstdout: {:?}\nstderr: {:?}",
            exit, out, err,
        );
        Err(QemuError::QemuImg(exit, err))
    } else {
        Ok(())
    }
}

#[derive(Debug, Default)]
struct QemuArgs {
    args: Vec<String>,
}

impl QemuArgs {
    fn with_arg(mut self, arg: &str) -> Self {
        self.args.push(arg.into());
        self
    }

    fn with_valued_arg(mut self, arg: &str, value: &str) -> Self {
        self.args.push(arg.into());
        self.args.push(value.into());
        self
    }

    fn with_ipflash(mut self, unit: usize, path: &str, readonly: bool) -> Self {
        self.args.push("-drive".into());
        self.args.push(format!(
            "if=pflash,format=raw,unit={},file={}{}",
            unit,
            path,
            if readonly { ",readonly=on" } else { "" },
        ));
        self
    }

    fn with_qcow2(mut self, path: &str) -> Self {
        self.args.push("-drive".into());
        self.args
            .push(format!("format=qcow2,if=virtio,file={}", path));
        self
    }

    fn with_raw(mut self, path: &Path, readonly: bool) -> Self {
        self.args.push("-drive".into());
        self.args.push(format!(
            "format=raw,if=virtio,file={}{}",
            path.display(),
            if readonly { ",readonly=on" } else { "" },
        ));
        self
    }

    fn iter(&self) -> impl Iterator<Item = &str> {
        self.args.iter().map(|s| s.as_str())
    }
}

/// Possible errors from running Qemu.
#[allow(missing_docs)]
#[derive(Debug, thiserror::Error)]
pub enum QemuError {
    #[error("missing field in QemuBuilder: {0}")]
    Missing(&'static str),

    #[error("failed to invoke QEMU")]
    Invoke(#[source] std::io::Error),

    #[error("QEMU process failed")]
    Run(#[source] std::io::Error),

    #[error("qemu-img failed: exit code {0}: stderr:\n{1}")]
    QemuImg(i32, String),

    #[error("failed to create a temporary directory")]
    TempDir(#[source] std::io::Error),

    #[error("failed to copy to temporary directory: {0}")]
    Copy(PathBuf, #[source] std::io::Error),

    #[error("failed to read log file {0}")]
    ReadLog(PathBuf, #[source] std::io::Error),

    #[error("failed to open log file for writing")]
    Log(PathBuf, #[source] std::io::Error),

    #[error("failed to create a tar archive from {0}")]
    Tar(PathBuf, #[source] VirtualDriveError),

    #[error("failed to extract cache drive to {0}")]
    ExtractCache(PathBuf, #[source] VirtualDriveError),

    #[error("failed to read temporary file for logging")]
    TemporaryLog(#[source] std::io::Error),

    #[error("run log lacks exit code of run")]
    NoExit,

    #[error("failed to get length of file {0}")]
    Metadata(PathBuf, #[source] std::io::Error),

    #[error("failed to set length of file to {0}: {1}")]
    SetLen(u64, PathBuf, #[source] std::io::Error),

    #[error("failed to run actions in QEMU")]
    QemuFailed(i32),

    #[error("failed to create cloud-init ISO file {0}")]
    Iso(PathBuf, #[source] crate::cloud_init::CloudInitError),
}
