use std::{
    collections::HashMap,
    io::Write,
    path::{Path, PathBuf},
};

use clap::Parser;
use serde::Serialize;
use walkdir::WalkDir;

fn main() {
    if let Err(err) = fallible_main() {
        eprintln!("ERROR: {err}");
        let mut underlying = err.source();
        while let Some(err) = underlying {
            eprintln!("caused by: {err}");
            underlying = err.source();
        }
        std::process::exit(1);
    }
}

fn fallible_main() -> Result<(), Box<dyn std::error::Error>> {
    let args = Args::parse();
    let mut filecounts = FileCounts::default();
    for dir in args.dirs() {
        filecounts.filecount(dir)?;
    }
    if args.json() {
        filecounts.to_json(std::io::stdout())?;
    } else {
        filecounts.to_text(std::io::stdout())?;
    }
    Ok(())
}
#[derive(Default, Serialize)]
struct FileCounts {
    counts: HashMap<PathBuf, usize>,
}

impl FileCounts {
    fn filecount<P: AsRef<Path>>(&mut self, dir: P) -> Result<(), Box<dyn std::error::Error>> {
        let dir = dir.as_ref();
        let mut count = 0;
        for entry in WalkDir::new(dir) {
            entry?;
            count += 1;
        }
        self.counts.insert(dir.to_path_buf(), count);
        Ok(())
    }

    fn to_text(&self, mut writer: impl Write) -> Result<(), Box<dyn std::error::Error>> {
        let mut dirs: Vec<&PathBuf> = self.counts.keys().collect();
        dirs.sort();
        for dir in dirs {
            let count = self.counts.get(dir).unwrap();
            writer.write_all(format!("{:8} {}\n", *count, dir.display()).as_bytes())?;
        }
        Ok(())
    }

    fn to_json(&self, mut writer: impl Write) -> Result<(), Box<dyn std::error::Error>> {
        serde_json::to_writer(&mut writer, &self.counts)?;
        writer.write_all(b"\n")?;
        Ok(())
    }
}

/// Count files in directories.
#[derive(Debug, Parser)]
#[clap(version)]
struct Args {
    /// Count files in this directory. Default is current working directory.
    #[clap(default_value = ".")]
    dirs: Vec<PathBuf>,

    /// Output JSON in one line. Use something like `jq` to format it.
    #[clap(long)]
    json: bool,
}

impl Args {
    fn dirs(&self) -> &[PathBuf] {
        &self.dirs
    }

    fn json(&self) -> bool {
        self.json
    }
}
