use std::{collections::HashMap, path::PathBuf};

use clap::Parser;
use log::debug;
use serde::Serialize;

use obnam::{
    chunk::{Chunk, ChunkError, DataChunk, Id, Label, Metadata},
    cipher::{Engine, EngineKind, Key},
    client::ClientChunk,
    config::Config,
    plaintext::{Plaintext, PlaintextError},
    store::{ChunkStore, StoreError},
    util::{read, read_file, write, write_file},
};

use super::{Args, Leaf, MainError};

/// Encrypt plain text data as a chunk.
///
/// This creates a new, encrypted chunk from some plain text data and
/// a chunk label. It invents a new random identifier for the chunk.
#[derive(Parser)]
pub struct Encrypt {
    /// Name of file whose content will be put in the chunk; default
    /// is to read chunk from stdin.
    filename: Option<PathBuf>,

    /// Compress data before encryption.
    #[clap(long)]
    compress: bool,

    /// Label for the chunk to declare what is inside the chunk.
    #[clap(long)]
    label: String,

    /// Encryption key.
    #[clap(long)]
    key: Option<String>,

    /// Name of encryption key from client chunk.
    #[clap(long)]
    key_name: Option<String>,

    /// Name of client.
    #[clap(long)]
    client_name: Option<String>,

    /// Store chunk from repository with this ID. This is meant for
    /// testing.
    #[clap(long)]
    id: Option<Id>,

    /// Write the encoded chunk to this file. Default is to store it
    /// in the repository set with the global `--repository` option.
    #[clap(long)]
    output: Option<PathBuf>,
}

impl Leaf for Encrypt {
    type Error = ChunkCmdError;

    fn run(&self, args: &Args, config: &Config) -> Result<(), Self::Error> {
        debug!("Create an encrypted chunk");
        let plaintext = read(&self.filename).map_err(ChunkCmdError::ReadChunk)?;
        let plaintext = if self.compress {
            debug!("Compress data using DEFLATE before encrypting");
            Plaintext::deflated(&plaintext).map_err(ChunkCmdError::Plaintext)?
        } else {
            debug!("Don't compresss data before encrypting");
            Plaintext::uncompressed(&plaintext)
        };

        let engine = if let Some(key) = &self.key {
            Engine::new(EngineKind::Aead, Key::from(key))
        } else {
            let store = Self::open_repository(config)?;
            if let Ok(client_key) = Self::client_key_from_store(args, config, &store) {
                let key_name = self.key_name.as_ref().ok_or(ChunkCmdError::ClientName)?;
                let client_name = self.client_name.as_ref().ok_or(ChunkCmdError::ClientName)?;
                let client_engine = Engine::new(EngineKind::Aead, client_key);
                let (_, client) = client(&client_engine, &store, client_name)?;
                let key = client.key(key_name).ok_or(ChunkCmdError::NoSuchKey(
                    key_name.to_string(),
                    client_name.to_string(),
                ))?;
                Engine::new(EngineKind::Aead, key.clone())
            } else {
                return Err(ChunkCmdError::NeedKey);
            }
        };

        let metadata = if let Some(id) = &self.id {
            Metadata::from_label_and_id(id, &self.label)
        } else {
            Metadata::from_label(&self.label)
        };

        let chunk = Chunk::data(
            DataChunk::encrypt(&engine, &plaintext, metadata).map_err(ChunkCmdError::Encrypt)?,
        );

        if let Some(filename) = &self.output {
            debug!("Write encrypted chunk to file {}", filename.display());
            let ciphertext = chunk.serialize().map_err(ChunkCmdError::Serialize)?;
            write_file(filename, &ciphertext).map_err(ChunkCmdError::WriteChunk)
        } else {
            debug!("Store encrypted chunk in backup repository");
            let store = Self::open_repository(config)?;
            store.add_chunk(&chunk).map_err(ChunkCmdError::AddChunk)
        }
    }
}

/// Decrypt a chunk.
#[derive(Parser)]
pub struct Decrypt {
    /// Name of file whose content will be put in the chunk. If not
    /// given, chunk is read from stdin. Default is to look up chunk
    /// by ID from repository.
    filename: Option<PathBuf>,

    /// Decrypt chunk with this ID from repository. This is overridden
    /// by --filename.
    #[clap(long)]
    id: Option<Id>,

    /// Encryption key.
    #[clap(long)]
    key: Option<String>,

    /// Name of encryption key from client chunk.
    #[clap(long)]
    key_name: Option<String>,

    /// Name of client.
    #[clap(long)]
    client_name: Option<String>,

    /// Write the encoded chunk to this file. Default is to the
    /// standard output.
    #[clap(long)]
    output: Option<PathBuf>,
}

impl Leaf for Decrypt {
    type Error = ChunkCmdError;

    fn run(&self, args: &Args, config: &Config) -> Result<(), Self::Error> {
        let chunk = if let Some(filename) = &self.filename {
            debug!("Decrypt a chunk from file {}", filename.display());
            let ciphertext = read_file(filename).map_err(ChunkCmdError::ReadChunk)?;
            Chunk::parse(&ciphertext).map_err(ChunkCmdError::ParseChunk)?
        } else if let Some(id) = &self.id {
            debug!("Decrypt chunk from chunk repository: {id}");
            let store = Self::open_repository(config)?;
            store.get_data_chunk(id).map_err(ChunkCmdError::GetChunk)?
        } else {
            return Err(ChunkCmdError::DecryptWhat);
        };

        let engine = if let Some(key) = &self.key {
            Engine::new(EngineKind::Aead, Key::from(key))
        } else {
            let store = Self::open_repository(config)?;
            if let Ok(client_key) = Self::client_key_from_store(args, config, &store) {
                let key_name = self.key_name.as_ref().ok_or(ChunkCmdError::ClientName)?;
                let client_name = self.client_name.as_ref().ok_or(ChunkCmdError::ClientName)?;
                let client_engine = Engine::new(EngineKind::Aead, client_key);
                let (_, client) = client(&client_engine, &store, client_name)?;
                let key = client.key(key_name).ok_or(ChunkCmdError::NoSuchKey(
                    key_name.to_string(),
                    client_name.to_string(),
                ))?;
                Engine::new(EngineKind::Aead, key.clone())
            } else {
                return Err(ChunkCmdError::NeedKey);
            }
        };

        let plaintext = match chunk {
            Chunk::Data(data) => data.decrypt(&engine).map_err(ChunkCmdError::Decrypt)?,
            _ => return Err(ChunkCmdError::NotData),
        };
        let plaintext = plaintext.as_bytes().map_err(ChunkCmdError::Plaintext)?;
        write(&self.output, &plaintext).map_err(ChunkCmdError::WriteChunk)?;
        Ok(())
    }
}

/// Show details of what is in a chunk.
///
/// This is a troubleshooting command to help you figure what a chunk
/// contains.
///
/// The output contains the chunk identifier, label, and content,
/// encoded as JSON.
#[derive(Parser)]
pub struct Inspect {
    /// Read chunk from this file.
    #[clap(long)]
    filename: Option<PathBuf>,

    /// Inspect chunk with this ID from repository. This is overridden
    /// by --filename.
    #[clap(long)]
    id: Option<Id>,

    /// Encryption key.
    #[clap(long)]
    key: Option<String>,

    /// Name of encryption key from client chunk.
    #[clap(long)]
    key_name: Option<String>,

    /// Name of client.
    #[clap(long)]
    client_name: Option<String>,
}

impl Leaf for Inspect {
    type Error = ChunkCmdError;

    fn run(&self, args: &Args, config: &Config) -> Result<(), Self::Error> {
        let chunk = if let Some(filename) = &self.filename {
            debug!("inspect chunk from file {}", filename.display());
            let data = read_file(filename).map_err(ChunkCmdError::ReadChunk)?;
            Chunk::parse(&data).map_err(ChunkCmdError::ParseChunk)?
        } else if let Some(id) = &self.id {
            debug!("inspect chunk from chunk repository {id}");
            let store = Self::open_repository(config)?;
            store.get_data_chunk(id).map_err(ChunkCmdError::GetChunk)?
        } else {
            return Err(ChunkCmdError::InspectWhat);
        };

        let without_decryption = InspectedChunk {
            id: chunk.metadata().lossy_id(),
            label: chunk.metadata().lossy_label(),
            data: None,
        };

        let inspected = if let Some(key) = &self.key {
            let engine = Engine::new(EngineKind::Aead, Key::from(key));
            let plaintext = match &chunk {
                Chunk::Data(data) => data.decrypt(&engine).map_err(ChunkCmdError::Decrypt)?,
                _ => return Err(ChunkCmdError::NotData),
            };
            let plaintext = plaintext.as_bytes().map_err(ChunkCmdError::Plaintext)?;
            let plaintext = String::from_utf8_lossy(&plaintext).to_string();

            InspectedChunk {
                id: chunk.metadata().lossy_id(),
                label: chunk.metadata().lossy_label(),
                data: Some(plaintext),
            }
        } else if let Ok(store) = Self::open_repository(config) {
            if let Ok(client_key) = Self::client_key_from_store(args, config, &store) {
                let key_name = self.key_name.as_ref().ok_or(ChunkCmdError::ClientName)?;
                let client_name = self.client_name.as_ref().ok_or(ChunkCmdError::ClientName)?;
                let client_engine = Engine::new(EngineKind::Aead, client_key);
                let (_, client) = client(&client_engine, &store, client_name)?;
                let key = client.key(key_name).ok_or(ChunkCmdError::NoSuchKey(
                    key_name.to_string(),
                    client_name.to_string(),
                ))?;
                let engine = Engine::new(EngineKind::Aead, key.clone());
                let plaintext = match &chunk {
                    Chunk::Data(data) => data.decrypt(&engine).map_err(ChunkCmdError::Decrypt)?,
                    _ => return Err(ChunkCmdError::NotData),
                };
                let plaintext = plaintext.as_bytes().map_err(ChunkCmdError::Plaintext)?;
                let plaintext = String::from_utf8_lossy(&plaintext).to_string();

                InspectedChunk {
                    id: chunk.metadata().lossy_id(),
                    label: chunk.metadata().lossy_label(),
                    data: Some(plaintext),
                }
            } else {
                without_decryption
            }
        } else {
            without_decryption
        };

        let json = serde_json::to_string_pretty(&inspected).map_err(ChunkCmdError::ToJson)?;
        println!("{json}");

        Ok(())
    }
}

#[derive(Serialize)]
struct InspectedChunk {
    id: String,
    label: String,
    data: Option<String>,
}

fn client(
    engine: &Engine,
    store: &ChunkStore,
    wanted: &str,
) -> Result<(Id, ClientChunk), ChunkCmdError> {
    fn is_old(id: &Id, map: &HashMap<Id, ClientChunk>) -> bool {
        for client in map.values() {
            if client.old_versions().contains(id) {
                return true;
            }
        }
        false
    }

    let label = Label::from("client");
    let metas = store
        .find_chunks(&label)
        .map_err(ChunkCmdError::FindClient)?;

    let mut map: HashMap<Id, ClientChunk> = HashMap::new();
    for x in metas {
        let chunk = store.get_client_chunk(x.id()).unwrap();
        let plaintext = match chunk {
            Chunk::Client(data) => data.decrypt(engine).map_err(ChunkCmdError::Decrypt)?,
            _ => return Err(ChunkCmdError::NotData),
        };
        let client = ClientChunk::try_from(&plaintext).unwrap();
        map.insert(x.id().clone(), client);
    }

    let mut roots: Vec<&Id> = map.keys().collect();
    loop {
        if roots.is_empty() {
            return Err(ChunkCmdError::NoSuchClient(wanted.to_string()));
        }

        let mut found = None;
        for (i, id) in roots.iter().enumerate() {
            if is_old(id, &map) {
                found = Some(i);
                break;
            }
        }

        if let Some(i) = found {
            roots.remove(i);
            if roots.len() == 1 {
                break;
            }
        } else {
            break;
        }
    }

    match roots.len() {
        0 => Err(ChunkCmdError::NoSuchClient(wanted.to_string())),
        1 => {
            let id = roots
                .first()
                .ok_or(ChunkCmdError::NoSuchClient(wanted.to_string()))?;
            let c = map
                .get(id)
                .ok_or(ChunkCmdError::NoSuchClient(wanted.to_string()))?;
            Ok(((*id).clone(), c.clone()))
        }
        _ => Err(ChunkCmdError::TooManyClients(wanted.to_string())),
    }
}

#[derive(Debug, thiserror::Error)]
pub enum ChunkCmdError {
    #[error("failed to read chunk")]
    ReadChunk(#[source] obnam::util::UtilError),

    #[error("failed to create chunk")]
    Serialize(#[source] ChunkError),

    #[error("failed to parse chunk")]
    ParseChunk(#[source] ChunkError),

    #[error("failed to write chunk")]
    WriteChunk(#[source] obnam::util::UtilError),

    #[error("failed to encrypt chunk")]
    Encrypt(#[source] ChunkError),

    #[error("failed to decrypt chunk")]
    Decrypt(#[source] ChunkError),

    #[error("failed to serialize chunk or metadata to JSON")]
    ToJson(#[source] serde_json::Error),

    #[error("failed to add chunk to repository")]
    AddChunk(#[source] StoreError),

    #[error("failed to get chunk from repository")]
    GetChunk(#[source] StoreError),

    #[error("failed to parse chunk from repository as a data chunk")]
    NotData,

    #[error("must use --id or --filename for chunk decrypt")]
    DecryptWhat,

    #[error("must use --id or --filename for chunk inspect")]
    InspectWhat,

    #[error(transparent)]
    Main(#[from] MainError),

    #[error(transparent)]
    Plaintext(#[from] PlaintextError),

    #[error("need either chunk key (--key) or client key (--client-key)")]
    NeedKey,

    #[error("failed to find client chunks")]
    FindClient(#[source] StoreError),

    #[error("no client found named {0}")]
    NoSuchClient(String),

    #[error("more than one client found named {0}")]
    TooManyClients(String),

    #[error("if --key-name is used, --client-name must also be used")]
    ClientName,

    #[error("no key {0} for client {1}")]
    NoSuchKey(String, String),
}
