use std::path::PathBuf;

use clap::Parser;

use log::debug;
use obnam::{
    chunk::{Chunk, Id, Label, Metadata},
    config::Config,
    store::ChunkStore,
    util::{read_file, write},
};

use super::{Args, Leaf, MainError};

/// Has a directory been initialized as a backup repository?
#[derive(Parser)]
pub struct IsRepo {}

impl Leaf for IsRepo {
    type Error = StoreCmdError;

    fn run(&self, _args: &Args, config: &Config) -> Result<(), Self::Error> {
        let dir = Self::need_repository(config)?;
        debug!(
            "Check if directory is a chunk repository: {}",
            dir.display()
        );
        if ChunkStore::is_init(dir) {
            Ok(())
        } else {
            Err(StoreCmdError::NotInit(dir.into()))
        }
    }
}

/// Initialize a directory as a backup repository. The directory must exist.
#[derive(Parser)]
pub struct InitRepo {}

impl Leaf for InitRepo {
    type Error = StoreCmdError;

    fn run(&self, _args: &Args, config: &Config) -> Result<(), Self::Error> {
        let dir = Self::need_repository(config)?;
        debug!(
            "Initialize a direcory as a chunk repository: {}",
            dir.display()
        );
        ChunkStore::init(dir).map_err(StoreCmdError::Init)?;
        Ok(())
    }
}

/// Add a chunk to the backup repository.
#[derive(Parser)]
pub struct RepoAddChunk {
    id: Id,
    label: Label,
    filename: PathBuf,
}

impl Leaf for RepoAddChunk {
    type Error = StoreCmdError;

    fn run(&self, _args: &Args, config: &Config) -> Result<(), Self::Error> {
        debug!("Add a chunk to the chunk repository");
        let store = Self::open_repository(config)?;

        let data = read_file(&self.filename).map_err(StoreCmdError::ReadChunk)?;
        let metadata = Metadata::new(self.id.clone(), self.label.clone());
        let chunk = Chunk::blob(data, metadata);

        store.add_chunk(&chunk).map_err(StoreCmdError::AddChunk)?;

        Ok(())
    }
}

/// List identifies for all chunks in the backup repository.
#[derive(Parser)]
pub struct ListRepoChunks {}

impl Leaf for ListRepoChunks {
    type Error = StoreCmdError;

    fn run(&self, _args: &Args, config: &Config) -> Result<(), Self::Error> {
        debug!("List chunks in the chunk repository");
        let store = Self::open_repository(config)?;
        for metadata in store.all_chunks().map_err(StoreCmdError::ListChunks)? {
            println!("{}", fuzzy_id(metadata.id()));
        }
        Ok(())
    }
}

/// Find chunks with a given label in in the backup repository.
#[derive(Parser)]
pub struct FindRepoChunks {
    label: Label,
}

impl Leaf for FindRepoChunks {
    type Error = StoreCmdError;

    fn run(&self, _args: &Args, config: &Config) -> Result<(), Self::Error> {
        debug!(
            "Find chunks in the chunk repository with label {}",
            self.label
        );
        let store = Self::open_repository(config)?;
        for metadata in store
            .find_chunks(&self.label)
            .map_err(StoreCmdError::ListChunks)?
        {
            println!("{}", fuzzy_id(metadata.id()));
        }

        Ok(())
    }
}

/// Output the path to a chunk with a given identifier in the backup repository.
#[derive(Parser)]
pub struct RepoChunkPath {
    id: Id,

    #[clap(long, short)]
    output: Option<PathBuf>,
}

impl Leaf for RepoChunkPath {
    type Error = StoreCmdError;

    fn run(&self, _args: &Args, config: &Config) -> Result<(), Self::Error> {
        debug!(
            "Show file system path to chunks in the chunk repository: {}",
            self.id
        );
        let store = Self::open_repository(config)?;
        let filename = store.chunk_filename(&self.id);
        write(&self.output, filename.as_os_str().as_encoded_bytes())
            .map_err(StoreCmdError::WriteCHunkFilename)?;
        Ok(())
    }
}

/// Remove a chunk from a backup repository.
#[derive(Parser)]
pub struct RepoRemoveChunk {
    id: Id,
}

impl Leaf for RepoRemoveChunk {
    type Error = StoreCmdError;

    fn run(&self, _args: &Args, config: &Config) -> Result<(), Self::Error> {
        debug!("Remove chunk from the chunk repository: {}", self.id);
        let store = Self::open_repository(config)?;
        store
            .remove_chunk(&self.id)
            .map_err(|err| StoreCmdError::Remove(self.id.clone(), err))?;
        Ok(())
    }
}

fn fuzzy_id(id: &Id) -> String {
    String::from_utf8_lossy(id.as_bytes()).into()
}

#[derive(Debug, thiserror::Error)]
pub enum StoreCmdError {
    #[error("backup repository directory has not been initialized: {0}")]
    NotInit(PathBuf),

    #[error("failed to initialize backup repository directory")]
    Init(#[source] obnam::store::StoreError),

    #[error("failed to read chunk file")]
    ReadChunk(#[source] obnam::util::UtilError),

    #[error("failed add chunk to backup repository")]
    AddChunk(#[source] obnam::store::StoreError),

    #[error("failed list chunks in backup repository")]
    ListChunks(#[source] obnam::store::StoreError),

    #[error("failed to write chunk filename")]
    WriteCHunkFilename(#[source] obnam::util::UtilError),

    #[error("failed to remove chunk from backup repository: {0:?}")]
    Remove(Id, #[source] obnam::store::StoreError),

    #[error(transparent)]
    Main(#[from] MainError),
}
