use std::{
    collections::HashMap,
    path::{Path, PathBuf},
};

use ambient_ci::action::{TrustedAction, UnsafeAction};
use nonempty::{nonempty, NonEmpty};
use radicle_ci_broker::msg::helper::{indented, runcmd, AdminLog};
use serde::{Deserialize, Serialize};
use tempfile::tempdir;

use crate::AdapterError;

// Read `.radicle/ambient.yaml` from the source tree.
#[derive(Debug, Default, Deserialize, Serialize)]
#[serde(deny_unknown_fields)]
pub struct DistilledPlan {
    pre_plan: Option<Vec<TrustedAction>>,
    plan: Option<Vec<UnsafeAction>>,
    post_plan: Option<Vec<TrustedAction>>,
}

impl DistilledPlan {
    pub fn read(filename: &Path) -> Result<Self, AdapterError> {
        let data =
            std::fs::read(filename).map_err(|err| AdapterError::ReadPlan(filename.into(), err))?;
        serde_yml::from_slice(&data).map_err(|err| AdapterError::ParsePlan(filename.into(), err))
    }

    pub fn drop_post_plan(&mut self) {
        self.post_plan = None;
    }
}

#[derive(Debug, Serialize)]
pub struct Project {
    source: PathBuf,
    image: PathBuf,
    pre_plan: Vec<TrustedAction>,
    plan: Vec<UnsafeAction>,
    post_plan: Vec<TrustedAction>,
    artifact_max_size: Option<u64>,
    cache_max_size: Option<u64>,
}

#[derive(Debug, Serialize)]
pub struct Projects {
    projects: HashMap<String, Project>,
}

impl Projects {
    pub fn write(&self, filename: &Path) -> Result<String, AdapterError> {
        let text = serde_yml::to_string(&self).map_err(AdapterError::ToYaml)?;
        std::fs::write(filename, text.as_bytes()).map_err(AdapterError::WriteProjects)?;
        Ok(text)
    }

    pub fn project_name(&self) -> String {
        let keys: Vec<&str> = self.projects.keys().map(|s| s.as_str()).collect();
        assert_eq!(keys.len(), 1);
        keys[0].to_string()
    }
}

#[derive(Debug, Default)]
pub struct ProjectsBuilder {
    name: Option<String>,
    source: Option<PathBuf>,
    image: Option<PathBuf>,
    pre_plan: Vec<TrustedAction>,
    plan: Vec<UnsafeAction>,
    post_plan: Vec<TrustedAction>,
    artifact_max_size: Option<u64>,
    cache_max_size: Option<u64>,
}

impl ProjectsBuilder {
    pub fn name(mut self, name: &str) -> Self {
        self.name = Some(name.into());
        self
    }

    pub fn source(mut self, path: &Path) -> Self {
        self.source = Some(path.into());
        self
    }

    pub fn image(mut self, path: &Path) -> Self {
        self.image = Some(path.into());
        self
    }

    pub fn artifact_max_size(mut self, size: u64) -> Self {
        self.artifact_max_size = Some(size);
        self
    }

    pub fn cache_max_size(mut self, size: u64) -> Self {
        self.cache_max_size = Some(size);
        self
    }

    pub fn plan(mut self, plan: &DistilledPlan) -> Self {
        if let Some(actions) = &plan.pre_plan {
            self.pre_plan = actions.to_vec();
        }
        if let Some(actions) = &plan.plan {
            self.plan = actions.to_vec();
        }
        if let Some(actions) = &plan.post_plan {
            self.post_plan = actions.to_vec();
        }
        self
    }

    pub fn build(self) -> Projects {
        let projects: HashMap<String, Project> = HashMap::from([(
            self.name.unwrap(),
            Project {
                source: self.source.expect("ProjectsBuilder::source is set"),
                image: self.image.expect("ProjectsBuilder::image is set"),
                pre_plan: self.pre_plan.clone(),
                plan: self.plan.clone(),
                post_plan: self.post_plan.clone(),
                artifact_max_size: self.artifact_max_size,
                cache_max_size: self.cache_max_size,
            },
        )]);
        Projects { projects }
    }
}

pub fn run_ambient_ci(
    adminlog: &mut AdminLog,
    dry_run: bool,
    projects: &Projects,
    extra_ambient_config: Option<&Path>,
    extra_ambient_config_values: &HashMap<String, serde_yml::Value>,
) -> Result<CiOutput, AdapterError> {
    let tmp = tempdir().map_err(AdapterError::TempDir)?;
    let projects_filename = tmp.path().join("projects.yaml");

    let project_name = projects.project_name();
    let projects = projects.write(&projects_filename)?;
    indented(adminlog, "projects", projects.as_bytes());

    adminlog
        .writeln("execute_ci: run ambient")
        .map_err(AdapterError::AdminLog)?;
    let filename = projects_filename.display().to_string();

    // Construct an argv array to give to the `runcmd` function. We
    // first make it an array of owned strings so that we can easily
    // add extra arguments, without the borrow checker being bothered.
    let mut argv: Vec<String> = ["env", "AMBIENT_LOG=debug", "ambient"]
        .iter()
        .map(|s| s.to_string())
        .collect();

    // If requested add an extra config file.
    if let Some(extra) = extra_ambient_config {
        let extra = extra.display().to_string();
        argv.extend(["--config".to_string(), extra]);
    }

    // If requested create a new config file with specific config
    // values, and add that to the argument list.
    if !extra_ambient_config_values.is_empty() {
        let yaml: String = serde_yml::to_string(extra_ambient_config_values)
            .map_err(AdapterError::AmbientConfigToYaml)?;

        let filename = tmp.path().join("ambient_extra_values.yaml");
        std::fs::write(&filename, &yaml).map_err(AdapterError::WriteConfig)?;

        argv.extend(["--config".to_string(), filename.display().to_string()]); // FIXME
    }

    // Make a list of string slices, as that's what `runcmd` actually
    // wants.
    let mut borrowed: Vec<&str> = argv.iter().map(String::as_str).collect();

    borrowed.extend_from_slice(&["run", "--force", "--projects", filename.as_str()]);
    let borrowed = NonEmpty::from_vec(borrowed).unwrap();

    // Run `ambient`.
    let (exit, stdout) = runcmd(adminlog, dry_run, &borrowed, Path::new("."))
        .map_err(AdapterError::MessageHelper)?;
    adminlog
        .writeln(&format!("stdout: {}", String::from_utf8_lossy(&stdout)))
        .map_err(AdapterError::AdminLog)?;

    // Get the run log from `ambient`.
    let (_, run_log_raw) = get_log(adminlog, dry_run, &project_name, extra_ambient_config)?;

    Ok(CiOutput {
        exit,
        stdout,
        stderr: vec![],
        run_log: run_log_raw,
    })
}

fn get_log(
    adminlog: &mut AdminLog,
    dry_run: bool,
    project_name: &str,
    extra_ambient_config: Option<&Path>,
) -> Result<(i32, Vec<u8>), AdapterError> {
    if let Some(extra) = extra_ambient_config {
        let extra = extra.display().to_string();
        let argv = nonempty!["ambient", "--config", extra.as_str(), "log", project_name];
        runcmd(adminlog, dry_run, &argv, Path::new(".")).map_err(AdapterError::MessageHelper)
    } else {
        let argv = nonempty!["ambient", "log", project_name];
        runcmd(adminlog, dry_run, &argv, Path::new(".")).map_err(AdapterError::MessageHelper)
    }
}

pub struct CiOutput {
    pub exit: i32,
    pub stdout: Vec<u8>,
    pub stderr: Vec<u8>,
    pub run_log: Vec<u8>,
}
