//! This is a skeleton Radicle CI adapter for demonstration purposes.
//!
//! This program shows how to build a very simple adapter for Radicle
//! CI. As per the CI broker/adapter protocol, it reads a trigger
//! request message from its stdin, and writes two response messages
//! to its stdout. Based on the trigger, it clones the relevant Git
//! repository, and checks out the relevant Git commit, then runs
//! executed a silly CI plan. To keep things simple, and quick, in
//! this program the plan is "does the .radicle directory exist?". If
//! it exists, CI succeeds, otherwise it fails. If any of the previous
//! steps fails, CI also fails.
//!
//! Any errors are reported to stderr. If things go wrong very early,
//! the first response message may be a failure message, rather than a
//! "triggered" message. The CI broke will hopefully handle this.

#![allow(clippy::result_large_err)]

use std::{error::Error, fs::create_dir_all, path::Path, time::SystemTime};

use clap::Parser;
use tempfile::tempdir;

use radicle::prelude::{Profile, ReadStorage, RepoId};

use radicle_ci_broker::msg::{
    helper::{get_sources, read_request, write_failed, write_succeeded, write_triggered, AdminLog},
    Request, RunId, RunResult,
};

use radicle_ci_ambient::{
    ambient::{self, run_ambient_ci, CiOutput},
    config::Config,
    runlog::RunLog,
    AdapterError,
};

// Name of environment variable that has filename for configuration
// file.
const CONFIG_ENV: &str = "RADICLE_CI_AMBIENT";

fn main() -> Result<(), AdapterError> {
    match fallible_main() {
        Ok(true) => write_succeeded().map_err(AdapterError::Succeded)?,
        Ok(false) => write_failed().map_err(AdapterError::Failed)?,
        Err(err) => {
            eprintln!("ERROR: {err}");
            write_failed().map_err(AdapterError::Failed)?;
            std::process::exit(1);
        }
    }

    Ok(())
}

fn fallible_main() -> Result<bool, AdapterError> {
    let args = Args::parse();

    let config = Config::load_via_env(CONFIG_ENV)?;

    if args.show_config {
        eprintln!("{config:#?}");
        return Ok(false);
    }

    let mut adminlog = config.open_log()?;
    adminlog.writeln(&format!("radicle-ci-ambient starts: {config:#?}"))?;

    // Read the trigger request message and extract the repository ID
    // and commit from it, if possible.
    let req = read_request().map_err(AdapterError::ReadRequest)?;
    let repoid = req.repo();
    let commit = req.commit().map_err(AdapterError::Commit)?;

    // Find project name.
    let profile = Profile::load().map_err(AdapterError::Profile)?;
    let doc = profile
        .storage
        .get(repoid)
        .map_err(|err| AdapterError::GetIdDoc(repoid, err))?;

    let repository_name = if let Some(doc) = doc {
        doc.project()
            .map_err(AdapterError::GetProject)?
            .name()
            .to_string()
    } else {
        "unknown Radicle repository name".into()
    };
    adminlog
        .writeln(&format!("repository name: {repository_name:?}"))
        .ok();

    adminlog
        .writeln(&format!(
            "trust repository name: {}",
            config.trust_repository_name()
        ))
        .ok();
    let project_name = if config.trust_repository_name() {
        repository_name
    } else {
        repoid.to_string()
    };
    adminlog
        .writeln(&format!("project name: {project_name:?}"))
        .ok();

    // Create a temporary directory and a sub-directory `src` where
    // we'll put the source code from Git.
    let tmp = tempdir().map_err(AdapterError::TempDir)?;
    let src = tmp.path().join("src");
    get_sources(&mut adminlog, args.dry_run, repoid, commit, &src)?;

    // Info about this CI run.
    let mut info = RunInfoBuilder::default();

    // Invent an adapter run ID and tell it to the CI broker.
    let run_id = RunId::default();
    info.set_adapter_run_id(&run_id);
    let project_log_dir = config.logdir.join(&project_name);
    if !project_log_dir.exists() {
        adminlog.writeln("need to create log directory")?;
        create_dir_all(&project_log_dir)
            .map_err(|err| AdapterError::CreateDirs(project_log_dir, err))?;
        adminlog.writeln("created log directory")?;
    }
    let run_log_relative = format!("{project_name}/{run_id}.html");

    let url = config
        .base_url
        .clone()
        .map(|base| format!("{base}/{run_log_relative}"));
    write_triggered(&run_id, url.as_deref()).map_err(AdapterError::Triggered)?;

    // Execute CI in the source directory.
    let result = execute_ci(
        &mut adminlog,
        args.dry_run,
        &config,
        info,
        &req,
        &repoid,
        &project_name,
        &run_log_relative,
        &src,
    );
    adminlog.writeln(&format!(
        "radicle-ci-ambient ends: Ambient result is {result:?}"
    ))?;

    if let Err(err) = &result {
        adminlog.writeln(&format!("ERROR: {err}"))?;
        let mut err = err.source();
        while let Some(cause) = err {
            adminlog.writeln(&format!("caused by {cause}"))?;
            err = cause.source();
        }
    }

    result
}

#[derive(Parser)]
#[clap(version = env!("VERSION"))]
struct Args {
    #[clap(long)]
    dry_run: bool,

    #[clap(long)]
    show_config: bool,
}

// This is what executes the steps CI is meant to execute. Edit this
// according to your needs.
#[allow(clippy::too_many_arguments)]
fn execute_ci(
    adminlog: &mut AdminLog,
    dry_run: bool,
    config: &Config,
    mut info: RunInfoBuilder,
    trigger: &Request,
    repo_id: &RepoId,
    project_name: &str,
    run_log_relative: &str,
    src: &Path,
) -> Result<bool, AdapterError> {
    adminlog.writeln(&format!("execute_ci: src={}", src.display()))?;

    let mut distilled = if dry_run {
        ambient::DistilledPlan::default()
    } else {
        let filename = src.join(".radicle/ambient.yaml");
        ambient::DistilledPlan::read(&filename)?
    };

    if !config.allow_post_plan() {
        distilled.drop_post_plan();
    }

    let projects = ambient::ProjectsBuilder::default()
        .name(project_name)
        .source(src)
        .image(&config.image)
        .plan(&distilled)
        .build();

    let output = run_ambient_ci(
        adminlog,
        dry_run,
        &projects,
        config.extra_ambient_config(),
        config.extra_ambient_config_values(),
    )?;
    if output.exit == 0 {
        info.set_result(RunResult::Success);
    } else {
        info.set_result(RunResult::Failure);
    }

    let info = info.build()?;
    write_log_files(
        &info,
        config,
        adminlog,
        trigger,
        repo_id,
        project_name,
        run_log_relative,
        &output,
    )?;

    if dry_run {
        adminlog.writeln("execute_ci: Ambient is pretending to be OK")?;
        Ok(true)
    } else if output.exit == 0 {
        adminlog.writeln("execute_ci: Ambient is OK")?;
        Ok(true)
    } else {
        adminlog.writeln(&format!("ERROR: ambient exit code {}", output.exit))?;
        Ok(false)
    }
}

struct RunInfoBuilder {
    started: SystemTime,
    adapter_run_id: Option<RunId>,
    result: Option<RunResult>,
}

impl Default for RunInfoBuilder {
    fn default() -> Self {
        Self {
            started: SystemTime::now(),
            adapter_run_id: None,
            result: None,
        }
    }
}

impl RunInfoBuilder {
    fn build(self) -> Result<RunInfo, AdapterError> {
        Ok(RunInfo {
            started: self.started,
            adapter_run_id: self
                .adapter_run_id
                .ok_or(AdapterError::Missing("adapter_run_id"))?,
            result: self.result.ok_or(AdapterError::Missing("result"))?,
        })
    }

    fn set_adapter_run_id(&mut self, id: &RunId) {
        self.adapter_run_id = Some(id.clone());
    }

    fn set_result(&mut self, result: RunResult) {
        self.result = Some(result);
    }
}

struct RunInfo {
    started: SystemTime,
    adapter_run_id: RunId,
    result: RunResult,
}

#[allow(clippy::too_many_arguments)]
fn write_log_files(
    info: &RunInfo,
    config: &Config,
    adminlog: &mut AdminLog,
    trigger: &Request,
    _repo_id: &RepoId,
    project_name: &str,
    run_log_relative: &str,
    ci_output: &CiOutput,
) -> Result<(), AdapterError> {
    if !config.logdir.exists() {
        adminlog.writeln("create log directory")?;
        std::fs::create_dir_all(&config.logdir)
            .map_err(|err| AdapterError::CreateDirs(config.logdir.clone(), err))?;
    }

    let mut log = RunLog::default();
    log.set_project_name(project_name.into());
    log.set_trigger(trigger.clone());
    log.set_started(info.started);
    log.set_adapter_run_id(info.adapter_run_id.clone());
    log.set_result(info.result.clone());
    log.set_ambient_output(ci_output.stdout.clone(), ci_output.stderr.clone());
    log.set_ambient_run_log(ci_output.run_log.clone());
    let page = log.to_html();

    adminlog.writeln("convert terminal output to HTML and write that")?;
    let filename = config.logdir.join(run_log_relative).with_extension("html");
    std::fs::write(&filename, page.to_string())
        .map_err(|err| AdapterError::WriteRunLog(filename.clone(), err))?;
    Ok(())
}
