use std::{
    collections::HashMap,
    io::Read,
    path::{Path, PathBuf},
};

use radicle_ci_broker::msg::helper::{AdminLog, LogError};
use serde::{Deserialize, Serialize};

#[derive(Debug, Serialize, Deserialize)]
#[serde(deny_unknown_fields)]
pub struct Config {
    /// Image to use for Ambient CI run.
    pub image: PathBuf,

    /// Directory where per-run directories are stored. Each run gets
    /// its own dedicated subdirectory.
    pub logdir: PathBuf,

    /// File where native CI should write a log.
    pub log: PathBuf,

    /// Optional base URL to information about each run. The run ID is
    /// appended, with a slash if needed.
    #[serde(skip_serializing_if = "Option::is_none")]
    pub base_url: Option<String>,

    /// Use the repository name as the Ambient project name, instead
    /// of the Radicle repository ID.
    trust_repository_name: Option<bool>,

    /// Give Ambient an extra configuration file, in addition to its
    /// default one.
    extra_ambient_config: Option<PathBuf>,

    /// Extra Ambient configuration, embedded here, not in a separate file.
    #[serde(default)]
    extra_ambient_config_values: HashMap<String, serde_norway::Value>,

    /// Allow post-plan actions? By default they are not allowed.
    allow_post_plan: Option<bool>,

    /// Run `ambient` on this host, over SSH, instead of locally. The argument
    /// is given to `ssh` as a target.
    ssh_target: Option<String>,
}

impl Config {
    pub fn load_via_env(env: &str) -> Result<Self, ConfigError> {
        let filename = std::env::var(env).map_err(|e| ConfigError::GetEnv(env.into(), e))?;
        let filename = Path::new(&filename);
        let config = Config::read(filename)?;
        Ok(config)
    }

    pub fn open_log(&self) -> Result<AdminLog, ConfigError> {
        Ok(AdminLog::open(&self.log)?)
    }

    /// Read configuration specification from a file.
    pub fn read(filename: &Path) -> Result<Self, ConfigError> {
        let mut file = std::fs::File::open(filename)
            .map_err(|e| ConfigError::ReadConfig(filename.into(), e))?;
        let mut data = vec![];
        file.read_to_end(&mut data)
            .map_err(|err| ConfigError::ReadConfig(filename.to_path_buf(), err))?;
        serde_norway::from_slice(&data).map_err(|e| ConfigError::ParseConfig(filename.into(), e))
    }

    /// Return configuration serialized to JSON.
    pub fn as_json(&self) -> String {
        // We don't check the result: we know a configuration can
        // always be serialized to JSON.
        serde_json::to_string_pretty(self).unwrap()
    }

    pub fn trust_repository_name(&self) -> bool {
        self.trust_repository_name.unwrap_or(false)
    }

    pub fn allow_post_plan(&self) -> bool {
        self.allow_post_plan == Some(true)
    }

    pub fn extra_ambient_config(&self) -> Option<&Path> {
        self.extra_ambient_config.as_deref()
    }

    pub fn extra_ambient_config_values(&self) -> &HashMap<String, serde_norway::Value> {
        &self.extra_ambient_config_values
    }

    pub fn ssh_target(&self) -> Option<&str> {
        self.ssh_target.as_deref()
    }
}

#[derive(Debug, thiserror::Error)]
pub enum ConfigError {
    #[error("failed to read configuration file {0}")]
    ReadConfig(PathBuf, #[source] std::io::Error),

    #[error("failed to parse configuration file as YAML: {0}")]
    ParseConfig(PathBuf, #[source] serde_norway::Error),

    #[error("failed to get environment variable {0}")]
    GetEnv(String, #[source] std::env::VarError),

    #[error(transparent)]
    Log(#[from] LogError),
}
