use std::{
    collections::HashMap,
    io::{pipe, Read},
    path::{Path, PathBuf},
    process::{Command, Stdio},
};

use ambient_ci::action::{TrustedAction, UnsafeAction};
use radicle_ci_broker::msg::helper::{indented, AdminLog};
use serde::{Deserialize, Serialize};
use tempfile::tempdir;

use crate::AdapterError;

// Exit code to indicate we didn't get one from the process.
pub const NO_EXIT: i32 = 999;

// Read `.radicle/ambient.yaml` from the source tree.
#[derive(Debug, Default, Deserialize, Serialize)]
#[serde(deny_unknown_fields)]
pub struct DistilledPlan {
    pre_plan: Option<Vec<TrustedAction>>,
    plan: Option<Vec<UnsafeAction>>,
    post_plan: Option<Vec<TrustedAction>>,
}

impl DistilledPlan {
    pub fn read(filename: &Path) -> Result<Self, AdapterError> {
        let data =
            std::fs::read(filename).map_err(|err| AdapterError::ReadPlan(filename.into(), err))?;
        serde_norway::from_slice(&data).map_err(|err| AdapterError::ParsePlan(filename.into(), err))
    }

    pub fn drop_post_plan(&mut self) {
        self.post_plan = None;
    }
}

#[derive(Debug, Serialize)]
pub struct Project {
    source: PathBuf,
    image: PathBuf,
    pre_plan: Vec<TrustedAction>,
    plan: Vec<UnsafeAction>,
    post_plan: Vec<TrustedAction>,
    artifact_max_size: Option<u64>,
    cache_max_size: Option<u64>,
}

#[derive(Debug, Serialize)]
pub struct Projects {
    projects: HashMap<String, Project>,
}

impl Projects {
    pub fn write(&self, filename: &Path) -> Result<String, AdapterError> {
        let text = serde_norway::to_string(&self).map_err(AdapterError::ToYaml)?;
        std::fs::write(filename, text.as_bytes()).map_err(AdapterError::WriteProjects)?;
        Ok(text)
    }

    pub fn project_name(&self) -> String {
        let keys: Vec<&str> = self.projects.keys().map(|s| s.as_str()).collect();
        assert_eq!(keys.len(), 1);
        keys[0].to_string()
    }
}

#[derive(Debug, Default)]
pub struct ProjectsBuilder {
    name: Option<String>,
    source: Option<PathBuf>,
    image: Option<PathBuf>,
    pre_plan: Vec<TrustedAction>,
    plan: Vec<UnsafeAction>,
    post_plan: Vec<TrustedAction>,
    artifact_max_size: Option<u64>,
    cache_max_size: Option<u64>,
}

impl ProjectsBuilder {
    pub fn name(mut self, name: &str) -> Self {
        self.name = Some(name.into());
        self
    }

    pub fn source(mut self, path: &Path) -> Self {
        self.source = Some(path.into());
        self
    }

    pub fn image(mut self, path: &Path) -> Self {
        self.image = Some(path.into());
        self
    }

    pub fn artifact_max_size(mut self, size: u64) -> Self {
        self.artifact_max_size = Some(size);
        self
    }

    pub fn cache_max_size(mut self, size: u64) -> Self {
        self.cache_max_size = Some(size);
        self
    }

    pub fn plan(mut self, plan: &DistilledPlan) -> Self {
        if let Some(actions) = &plan.pre_plan {
            self.pre_plan = actions.to_vec();
        }
        if let Some(actions) = &plan.plan {
            self.plan = actions.to_vec();
        }
        if let Some(actions) = &plan.post_plan {
            self.post_plan = actions.to_vec();
        }
        self
    }

    pub fn build(self) -> Projects {
        let projects: HashMap<String, Project> = HashMap::from([(
            self.name.unwrap(),
            Project {
                source: self.source.expect("ProjectsBuilder::source is set"),
                image: self.image.expect("ProjectsBuilder::image is set"),
                pre_plan: self.pre_plan.clone(),
                plan: self.plan.clone(),
                post_plan: self.post_plan.clone(),
                artifact_max_size: self.artifact_max_size,
                cache_max_size: self.cache_max_size,
            },
        )]);
        Projects { projects }
    }
}

pub fn run_ambient_ci(
    adminlog: &mut AdminLog,
    dry_run: bool,
    projects: &Projects,
    extra_ambient_config: Option<&Path>,
    extra_ambient_config_values: &HashMap<String, serde_norway::Value>,
    ssh_target: Option<&str>,
) -> Result<CiOutput, AdapterError> {
    let tmp = tempdir().map_err(AdapterError::TempDir)?;
    let projects_filename = tmp.path().join("projects.yaml");

    let project_name = projects.project_name();
    let projects = projects.write(&projects_filename)?;
    indented(adminlog, "projects", projects.as_bytes());

    adminlog
        .writeln("execute_ci: run ambient")
        .map_err(AdapterError::AdminLog)?;

    let mut cmd = if let Some(target) = ssh_target {
        let mut cmd = Command::new("ssh");
        cmd.arg("--");
        cmd.arg(target);
        cmd.arg("env");
        cmd
    } else {
        Command::new("env")
    };
    cmd.args(["AMBIENT_LOG=debug", "ambient"]);

    // If requested add an extra config file.
    if let Some(extra) = extra_ambient_config {
        cmd.arg("--config");
        cmd.arg(extra);
    }

    // If requested create a new config file with specific config
    // values, and add that to the argument list.
    if !extra_ambient_config_values.is_empty() {
        let yaml: String = serde_norway::to_string(extra_ambient_config_values)
            .map_err(AdapterError::AmbientConfigToYaml)?;

        let filename = tmp.path().join("ambient_extra_values.yaml");
        std::fs::write(&filename, &yaml).map_err(AdapterError::WriteConfig)?;

        cmd.arg("--config");
        cmd.arg(filename);
    }

    cmd.args(["run", "--force", "--projects"]);
    cmd.arg(projects_filename);

    adminlog.writeln(&format!("cmd={cmd:#?}"))?;

    // Run `ambient`.
    let (exit, stdout): (i32, Vec<u8>) = if dry_run {
        adminlog.writeln("run_ambient_ci: pretend to run ambient")?;
        (0, vec![])
    } else {
        adminlog.writeln("run_ambient_ci: run ambient for real")?;
        let (mut r, w) = pipe().unwrap();
        eprintln!("r={r:?} w={w:?}");
        let mut child = cmd
            .stdin(Stdio::null())
            .stdout(w.try_clone().unwrap())
            .stderr(w)
            .spawn()
            .map_err(|err| AdapterError::Command("ambient", err))?;

        std::mem::drop(cmd);
        eprintln!("read combined output from child");
        let mut buf = vec![];
        r.read_to_end(&mut buf).unwrap();
        eprintln!("read {} bytes", buf.len());

        let status = child.wait().unwrap();

        let exit = status.code().unwrap_or(NO_EXIT);
        adminlog.writeln(&format!("run_ambient_ci: exit={exit}"))?;

        if exit != 0 {
            indented(adminlog, "stdout+stderr", &buf);
        }

        (exit, buf)
    };

    adminlog.writeln("ambient finished")?;

    // Get the run log from `ambient`.
    let (_, run_log_raw) = get_log(
        adminlog,
        dry_run,
        &project_name,
        extra_ambient_config,
        ssh_target,
    )?;

    adminlog.writeln("run_ambient_ci: OK")?;
    Ok(CiOutput {
        exit,
        stdout,
        stderr: vec![],
        run_log: run_log_raw,
    })
}

fn get_log(
    adminlog: &mut AdminLog,
    dry_run: bool,
    project_name: &str,
    extra_ambient_config: Option<&Path>,
    ssh_target: Option<&str>,
) -> Result<(i32, Vec<u8>), AdapterError> {
    if dry_run {
        adminlog.writeln("pretend to run ambient log")?;
        Ok((0, vec![]))
    } else {
        let mut cmd = if let Some(target) = ssh_target {
            let mut cmd = Command::new("ssh");
            cmd.arg("--");
            cmd.arg(target);
            cmd.arg("ambient");
            cmd
        } else {
            Command::new("ambient")
        };

        if let Some(extra) = extra_ambient_config {
            cmd.arg("--config");
            cmd.arg(extra);
        }
        cmd.arg("log");
        cmd.arg(project_name);
        let output = cmd
            .output()
            .map_err(|err| AdapterError::Command("ambient log", err))?;
        let exit = output.status.code().unwrap_or(NO_EXIT);
        adminlog.writeln(&format!("get_log: exit={exit}"))?;

        if exit != 0 {
            indented(adminlog, "stdout", &output.stdout);
            indented(adminlog, "stderr", &output.stderr);
        }

        Ok((exit, output.stdout))
    }
}

pub struct CiOutput {
    pub exit: i32,
    pub stdout: Vec<u8>,
    pub stderr: Vec<u8>,
    pub run_log: Vec<u8>,
}
