//! This is a skeleton Radicle CI adapter for demonstration purposes.
//!
//! This program shows how to build a very simple adapter for Radicle
//! CI. As per the CI broker/adapter protocol, it reads a trigger
//! request message from its stdin, and writes two response messages
//! to its stdout. Based on the trigger, it clones the relevant Git
//! repository, and checks out the relevant Git commit, then runs
//! executed a silly CI plan. To keep things simple, and quick, in
//! this program the plan is "does the .radicle directory exist?". If
//! it exists, CI succeeds, otherwise it fails. If any of the previous
//! steps fails, CI also fails.
//!
//! Any errors are reported to stderr. If things go wrong very early,
//! the first response message may be a failure message, rather than a
//! "triggered" message. The CI broke will hopefully handle this.

#![allow(clippy::result_large_err)]

use std::{error::Error, fs::create_dir_all, path::Path, time::SystemTime};

use clap::Parser;
use tempfile::{tempdir, TempDir};

use radicle::prelude::{Profile, ReadStorage, RepoId};

use radicle_ci_broker::msg::{
    helper::{get_sources, read_request, write_failed, write_succeeded, write_triggered, AdminLog},
    Request, RunId, RunResult,
};

use radicle_ci_ambient::{
    ambient::{self, run_ambient_ci, CiOutput},
    config::Config,
    runlog::RunLog,
    AdapterError,
};

// Name of environment variable that has filename for configuration
// file.
const CONFIG_ENV: &str = "RADICLE_CI_AMBIENT";

fn main() -> Result<(), AdapterError> {
    match fallible_main() {
        Ok(true) => write_succeeded().map_err(AdapterError::Succeded)?,
        Ok(false) => write_failed().map_err(AdapterError::Failed)?,
        Err(err) => {
            eprintln!("ERROR: {err}");
            write_failed().map_err(AdapterError::Failed)?;
            std::process::exit(1);
        }
    }

    Ok(())
}

fn fallible_main() -> Result<bool, AdapterError> {
    let mut adapter = Adapter::new()?;
    adapter.execute()
}

#[derive(Debug)]
struct Adapter {
    args: Args,
    config: Config,
    adminlog: AdminLog,
    tmp: TempDir,
}

impl Adapter {
    fn new() -> Result<Self, AdapterError> {
        let args = Args::parse();
        let config = Config::load_via_env(CONFIG_ENV)?;
        let adminlog = config.open_log()?;
        Ok(Self {
            args,
            config,
            adminlog,
            tmp: tempdir().map_err(AdapterError::TempDir)?,
        })
    }

    fn log<S: Into<String>>(&mut self, msg: S) {
        self.adminlog.writeln(msg.into().as_ref()).ok();
    }

    fn get_request() -> Result<Request, AdapterError> {
        read_request().map_err(AdapterError::ReadRequest)
    }

    fn get_repository_name(&mut self, repoid: &RepoId) -> Result<String, AdapterError> {
        let profile = Profile::load().map_err(AdapterError::Profile)?;
        let doc = profile
            .storage
            .get(*repoid)
            .map_err(|err| AdapterError::GetIdDoc(*repoid, err))?;

        let repository_name = if let Some(doc) = doc {
            doc.project()
                .map_err(AdapterError::GetProject)?
                .name()
                .to_string()
        } else {
            "unknown Radicle repository name".into()
        };
        self.log(format!("repository name: {repository_name:?}"));
        Ok(repository_name)
    }

    fn execute(&mut self) -> Result<bool, AdapterError> {
        if self.args.show_config {
            eprintln!("{:#?}", self.config);
            return Ok(false);
        }

        self.log(format!("radicle-ci-ambient starts: {:#?}", self.config));

        // Read the trigger request message and extract the repository ID
        // and commit from it, if possible.
        let req = Self::get_request()?;
        let repoid = req.repo();
        let commit = req.commit().map_err(AdapterError::Commit)?;

        // Find project name.
        let repository_name = self.get_repository_name(&repoid)?;
        let project_name = if self.config.trust_repository_name() {
            repository_name
        } else {
            repoid.to_string()
        };
        self.log(format!(
            "project name: {project_name:?} {}",
            if self.config.trust_repository_name() {
                "trusted repo name"
            } else {
                "repo id"
            }
        ));

        // Create a temporary directory and a sub-directory `src` where
        // we'll put the source code from Git.
        let src = self.tmp.path().join("src");
        get_sources(&mut self.adminlog, self.args.dry_run, repoid, commit, &src)?;

        // Info about this CI run.
        let mut info = RunInfoBuilder::default();

        // Invent an adapter run ID and tell it to the CI broker.
        let run_id = RunId::default();
        info.set_adapter_run_id(&run_id);
        let project_log_dir = self.config.logdir.join(&project_name);
        if !project_log_dir.exists() {
            self.log("need to create log directory");
            create_dir_all(&project_log_dir)
                .map_err(|err| AdapterError::CreateDirs(project_log_dir, err))?;
            self.log("created log directory");
        }
        let run_log_relative = format!("{project_name}/{run_id}.html");

        let url = self
            .config
            .base_url
            .clone()
            .map(|base| format!("{base}/{run_log_relative}"));
        write_triggered(&run_id, url.as_deref()).map_err(AdapterError::Triggered)?;

        // Execute CI in the source directory.
        let result = self.execute_ci(info, &req, &project_name, &run_log_relative, &src);
        self.log(format!(
            "radicle-ci-ambient ends: Ambient result is {result:?}"
        ));

        if let Err(err) = &result {
            self.log(format!("ERROR: {err}"));
            let mut err = err.source();
            while let Some(cause) = err {
                self.log(format!("caused by {cause}"));
                err = cause.source();
            }
        }

        result
    }

    fn get_distilled_plan(&self, src: &Path) -> Result<ambient::DistilledPlan, AdapterError> {
        let mut distilled = if self.args.dry_run {
            ambient::DistilledPlan::default()
        } else {
            let filename = src.join(".radicle/ambient.yaml");
            ambient::DistilledPlan::read(&filename)?
        };

        if !self.config.allow_post_plan() {
            distilled.drop_post_plan();
        }
        Ok(distilled)
    }

    fn execute_ci(
        &mut self,
        mut info: RunInfoBuilder,
        trigger: &Request,
        project_name: &str,
        run_log_relative: &str,
        src: &Path,
    ) -> Result<bool, AdapterError> {
        self.log(format!("execute_ci: src={}", src.display()));

        let distilled = self.get_distilled_plan(src)?;
        let projects = ambient::ProjectsBuilder::default()
            .name(project_name)
            .source(Path::new("src"))
            .image(if self.config.ssh_target().is_some() {
                self.config
                    .remote_image
                    .as_ref()
                    .unwrap_or(&self.config.image)
            } else {
                &self.config.image
            })
            .plan(&distilled)
            .build();

        let ssh_remote = self.config.ssh_target();
        let output = run_ambient_ci(
            self.tmp.path(),
            &mut self.adminlog,
            self.args.dry_run,
            &projects,
            self.config.extra_ambient_config(),
            self.config.extra_ambient_config_values(),
            ssh_remote,
        )?;
        if output.exit == 0 {
            info.set_result(RunResult::Success);
        } else {
            info.set_result(RunResult::Failure);
        }

        let info = info.build()?;
        self.write_log_files(&info, trigger, project_name, run_log_relative, &output)?;

        if self.args.dry_run {
            self.log("execute_ci: Ambient is pretending to be OK");
            Ok(true)
        } else if output.exit == 0 {
            self.log("execute_ci: Ambient is OK");
            Ok(true)
        } else {
            self.log(format!("ERROR: ambient exit code {}", output.exit));
            Ok(false)
        }
    }

    fn write_log_files(
        &mut self,
        info: &RunInfo,
        trigger: &Request,
        project_name: &str,
        run_log_relative: &str,
        ci_output: &CiOutput,
    ) -> Result<(), AdapterError> {
        let logdir = self.config.logdir.clone();
        if !logdir.exists() {
            self.log("create log directory");
            std::fs::create_dir_all(&logdir)
                .map_err(|err| AdapterError::CreateDirs(logdir, err))?;
        }

        let mut log = RunLog::default();
        log.set_project_name(project_name.into());
        log.set_trigger(trigger.clone());
        log.set_started(info.started);
        log.set_adapter_run_id(info.adapter_run_id.clone());
        log.set_result(info.result.clone());
        log.set_ambient_output(ci_output.stdout.clone(), ci_output.stderr.clone());
        log.set_ambient_run_log(ci_output.run_log.clone());
        let page = log.to_html();

        self.log("convert terminal output to HTML and write that");
        let filename = self
            .config
            .logdir
            .join(run_log_relative)
            .with_extension("html");
        std::fs::write(&filename, page.to_string())
            .map_err(|err| AdapterError::WriteRunLog(filename.clone(), err))?;
        Ok(())
    }
}

#[derive(Debug, Parser)]
#[clap(version = env!("VERSION"))]
struct Args {
    #[clap(long)]
    dry_run: bool,

    #[clap(long)]
    show_config: bool,
}

struct RunInfoBuilder {
    started: SystemTime,
    adapter_run_id: Option<RunId>,
    result: Option<RunResult>,
}

impl Default for RunInfoBuilder {
    fn default() -> Self {
        Self {
            started: SystemTime::now(),
            adapter_run_id: None,
            result: None,
        }
    }
}

impl RunInfoBuilder {
    fn build(self) -> Result<RunInfo, AdapterError> {
        Ok(RunInfo {
            started: self.started,
            adapter_run_id: self
                .adapter_run_id
                .ok_or(AdapterError::Missing("adapter_run_id"))?,
            result: self.result.ok_or(AdapterError::Missing("result"))?,
        })
    }

    fn set_adapter_run_id(&mut self, id: &RunId) {
        self.adapter_run_id = Some(id.clone());
    }

    fn set_result(&mut self, result: RunResult) {
        self.result = Some(result);
    }
}

struct RunInfo {
    started: SystemTime,
    adapter_run_id: RunId,
    result: RunResult,
}
