use std::{
    error::Error,
    path::PathBuf,
    thread::{sleep, spawn},
    time::Duration,
};

use log::{debug, info};

use radicle::prelude::Profile;
use radicle_ci_broker::{
    adapter::Adapter,
    broker::Broker,
    config::Config,
    error::BrokerError,
    event::NodeEventSource,
    msg::RequestBuilder,
    pages::{PageBuilder, StatusPage},
};

fn main() {
    if let Err(e) = fallible_main() {
        eprintln!("ERROR: {}", e);
        let mut e = e.source();
        while let Some(source) = e {
            eprintln!("caused by: {}", source);
            e = source.source();
        }
    }
}

fn fallible_main() -> Result<(), BrokerError> {
    pretty_env_logger::init();
    info!("Radicle CI broker starts");

    let mut args = std::env::args().skip(1);
    let filename: PathBuf = if let Some(filename) = args.next() {
        PathBuf::from(filename)
    } else {
        return Err(BrokerError::Usage);
    };

    let config = Config::load(&filename)?;
    debug!("loaded configuration: {:#?}", config);

    let mut broker = Broker::new(config.db())?;
    debug!(
        "created broker, db has {} CI runs",
        broker.all_runs()?.len()
    );

    // FIXME: this is broken. the config file only lists how to invoke
    // each adapter, not what adapter to use for each repo.
    // for (rid, spec) in config.adapters.iter() {
    //     debug!("setting adapter for {rid:?} to {spec:#?}");
    //     let rid = RepoId::from_urn(rid).map_err(|e| BrokerError::BadRepoId(rid.into(), e))?;
    //     let adapter = Adapter::new(&spec.command).with_environment(spec.envs());
    //     broker.set_repository_adapter(&rid, &adapter);
    // }
    // debug!("set per-repository adapters");

    let spec =
        config
            .adapter(&config.default_adapter)
            .ok_or(BrokerError::UnknownDefaultAdapter(
                config.default_adapter.clone(),
            ))?;
    let adapter = Adapter::new(&spec.command).with_environment(spec.envs());
    broker.set_default_adapter(&adapter);
    debug!("set default adapter");

    let profile = Profile::load()?;
    debug!("loaded profile {profile:#?}");

    let mut source = NodeEventSource::new(&profile)?;
    debug!("created node event source");

    for filter in config.filters.iter() {
        source.allow(filter.clone());
    }
    debug!("added filters to node event source");

    // Spawn a thread that updates the status pages.
    let mut page = PageBuilder::default()
        .node_alias(&profile.config.node.alias)
        .runs(broker.all_runs()?)
        .build()?;
    let page2 = page.clone();
    let report_dir = if let Some(dir) = &config.report_dir {
        dir.to_path_buf()
    } else {
        PathBuf::from(".")
    };
    let interval = Duration::from_secs(config.status_page_update_interval());
    let status_thread = spawn(move || status_updater(report_dir, page2, interval));
    debug!(
        "started thread to update status pages in the background: {:?}",
        status_thread.thread().id()
    );

    // This loop ends when there's an error, e.g., failure to read an
    // event from the node.
    loop {
        debug!("waiting for event from node");
        for e in source.event()? {
            page.broker_event(&e);
            debug!("broker event {e:#?}");
            let req = RequestBuilder::default()
                .profile(&profile)
                .broker_event(&e)
                .build_trigger()?;
            broker.execute_ci(&req, &mut page)?;
        }
    }
}

fn status_updater(dirname: PathBuf, mut page: StatusPage, interval: Duration) {
    let filename = dirname.join("status.json");
    loop {
        page.update_timestamp();
        if let Err(e) = page.write_json(&filename) {
            eprintln!("ERROR: failed to update {}: {e}", filename.display());
        }
        if let Err(e) = page.write(&dirname) {
            eprintln!(
                "ERROR: failed to update repot pages in {}: {e}",
                dirname.display()
            );
        }
        sleep(interval);
    }
}
